/*
 * adddefpriv.c
 *
 *	Add a default privilege to the system.
 *
 *
 *  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.10 $"

#include <locale.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <stdio.h>
#include <errno.h>
#include <locale.h>
#include <fcntl.h>

#include <sysadm/i18n.h>
#include <sysadm/privilege.h>
#include <sysadm/auth.h>
#include <sysadm/format.h>

#define SaADDDEFPRIV CONFIG_SYSADM_BIN_DIR "/adddefpriv"

static char *gAuthScheme;
static char *gPrivilege;

/*
 *  static void ParseArgs(int argc, char *argv[])
 *
 *  Description:
 *      Parse adddefpriv command line arguments.  Usage is:
 *          adddefpriv [ -auth authScheme ] <privilege>
 *
 *  Parameters:
 *      argc  Number of args
 *      argv  Vector of args
 */
static void ParseArgs(int argc, char *argv[])
{
    /*
     * Watch out for people mucking with us.  Remember, we're a setuid
     * root program.
     */
    if (argc < 1) {
	exit(SaPRIVERROR);
    }

    if (argc < 2) {
	fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <privilege>\n"), SaADDDEFPRIV);
	exit(1);
    }

    if (strcmp(argv[1], "-auth") == 0) {
	if (argc != 4) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <privilege>\n"), SaADDDEFPRIV);
	    exit(1);
	}
	gAuthScheme = argv[2];
	gPrivilege = argv[3];
    } else {
	if (argc != 2) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <privilege>\n"), SaADDDEFPRIV);
	    exit(1);
	}
	gPrivilege = argv[1];
    }

    /*
     * This prevents us from creating files in the wrong place, e.g.
     * "adddefpriv ../../../.rhosts".
     */
    SaPrivValidatePrivName(gPrivilege);

    /*
     * It's hopeless to run adddefpriv if you're not root without
     * providing an authentication scheme.
     */
    if (getuid() && !gAuthScheme) {
	fprintf(stderr, "%s\n",
                i18n("non-root users must supply -auth argument."));
	exit(1);
    }

}

/*
 *  static void Authenticate(void)
 *
 *  Description:
 *      Authenticate the user.
 */
static void Authenticate(void)
{
    if (!gAuthScheme) {
	/*
	 * This should never happen.  This is an assert which won't
	 * compile out.
	 */
	if (getuid()) {
	    exit(SaPRIVERROR);
	}
	return;
    }

    if (!SaPrivAuthenticate(gAuthScheme)) {
	fprintf(stderr, "%s\n", i18n("Authentication failed."));
	exit(1);
    }
}

/*
 *  static void AddDefaultPrivilege(void)
 *
 *  Description:
 *      Make "gPrivilege" a default privilege.
 */
static void AddDefaultPrivilege(void)
{
    char defPrivFile[PATH_MAX];
    int fd;

    (void)SaStringFormat(defPrivFile, sizeof defPrivFile,
			 "%s/%s", SaPRIVDEFDIR, gPrivilege);
    fd = open(defPrivFile, O_WRONLY | O_TRUNC | O_CREAT, 0644);
    if (fd == -1)  {
	perror(defPrivFile);
	exit(1);
    }

    if (write(fd, "on\n", 3) != 3) {
	perror(defPrivFile);
	unlink(defPrivFile);
	exit(1);
    }

    (void)close(fd);
}

/*
 * adddefpriv main program.
 */
void main(int argc, char *argv[])
{
    SaPrivFixUmask();


    /*
     * Don't even try if we're not setuid root.
     */
    if (geteuid() != 0) {
	fprintf(stderr, "%s\n", i18n("setuid root bit is off"));
	exit(SaPRIVNOTSETUID);
    }

    ParseArgs(argc, argv);
    Authenticate();
    AddDefaultPrivilege();

    exit(0);
}
