/*
 * addpriv.c
 *
 *	Add a privilege/user combination to the privilege database.
 *	
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.17 $"

#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <unistd.h>
#include <fcntl.h>
#include <locale.h>
#include <pwd.h>
#include <syslog.h>

#include <sysadm/i18n.h>
#include <sysadm/privilege.h>
#include <sysadm/auth.h>
#include <sysadm/format.h>

/*
 * Struct used by AddPrivileges for keeping track of privileges that have
 * been seen.
 */
typedef struct _PrivInfo {
    const char *privilege;
    bool added;
} PrivInfo;

static char *gAuthScheme;	/* authentication scheme specified by */
				/* "-auth" argument.  */
static char **gPrivileges;	/* privileges to add. */
static int gNumPrivileges;	/* Number of privileges to add */
static char *gUserName;		/* user to giv privilege to. */
static int gLockfd;		/* File descriptor for database lock file */
static char gTempFile[PATH_MAX];/* File name for temporary database */
static bool gChkConfig = false; /* Are we running chkconfig to */
				       /* enable privileges? */
static bool gNisUser = false;   /* Is gUserName an NIS user? */

/*
 *  static void ParseArgs(int argc, char *argv[])
 *
 *  Description:
 *      Parse command line arguments.  Usage is:
 *      
 *      addpriv [ -auth <auth> ] <user> <privilege>
 *
 *  Parameters:
 *      argc  Number of command line arguments.
 *      argv  Command line arguments.
 */
static void ParseArgs(int argc, char *argv[])
{
    struct passwd *pwd;
#ifdef HAVE__GETPWENT_NO_YP
    extern int _getpwent_no_yp; 
    int saveNoYP;
#endif /* HAVE__GETPWENT_NO_YP */
    /*
     * Watch out for people mucking with us.  Remember, we're a setuid
     * root program.
     */
    if (argc < 1) {
	exit(SaPRIVERROR);
    }

    if (argc < 2) {
	fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <userid> "
	        "<privilege> [ <privilege> ... ]\n"), SaADDPRIV);
	exit(1);
    }

    argv++;
    argc--;
    if (strcmp(argv[0], "-auth") == 0) {
	if (argc < 3) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <userid> "
	            "<privilege> [ <privilege> ... ]\n"), SaADDPRIV);
	    exit(1);
	}
	gAuthScheme = argv[1];
	argc -= 2;
	argv += 2;
    }

    if (strcmp(argv[0], "-chkconfig") == 0) {
	if (argc != 1) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <userid> "
	            "<privilege> [ <privilege> ... ]\n"), SaADDPRIV);
	    exit(1);
	}
	gChkConfig = true;
    } else {
	if (argc < 2) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <userid> "
	            "<privilege> [ <privilege> ... ]\n"), SaADDPRIV);
	    exit(1);
	}

	gUserName = argv[0];
	gPrivileges = argv + 1;
	gNumPrivileges = argc - 1;

#ifdef HAVE__GETPWENT_NO_YP
	/*
	 * We want to generate a warning if we're adding privileges for an
	 * NIS user.
	 */
	saveNoYP = _getpwent_no_yp;
	_getpwent_no_yp = 1;
	pwd = getpwnam(gUserName);
	_getpwent_no_yp = saveNoYP;

	if (pwd == NULL) {
	    /*
	     * Setting _getpwent_no_yp back to its old value (which we
	     * did up above) should result in getpwnam returning
	     * passwd entries for NIS accounts.
	     */
	    pwd = getpwnam(gUserName);
	    if (pwd == NULL) {
		fprintf(stderr, i18n("%s is not a valid userid.\n"), gUserName);
		exit(1);
	    } else {
		gNisUser = true;
	    }
	}
#endif /* HAVE__GETPWENT_NO_YP */
    }

    /*
     * It's hopeless to run addpriv if you're not root without
     * providing an authentication scheme.
     */
    if (getuid() && !gAuthScheme) {
	fprintf(stderr, "%s\n",
	        i18n("non-root users must supply -auth argument."));
	exit(1);
    }
}

/*
 *  static void Authenticate(void)
 *
 *  Description:
 *      Check to see if user is allowed to run addpriv.  We use the
 *      authentication scheme provided on the command line to
 *      authenticate the user.
 */
static void Authenticate(void)
{
    if (!gAuthScheme) {
	/*
	 * This should never happen.  This is an assert which won't
	 * compile out.
	 */
	if (getuid()) {
	    exit(SaPRIVERROR);
	}
	return;
    }

    if (!SaPrivAuthenticate(gAuthScheme)) {
	fprintf(stderr, "%s\n", i18n("Authentication failed."));
	exit(1);
    }
}

/*
 *  static void ValidatePrivileges(void)
 *
 *  Description:
 *      Do some sanity checking to make sure user isn't adding some
 *      bogus privileges.
 */
static void ValidatePrivileges(void)
{
    int priv;

    for (priv = 0; priv < gNumPrivileges; priv++) {
	SaPrivValidatePrivName(gPrivileges[priv]);
    }
}

/*
 *  static void LockDB(void)
 *
 *  Description:
 *      Lock the privilege database, so that no other programs will
 *      modify it.  Locking is advisory; only programs that
 *      participate in our locking scheme which uses a separate lock
 *      file will be locked out.
 */
static void LockDB(void)
{
    static char lockFile[] = SaPRIVLOCK;

    gLockfd = open(lockFile, O_WRONLY | O_CREAT, 020600);
    if (gLockfd == -1 || lockf(gLockfd, F_LOCK, 0) != 0) {
	perror(lockFile);
	exit(1);
    }
}

/*
 *  void UnLockDB(void)
 *
 *  Description:
 *      Release our lock on the privilege database.
 */
void UnLockDB(void)
{
    (void)lockf(gLockfd, F_ULOCK, 0);
    (void)close(gLockfd);
}

/*
 *  static PrivInfo *CreatePrivInfo(char * const *privs,
 *  				    unsigned int nPrivs)
 *
 *  Description:
 *      Create an array of PrivInfo structs that AddPrivileges can use to
 *      keep track of which privileges have been added.
 *
 *  Parameters:
 *      privs   privileges to make PrivInfo for.
 *      nPrivs  number of privileges passed in.
 *
 *  Returns:
 *	PrivInfo array.
 */
static PrivInfo *CreatePrivInfo(char * const *privs, unsigned int nPrivs)
{
    PrivInfo *privInfo;
    int i;

    privInfo = malloc(sizeof *privInfo * nPrivs);
    for (i = 0; i < nPrivs; i++) {
	privInfo[i].privilege = privs[i];
	privInfo[i].added = false;
    }

    return privInfo;
}

/*
 *  static void DestroyPrivInfo(PrivInfo *privInfo)
 *
 *  Description:
 *      Destroy an array of PrivInfo created by CreatePrivInfo.
 *
 *  Parameters:
 *      privInfo  The PrivInfo to destroy.
 */
static void DestroyPrivInfo(PrivInfo *privInfo)
{
    free(privInfo);
}

/*
 *  FILE *CreateTempDB(void)
 *
 *  Description:
 *      Create a temporary file to hold the contents of the new
 *      database.
 *
 *  Returns:
 *	FILE * handle to temporary file.  exits if an error occurs.
 */
FILE *CreateTempDB(void)
{
    int fd;
    FILE *newdb;
    struct stat f;

    /*
     * Create a temporary file.
     */
    (void)SaStringFormat(gTempFile, sizeof gTempFile, "%sXXXXXX", SaPRIVDB);
    fd = mkstemp(gTempFile);
    if (fd == -1 || (newdb = fdopen(fd, "w")) == NULL) {
	perror(gTempFile);
	exit(1);
    }

    /*
     * Preserve privilege database permissions.
     */
    if (stat(SaPRIVDB, &f) == 0) {
	(void)fchmod(fd, f.st_mode);
    } else {
	(void)fchmod(fd, 0644);
    }

    return newdb;
}

/*
 *  void CloseTempDB(FILE *newdb)
 *
 *  Description:
 *      Close the temporary database file.  We also check for errors
 *      here; if anything went wrong, we exit with an error message.
 *      We don't want to corrupt the privilege database; this
 *      temporary file is about to become the real privilege database.
 *
 *  Parameters:
 *      newdb  The database to close.
 */
void CloseTempDB(FILE *newdb)
{
    /*
     * Make sure everything was cool.
     */
    int fd = fileno(newdb);
    if ((ferror(newdb) || fclose(newdb) == EOF)) {
	perror(gTempFile);
	exit(1);
    }
    (void)close(fd);
}

/*
 *  static void AddPrivileges(void)
 *
 *  Description:
 *      Add the user/privileges combination specified on the command
 *      line to the privilege database.  We establish a lock on the
 *      lock file to avoid race conditions with other people running
 *      addpriv.  We copy the existing privilege database to a
 *      temporary file, looking for the privileges that we're adding.
 *      If we find a privilege while we're copying, we add the user
 *      to the end of the list (if the user isn't already in the
 *      list).  Otherwise, we add new lines at the end of the
 *      privilege database containing entries for these privileges and
 *      this user.
 *      
 *      At the end, we replace the old database with the new database.
 */
static void AddPrivileges(void)
{
    FILE *newdb;
    SaPrivFile *privFile;
    const SaPrivEnt *ent;
    int i, priv;
    PrivInfo *privInfo = CreatePrivInfo(gPrivileges, gNumPrivileges);
    PrivInfo *thisInfo;

    if (gNisUser) {
	/*
	 * If we're adding privileges for an NIS user, print a
	 * warning on stderr and to syslog.
	 */
	const char *fmt = i18n("Warning: addpriv(1M) adding privileges for "
	                       "NIS user: %s.\n");
	fprintf(stderr, fmt, gUserName);
	syslog(LOG_WARNING, fmt, gUserName);
    }

    LockDB();
    newdb = CreateTempDB();

    /*
     * Copy the contents of the old database to the temporary file,
     * while looking for the entries corresponding to the privileges
     * we're adding.  If we find one, we'll add our user to it.
     */
    privFile = SaOpenPrivDB();
    if (privFile) {
	while ((ent = SaGetPrivEnt(privFile)) != NULL) {
	    (void)fprintf(newdb, "%s:", ent->privilege);
	    thisInfo = NULL;
	    for (priv = 0; priv < gNumPrivileges; priv++) {
		if (strcmp(privInfo[priv].privilege, ent->privilege) == 0) {
		    thisInfo = privInfo + priv;
		    break;
		}
	    }

	    for (i = 0; i < ent->numUsers; i++) {
		(void)fprintf(newdb, "%s", ent->users[i]);
		/*
		 * Set added flag if the user is already in list for
		 * this privilege.
		 */
		if (thisInfo
		    && strcmp(ent->users[i], gUserName) == 0) {
		    thisInfo->added = true;
		}
		if (i < ent->numUsers - 1) {
		    (void)fputc(',', newdb);
		}
	    }

	    if (thisInfo && !thisInfo->added) {
		if (ent->numUsers) {
		    (void)fputc(',', newdb);
		}
		(void)fprintf(newdb, "%s", gUserName);
		thisInfo->added = true;
	    }

	    (void)fputc('\n', newdb);
	}
	SaClosePrivDB(privFile);
    }

    /*
     * If we didn't find entries for any of our privileges, add new
     * entries at the end.
     */
    for (priv = 0; priv < gNumPrivileges; priv++) {
	if (!privInfo[priv].added) {
	    (void)fprintf(newdb, "%s:%s\n", privInfo[priv].privilege,
			  gUserName);
	}
    }

    DestroyPrivInfo(privInfo);
    CloseTempDB(newdb);

    /*
     * Replace the old database with the new database.
     */
    if (rename(gTempFile, SaPRIVDB) == -1) {
	perror(SaPRIVDB);
	exit(1);
    }

    UnLockDB();
}

/*
 * addpriv main program.
 */
int main(int argc, char *argv[])
{
    SaPrivFixUmask();


    /*
     * Don't even try if we're not setuid root.
     */
    if (geteuid() != 0) {
	fprintf(stderr, "%s\n", i18n("setuid root bit is off"));
	exit(SaPRIVNOTSETUID);
    }

    ParseArgs(argc, argv);
    Authenticate();
    /*
     * Call ValidatePrivileges() after calling Authenticate(), so that
     * if there are any vulnerabilities in ValidatePrivileges() it's
     * "OK" because the user has already typed the root password.
     */
    ValidatePrivileges();
    if (gChkConfig) {
	execl("/etc/chkconfig", "chkconfig", "-f", "privileges", "on", NULL);
    } else {
	AddPrivileges();
    }
    exit(0);
}
