/*
 * checkpriv.c
 *
 *	Check whether a user is privileged
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.15 $"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <locale.h>
#include <limits.h>

#include <sysadm/i18n.h>
#include <sysadm/privilege.h>
#include <sysadm/auth.h>

static char *gAuthScheme = NULL; 	/* Authentication scheme to use */
static const char **gPrivileges;	/* Privileges to check */
static unsigned int gNumPrivileges;     /* Number of privileges */
static bool gPrintReason = false;/* Whether to print success reason */
static bool gAuthOnly = false;	/* Whether to just check auth */

/*
 *  static void ParseArgs(int argc, char *argv[])
 *
 *  Description:
 *      Parse checkpriv's command line arguments.
 *
 *  Parameters:
 *      argc  Number of command line arguments.
 *      argv  command line arguments.
 */
static void ParseArgs(int argc, char *argv[])
{
    argv++;

    if (argc < 2) {
	fprintf(stderr, i18n("usage: %s [ -auth <auth> ] [ -reason ] "
	        "<privilege> [ <privilege> ... ]\n"), SaCHECKPRIV);
	exit(1);
    }
	
    while (--argc > 0 && **argv == '-') {
	if (strcmp(*argv, "-auth") == 0) {
	    if (argc < 2) {
		fprintf(stderr, i18n("usage: %s [ -auth <auth> ] [ -reason ] "
		        "<privilege> [ <privilege> ... ]\n"), SaCHECKPRIV);
		exit(1);
	    }		
	    gAuthScheme = argv[1];
	    argv++;
	    argc--;
	} else if (strcmp(*argv, "-reason") == 0) {
	    gPrintReason = true;
	} else if (strcmp(*argv, "-authOnly") == 0) {
	    gAuthOnly = true;
	} else {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] [ -reason ] "
	            "<privilege> [ <privilege> ... ]\n"), SaCHECKPRIV);
	    exit(1);
	}
	argv++;
    }

    if (argc < 1 && !gAuthOnly) {
	fprintf(stderr, i18n("usage: %s [ -auth <auth> ] [ -reason ] "
	        "<privilege> [ <privilege> ... ]\n"), SaCHECKPRIV);
	exit(1);
    }

    gPrivileges = (const char **)argv;
    gNumPrivileges = argc;
}

/*
 *  static void ValidatePrivileges(void)
 *
 *  Description:
 *      Do some sanity checking to make sure user isn't checking some
 *      bogus privileges.
 */
static void ValidatePrivileges(void)
{
    int priv;

    for (priv = 0; priv < gNumPrivileges; priv++) {
	SaPrivValidatePrivName(gPrivileges[priv]);
    }
}

/*
 * checkpriv main program.
 */
void main(int argc, char *argv[])
{
    SaPrivResult privResult;

    SaPrivFixUmask();


    ParseArgs(argc, argv);
    
    if (gAuthScheme && SaPrivAuthenticate(gAuthScheme)) {
	if (gPrintReason) {
	    (void)printf("%d\n", SaAuthPassed);
	}
	exit(SaPRIVOK);
    }

    if (gAuthOnly) {
	exit(SaPRIVNOTPRIV);
    }

    ValidatePrivileges();

    privResult = SaCheckPriv(getuid(), gPrivileges, gNumPrivileges);
    if (privResult == SaNotPriv) {
	exit(SaPRIVNOTPRIV);
    }

    if (gPrintReason) {
	(void)printf("%d\n", privResult);
    }
    exit(SaPRIVOK);
}
