#!/usr/bin/perl -w
#
# checkpriv.sh
#
#	verify that checkpriv works right.
#	
# Copyright 1995, Silicon Graphics, Inc.
# ALL RIGHTS RESERVED
# 
# UNPUBLISHED -- Rights reserved under the copyright laws of the United
# States.   Use of a copyright notice is precautionary only and does not
# imply publication or disclosure.
# 
# U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
# Use, duplication or disclosure by the Government is subject to restrictions
# as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
# in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
# in similar or successor clauses in the FAR, or the DOD or NASA FAR
# Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
# 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
#
# THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
# INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
# DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
# PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
# GRAPHICS, INC.
# 
#
# $Revision: 1.9 $
#

if ($#ARGV == -1) {
    die "usage: $0 privilege\n";
}

$numPrivileges = $#ARGV + 1;
@privileges = @ARGV;

#
# Look in /etc/passwd to see if root has a password
#
sub hasRootPassword
{
    open(PASSWD, "/etc/passwd") || die "Error: Can't open /etc/passwd: $!\n";
    while (<PASSWD>) {
	@line = split(/:/);
	if ($#line > 0 && shift @line eq "root") {
	    if (shift @line eq "") {
		return 0;
	    } else {
		return 1;
	    }
	}
    }
    close(PASSWD);
    die "Error: Can't find root account in /etc/passwd!\n";
}

#
# Get current user's login name
#
sub getLoginName
{
    return (getpwuid($<))[0] || die "Error: Can't get login name\n";
}

#
# Find out if the user passed in is root.
#
sub isRoot
{
    local($user) = @_;
    return $user eq "root";
}

#
# Check to see if a privilege is a default privilege.
#
sub isDefaultPriv
{
    local($priv) = @_;
    local($isDefault) = (0);
    open(DEFFILE, "/var/sysadm/defaultPrivileges/$priv")
	|| return 0;
    if ($_ = <DEFFILE>) {
	chop;
	$isDefault = $_ eq "on";
    }
    close(DEFFILE);
    return $isDefault;
}

#
# Check to see if a user is a privileged user.
#
sub isPrivUser
{
    local($user) = @_;

    open(USERFILE, "/var/sysadm/privilegedUsers")
	|| return 0;
    while (<USERFILE>) {
	chop;
	if ($_ eq $user) {
	    close(USERFILE);
	    return 1;
	}
    }
    return 0;
}

#
# Check to see whether privileges are enabled.
#
sub arePrivilegesEnabled
{
    open(CHKCONFIG, "/etc/config/privileges") || return 0;
    $_ = <CHKCONFIG>;
    close(CHKCONFIG);
    chop;
    return $_ eq "on";
}

#
# Our own version of what "checkpriv" does; use different code to
# make sure that checkpriv is doing the right thing.
#
sub hasPriv
{
    local($privilege) = @_;

    $user = &getLoginName();
    if (&isRoot($user) || !&hasRootPassword()) {
	return 1;
    }

    if (!&arePrivilegesEnabled()) {
	return 0;
    }

    # Check to see if this is a default privilege.
    if (&isDefaultPriv($privilege)) {
	return 1;
    }

    # Check to see if this is a privileged user.
    if (&isPrivUser($user)) {
	return 1;
    }

    open(PRIVDB, "/var/sysadm/privilege")
	|| die "Error: Can't open privilege database: $!\n";
    while (<PRIVDB>) {
	if (! /^#/) {
	    @line = split(/[: \t,\n]+/);
	    if ($#line > 0 && $privilege eq $line[0]) {
		shift @line;
		foreach $userid (@line) {
		    if ($user eq $userid) {
			close(PRIVDB);
			return 1;
		    }
		}
	    }
	}
    }
    close(PRIVDB);
    return 0;
}

#
# Compare our answer to what the real checkpriv says, and print
# a report
#
$checkPrivSays = system("/usr/sysadm/bin/checkpriv @privileges") == 0;
$login = &getLoginName();

$printIt = 0;

if ($printIt) {
    if ($checkPrivSays) {
	print "checkpriv says $login has privileges @privileges\n";
    } else {
	print "checkpriv says $login doesn't have privilege @privileges\n";
    }
}

$weSay = 1;
$i = 0;
$user = &getLoginName();
while ($weSay && $i < $numPrivileges) {
    if (!&hasPriv($privileges[$i])) {
	$weSay = 0;
    }
    $i++;
}

if ($checkPrivSays != $weSay) {
    die "Error: /usr/sysadm/bin/checkpriv: got $login @privileges wrong!\n";
}
