/*
 * rmpriv.c
 *
 *	Remove a privilege/user combination from the privilege database.
 *	
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.17 $"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <limits.h>
#include <locale.h>
#include <pwd.h>

#include <sysadm/i18n.h>
#include <sysadm/format.h>

#include <sysadm/privilege.h>

static char **gPrivileges;	/* privilege to add. */
static int gNumPrivileges;	/* Number of privileges to remove */
static char *gUserName;		/* user to remove privilege from. */
static char *gAuthScheme;	/* Authentication scheme */
static bool gRemoveAll = false; /* Whether this privilege is to */
				       /* be removed for all users */
static bool gChkConfig = false; /* Whether we're to run */
				       /* chkconfig to remove all */
				       /* privileges */
static bool gRemoveAcct = false; /* Whether all privileges are to */
					/* be removed for a specific user. */

/*
 *  static void ParseArgs(int argc, char *argv[])
 *
 *  Description:
 *      Parse command line arguments.  Usage is:
 *      
 *      rmpriv <user> <privilege>
 *
 *  Parameters:
 *      argc  Number of command line arguments.
 *      argv  Command line arguments.
 */
static void ParseArgs(int argc, char *argv[])
{
    /*
     * Watch out for people mucking with us.  Remember, we're a setuid
     * root program.
     */
    if (argc < 1) {
	exit(SaPRIVERROR);
    }

    if (argc < 2) {
	fprintf(stderr, i18n("usage: %s [ -auth <auth> ] "
	    "< < <userid> | -all > <privilege> "
	    "[ <privilege> ... ] > | -acct user\n"), SaRMPRIV);
	exit(1);
    }

    if (strcmp(argv[1], "-auth") == 0) {
	gAuthScheme = argv[2];
	argc -= 2;
	argv += 2;
    }

    if (strcmp(argv[1], "-chkconfig") == 0) {
	if (argc != 2) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] "
	        "< < <userid> | -all > <privilege> "
	        "[ <privilege> ... ] > | -acct user\n"), SaRMPRIV);
	    exit(1);
	}	    
	gChkConfig = true;
    } else {
	if (argc < 3) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] "
	        "< < <userid> | -all > <privilege> "
	        "[ <privilege> ... ] > | -acct user\n"), SaRMPRIV);
	    exit(1);
	}
	if (strcmp(argv[1], "-acct") == 0)
	{
	    if (argc != 3)
	    {
		fprintf(stderr, i18n("usage: %s [ -auth <auth> ] "
		    "< < <userid> | -all > <privilege> "
		    "[ <privilege> ... ] > | -acct user\n"), SaRMPRIV);
		exit(1);
	    }
	    gRemoveAcct = true;
	    gUserName = argv[2];
	    gPrivileges = NULL;
	    gNumPrivileges = 0;
	}
	else if (strcmp(argv[1], "-all") == 0)
	{
	    if (argc != 3)
	    {
		fprintf(stderr, i18n("usage: %s [ -auth <auth> ] "
		    "< < <userid> | -all > <privilege> "
		    "[ <privilege> ... ] > | -acct user\n"), SaRMPRIV);
		exit(1);
	    }
	    gRemoveAll = true;
	    gPrivileges = argv + 2;
	    gNumPrivileges = argc - 2;
	}
	else
	{
	    gUserName = argv[1];
	    gPrivileges = argv + 2;
	    gNumPrivileges = argc - 2;
	}
    }

    /*
     * It's hopeless to run rmpriv if you're not root without
     * providing an authentication scheme.
     */
    if (getuid() && !gAuthScheme) {
	fprintf(stderr, "%s\n",
	    i18n("non-root users must supply -auth argument."));
	exit(1);
    }
}

/*
 *  static void Authenticate(void)
 *
 *  Description:
 *      Authenticate the user.
 */
static void Authenticate(void)
{
    if (!gAuthScheme) {
	/*
	 * This should never happen.  This is an assert which won't
	 * compile out.
	 */
	if (getuid()) {
	    exit(SaPRIVERROR);
	}
	return;
    }

    if (!SaPrivAuthenticate(gAuthScheme)) {
	fprintf(stderr, "%s\n", i18n("Authentication failed."));
	exit(1);
    }
}

/*
 *  static void RemovePrivileges(void)
 *
 *  Description:
 *      Remove a privilege/user combination from the privilege
 *      database.  Copy the contents of the database to a temp file,
 *      without writing the specified user's name on the privilege
 *      line for the specified privilge (gUserName and gPrivileges are
 *      globals).
 */
static void RemovePrivileges(void)
{
    char tempFile[PATH_MAX];
    static char lockFile[] = SaPRIVLOCK;
    int lockfd, fd;
    FILE *newdb;
    SaPrivFile *privFile;
    const SaPrivEnt *ent;
    int i;
    int nUsers;
    struct stat f;

    /*
     * Grab the lock file.
     */
    lockfd = open(lockFile, O_WRONLY | O_CREAT, 020600);
    if (lockfd == -1 || lockf(lockfd, F_LOCK, 0) != 0) {
	perror(lockFile);
	exit(1);
    }

    /*
     * Create a temporary file.
     */
    (void)SaStringFormat(tempFile, sizeof tempFile, "%sXXXXX", SaPRIVDB);
    fd = mkstemp(tempFile);
    if (fd == -1 || (newdb = fdopen(fd, "w")) == NULL) {
	perror(tempFile);
	exit(1);
    }

    /*
     * Preserve privilege database permissions.
     */
    if (stat(SaPRIVDB, &f) == -1) {
	perror(SaPRIVDB);
	exit(1);
    }
    (void)fchmod(fd, f.st_mode);
    
    privFile = SaOpenPrivDB();
    if (!privFile) {
	perror(SaPRIVDB);
	exit(1);
    }

    /*
     * Copy the database, except for entries that match
     * privilege/userName.
     */
    while ((ent = SaGetPrivEnt(privFile)) != NULL) {
	/*
	 * Skip copying this privilege if we're getting rid of it
	 * altogether.
	 */
	bool removeThisPriv = false;
	int j;
	for (j = 0; j < gNumPrivileges; j++) {
	    if (strcmp(ent->privilege, gPrivileges[j]) == 0) {
		removeThisPriv = true;
	    }
	}

	if (gRemoveAll && removeThisPriv) {
	    continue;
	}

	(void)fprintf(newdb, "%s:", ent->privilege);
	nUsers = 0;
	for (i = 0; i < ent->numUsers; i++) {
	    if ((removeThisPriv == false && gRemoveAcct == false)
		|| strcmp(ent->users[i], gUserName) != 0) {
		if (nUsers) {
		    (void)fputc(',', newdb);
		}
		(void)fprintf(newdb, "%s", ent->users[i]);
		nUsers++;
	    }
	}
	(void)fputc('\n', newdb);
    }
    SaClosePrivDB(privFile);

    /*
     * Make sure everything was cool.
     */
    if ((ferror(newdb) || fclose(newdb) == EOF)) {
	perror(tempFile);
	exit(1);
    }
			
    /*
     * Replace the old database with the new database.
     */
    if (rename(tempFile, SaPRIVDB) == -1) {
	perror(SaPRIVDB);
	exit(1);
    }

    /*
     * Release the lock.
     */
    (void)lockf(lockfd, F_ULOCK, 0);
    (void)close(lockfd);
}

/*
 * rmpriv main program.
 */
void main(int argc, char *argv[])
{
    int i;
    pid_t pid;
    int status;

    SaPrivFixUmask();


    ParseArgs(argc, argv);
    Authenticate();

    if (gChkConfig) {
	pid = fork();
	if (pid == -1) {
	    perror("fork");
	    exit(1);
	}
	if (pid == 0) {
	    execl("/etc/chkconfig", "chkconfig", "privileges", "off", NULL);
	    perror("/etc/chkconfig");
	    exit(1);
	}
	if (waitpid(pid, &status, 0) != pid) {
	    perror("/etc/chkconfig");
	    exit(1);
	}
	if (!WIFEXITED(status) || WEXITSTATUS(status) != 1) {
	    exit(1);
	}
	exit(0);
    }

    for (i = 0; i < gNumPrivileges; i++) {
	/*
	 * We are not calling SaPrivValidatePrivName() in this program
	 * because we want to let the user remove privileges which are
	 * not installed on the system.  That way you can prevent
	 * crufty privileges  from coming back to haunt you if
	 * something is deleted and reinstalled.
	 */
	if (SaIsDefaultPrivilege(gPrivileges[i])) {
	    fprintf(stderr, i18n("warning: \"%s\" is a default privilege.\n"
	        "\t You must also run rmdefpriv(1M) in order to prevent\n"
	        "\t users from running it.\n"), gPrivileges[i]);
	}
    }

    if (!gRemoveAll && SaIsPrivilegedUser(gUserName)) {
	fprintf(stderr, i18n("warning: \"%s\" is a privileged user.\n"
	    "\t You must also run rmprivuser(1M) in order to\n"
	    "\t revoke privileges from this user.\n"), gUserName);
    }

    RemovePrivileges();
    exit(0);
}
