/*
 * rmprivuser.c
 *
 *	Remove a privileged user from the system.
 *
 *
 *  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.10 $"

#include <sys/types.h>
#include <sys/stat.h>

#include <locale.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <stdio.h>
#include <errno.h>
#include <locale.h>
#include <fcntl.h>

#include <sysadm/i18n.h>
#include <sysadm/privilege.h>
#include <sysadm/auth.h>
#include <sysadm/format.h>

static char *gAuthScheme;
static char *gUser;

static int gLockfd;

/*
 *  static void ParseArgs(int argc, char *argv[])
 *
 *  Description:
 *      Parse rmprivuser command line arguments.  Usage is:
 *          rmprivuser [ -auth authScheme ] <user>
 *
 *  Parameters:
 *      argc  Number of args
 *      argv  Vector of args
 */
static void ParseArgs(int argc, char *argv[])
{
    /*
     * Watch out for people mucking with us.  Remember, we're a setuid
     * root program.
     */
    if (argc < 1) {
	exit(SaPRIVERROR);
    }

    if (argc < 2) {
	fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <user>\n"),
	        SaRMPRIVUSER);
	exit(1);
    }

    if (strcmp(argv[1], "-auth") == 0) {
	if (argc != 4) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <user>\n"),
	            SaRMPRIVUSER);
	    exit(1);
	}
	gAuthScheme = argv[2];
	gUser = argv[3];
    } else {
	if (argc != 2) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <user>\n"),
	            SaRMPRIVUSER);
	    exit(1);
	}
	gUser = argv[1];
    }

    /*
     * It's hopeless to run rmprivuser if you're not root without
     * providing an authentication scheme.
     */
    if (getuid() && !gAuthScheme) {
	fprintf(stderr, "%s\n",
	        i18n("non-root users must supply -auth argument."));
	exit(1);
    }
}

/*
 *  static void Authenticate(void)
 *
 *  Description:
 *      Authenticate the user.
 */
static void Authenticate(void)
{
    if (!gAuthScheme) {
	/*
	 * This should never happen.  This is an assert which won't
	 * compile out.
	 */
	if (getuid()) {
	    exit(SaPRIVERROR);
	}
	return;
    }

    if (!SaPrivAuthenticate(gAuthScheme)) {
	fprintf(stderr, "%s\n", i18n("Authentication failed."));
	exit(1);
    }
}

/*
 *  static void LockDB(void)
 *
 *  Description:
 *      Lock the privileged user database, so that no other programs will
 *      modify it.  Locking is advisory; only programs that
 *      participate in our locking scheme which uses a separate lock
 *      file will be locked out.
 */
static void LockDB(void)
{
    gLockfd = open(SaPRIVUSERLOCK, O_WRONLY | O_CREAT, 020600);
    if (gLockfd == -1 || lockf(gLockfd, F_LOCK, 0) != 0) {
	perror(SaPRIVUSERLOCK);
	exit(1);
    }
}

/*
 *  void UnLockDB(void)
 *
 *  Description:
 *      Release our lock on the privileged user database.
 */
void UnLockDB(void)
{
    (void)lockf(gLockfd, F_ULOCK, 0);
    (void)close(gLockfd);
}

/*
 *  static void RemovePrivUser(void)
 *
 *  Description:
 *      Remove a user from the database of privileged users.
 */
static void RemovePrivUser(void)
{
    FILE *fp, *newdb;
    char tempFile[PATH_MAX];
    char buf[80], *pc;
    bool itWasThere = false;
    struct stat f;
    int fd;

    (void)SaStringFormat(tempFile, sizeof tempFile, "%sXXXXXX", SaPRIVUSERDB);
    fd = mkstemp(tempFile);
    if (fd == -1 || (newdb = fdopen(fd, "w")) == NULL) {
	perror(tempFile);
	exit(1);
    }

    /*
     * Preserve privilege database permissions.
     */
    if (stat(SaPRIVUSERDB, &f) == 0) {
	(void)fchmod(fd, f.st_mode);
    } else {
	(void)fchmod(fd, 0644);
    }

    /*
     * Copy existing data base.
     */
    if ((fp = fopen(SaPRIVUSERDB, "r")) != NULL) {
	while (fgets(buf, sizeof buf, fp) != NULL) {
	    pc = strchr(buf, '\n');
	    if (pc) {
		*pc = '\0';
	    }
	    if (strcmp(buf, gUser) == 0) {
		itWasThere = true;
	    } else {
		(void)fprintf(newdb, "%s\n", buf);
	    }
	}
	if (ferror(fp)) {
	    perror(SaPRIVUSERDB);
	    unlink(tempFile);
	    exit(1);
	}
	(void)fclose(fp);
    }

    /*
     * If our entry wasn't there, don't do anything.
     */
    if (!itWasThere) {
	fclose(newdb);
	unlink(tempFile);
	return;
    }
	
    if (ferror(newdb)) {
	perror(tempFile);
	unlink(tempFile);
	exit(1);
    }

    (void)fclose(newdb);

    /*
     * Replace existing database with new database.
     */
    if (rename(tempFile, SaPRIVUSERDB) == -1) {
	perror(SaPRIVUSERDB);
	exit(1);
    }
}

/*
 * rmprivuser main program.
 */
void main(int argc, char *argv[])
{
    SaPrivFixUmask();


    /*
     * Don't even try if we're not setuid root.
     */
    if (geteuid() != 0) {
	fprintf(stderr, "%s\n", i18n("setuid root bit is off"));
	exit(SaPRIVNOTSETUID);
    }

    ParseArgs(argc, argv);
    Authenticate();
    LockDB();
    RemovePrivUser();
    UnLockDB();

    exit(0);
}
