/*
 * runpriv.c
 *
 *	Run a privileged command if the user is authorized.  Heavy
 *	precautions must be taken, since this is a setuid root
 *	program.
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.28 $"

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <unistd.h>
#include <string.h>
#include <locale.h>
#include <pwd.h>
#include <assert.h>

#ifdef RUNPRIV_TEST
/*
 * See comment for AlarmHandler(), below.
 */
#include <signal.h>
#endif

#include <sysadm/i18n.h>
#include <sysadm/privilege.h>
#include <sysadm/salog.h>
#include <sysadm/auth.h>
#include <sysadm/format.h>

static char *gPrivilege;        /* The privilege user is trying to perform */
static char **gNewEnviron;      /* environment for privileged command */
static char **gCmdArgs;		/* Arguments for privileged command */
static int gNumArgs;		/* Number of arguments */


/*
 *  static void ParseArgs(int argc, char *argv[])
 *
 *  Description:
 *      Parse command line arguments.  Usage is "runpriv <privilege> ...",
 *      where ... are args to be passed to privilege program, if any.
 *
 *  Parameters:
 *      argc  # of command line args
 *      argv  command line args
 */
static void ParseArgs(int argc, char *argv[])
{
    char *authScheme = NULL;
    /*
     * If someone's messing with us, bail immediately
     */
    if (argc < 1) {
	exit(SaPRIVERROR);
    }
    
    if (argc < 2) {
	fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <privilege> ...\n"),
	        SaRUNPRIV);
	exit(SaPRIVUSAGE);
    }

    argv++;

    if (strcmp(*argv, "-auth") == 0) {
	if (argc < 4) {
	    fprintf(stderr, i18n("usage: %s [ -auth <auth> ] <privilege> ...\n"),
	            SaRUNPRIV);
	    exit(SaPRIVUSAGE);
	}
	argv++;			/* Skip "-auth" */
	authScheme = *argv++;
	argc -= 2;
    }

    gPrivilege = *argv;
    SaPrivValidatePrivName(gPrivilege);
    gCmdArgs = argv;		/* privilege is also first arg */
    gNumArgs = argc - 1;

    /*
     * Make sure that if the privilege database exists, it's not
     * writable by anyone other than root.  Also check default
     * privilege dir and privileged user db.
     */
    if (access(SaPRIVDB, F_OK) == 0) {
	SaPrivCheckFile(SaPRIVDB);
    }
    if (access(SaPRIVDEFDIR, F_OK) == 0) {
	SaPrivCheckFile(SaPRIVDEFDIR);
    }
    if (access(SaPRIVUSERDB, F_OK) == 0) {
	SaPrivCheckFile(SaPRIVUSERDB);
    }

    /*
     * Always do authentication if "-auth" flag was passed, because
     * caller might be a program which will unconditionally send auth
     * parameters over stdin and this could hose the privileged
     * program if we don't gobble them up.  If user has an entry in
     * the privileged database, it doesn't matter what authentication
     * parameters were provided.
     */
    if (!(authScheme && SaPrivAuthenticate(authScheme)
	|| SaCheckPriv(getuid(), (const char **)&gPrivilege, 1))) {
	fprintf(stderr, i18n("%s: not privileged"), gPrivilege);
	exit(SaPRIVNOTPRIV);
    }
}

/*
 *  static void SetupEnvironment(void)
 *
 *  Description:
 *      Build an environment for the privileged program.  The
 *      environment is set up in the file SaPRIVENV, which consists
 *      of lines of the form
 *      
 *          VAR=value
 *      
 *      for environment variables that must be set to a specific
 *      value, or lines of the form
 *      
 *          VAR
 *      
 *      for environment variables that take their values from our
 *      environment.
 *      
 *      No environment variables that are not found in SaPRIVENV will
 *      be available in the privileged program.  This is to prevent
 *      hackers from spoofing us using environment variables.
 */
static void SetupEnvironment(void)
{
    FILE *fp;
    char line[2048];
    char *equal, *envStr, *newLine;
    int nVars;

    SaPrivCheckFile(SaPRIVENV);
    fp = fopen(SaPRIVENV, "r");
    if (fp == NULL) {
	perror(SaPRIVENV);
	exit(SaPRIVNOENV);
    }

    /*
     * Build environment on the heap as we go.
     */
    gNewEnviron = malloc(sizeof **gNewEnviron);

    gNewEnviron[0] = NULL;

    nVars = 0;
    while (fgets(line, sizeof line - 1, fp) != NULL) {
	newLine = strrchr(line, '\n');
	if (newLine) {
	    *newLine = '\0';
	}
	equal = strchr(line, '=');
	if (equal) {
	    /*
	     * It's a variable with a value
	     */
	    envStr = line;
	} else {
	    /*
	     * It's a variable without a value, which means we get the
	     * value out of the current environment.
	     */
	    envStr = getenv(line);
	    if (envStr) {
		/*
		 * Don't put NULL variables in the environment
		 */
		if (strlen(envStr)) {
		    envStr -= strlen(line) + 1;
		} else {
		    envStr = NULL;
		}
	    }
	}

	if (envStr) {
	    /*
	     * Add the new variable to the privileged program's
	     * environment.
	     */
	    nVars++;
	    gNewEnviron
		    = realloc(gNewEnviron, (nVars + 1) * sizeof *gNewEnviron);
	    gNewEnviron[nVars - 1] = strdup(envStr);
	    gNewEnviron[nVars] = NULL;
	}
    }
}

/*
 *  static char *QuoteArg(const char *arg)
 *
 *  Description:
 *      Quote a command line argument so that it can be cut-n-pasted
 *      from the log viewer to be re-run.
 *
 *  Parameters:
 *      arg
 *
 *  Returns:
 *	arg, with quotes as necessary, in a newly malloced buffer.
 */
static char *QuoteArg(const char *arg)
{
    int len;
    const char *pc;
    char *quotedArg, *dest;
    bool hasSpaces = false;
    bool hasPercents = false;

    len = 0;
    pc = arg;

    /*
     * Take a pass through "arg", looking for spaces, quotes and
     * percents. If there are no spaces, we don't care about the
     * quotes, but we look at both in this loop to avoid having to loop
     * again.
     *
     * len keeps track of the length of the quoted string.  We add one
     * to len for each character in the unquoted string, and an extra
     * one each time we find a " or % character (because those
     characters will turn into two characters in the quoted string).
     */
    while (*pc) {
	if (*pc == '"') {
	    len++;
	} else if (*pc == ' ') {
	  hasSpaces = true;
	} else if (*pc == '%') {
	  hasPercents = true;
	  len++;
	}
	pc++;
	len++;
    }

    /*
     * No spaces or percents means there's nothing we need to quote.
     */
    if (!hasSpaces && !hasPercents) {
	return strdup(arg);
    }

    if (hasSpaces) {
      len += 2;			/* quote at beginning and end */ 
    }

    quotedArg = malloc(len + 1);
    dest = quotedArg;

    if (hasSpaces) {
      *dest++ = '"';
    }

    for (pc = arg; *pc; pc++) {
	if (*pc == '"' && (pc == arg || *(pc - 1) != '\\')) {
	    *dest++ = '\\';
	} else if (*pc == '%') {
	  *dest++ = '%';
	}
	*dest++ = *pc;
    }

    if (hasSpaces) {
      *dest++ = '"';
    }
    *dest = '\0';

    assert(dest - quotedArg < len + 1);

    return quotedArg;
}

/*
 *  static void LogCommand(void)
 *
 *  Description:
 *      Write a message to syslog that a user is being allowed to run
 *      a privileged operation, and log a message to the system admin
 *      log describing the exact command that was run.
 */
static void LogCommand(void)
{
    SaLog *log;
    struct passwd *pwd;
    int len, i;
    SaLogErrorCode code;
    char *msg, **arg, **argv;

    pwd = getpwuid(getuid());
    if (pwd == NULL) {
	fprintf(stderr, i18n("%s: not privileged"), gPrivilege);
	exit(SaPRIVNOTPRIV);
    }

    /*
     * Log a message to syslog about the command to be run.
     */
    openlog("runpriv", LOG_CONS | LOG_PID, LOG_USER);
    syslog(LOG_INFO, i18n("Running privilege %s for user %s."), gPrivilege,
	   pwd->pw_name);

    /*
     * Log a message to the system admin log.
     */
    if ((code = SaLogOpen(gPrivilege, &log)) != SaLogErrNoError) {
	(void)fprintf(stderr, "%s: salog: %s\n", SaRUNPRIV,
		      SaLogErrorCodeToString(code));
	exit(SaPRIVNOLOG);
    }
	    
    len = strlen(SaRUNPRIV) + 1 + strlen(gPrivilege) + 1;

    if (gNumArgs > 1) {
	arg = gCmdArgs + 1;
	argv = malloc(sizeof *argv * gNumArgs);
	i = 0;
	while (*arg) {
	    argv[i] = QuoteArg(*arg);
	    arg++;
	    len += strlen(argv[i]) + 1;
	    i++;
	}
	
	argv[i++] = NULL;
	assert(i == gNumArgs);
    }

    msg = malloc(len);
    (void)sprintf(msg, "%s %s", SaRUNPRIV, gPrivilege);

    if (gNumArgs > 1) {
	arg = argv;
	while (*arg) {
	    (void)strcat(msg, " ");
	    (void)strcat(msg, *arg);
	    free(*arg);
	    arg++;
	}
	free(argv);
    }

    /*
     * Make sure we did our math right.
     */
    assert(len == strlen(msg) + 1);

    if (SaLogWriteMsg(log, SaLogCommand, msg) == SaLogStatusError) {
	(void)fprintf(stderr, "runpriv: %s\n", SaLogGetErrorString(log));
	exit(SaPRIVERROR);
    }		    
    free(msg);
    SaLogClose(log);
}

/*
 *  static void RunCommand(void)
 *
 *  Description:
 *      Run the privileged command.  The command name is formed by
 *      appending the name of the privilege to SaPRIVCMDDIR, the
 *      directory where privileged commands are kept.  Programs
 *      outside of this directory cannot be executed by runpriv.
 */
static void RunCommand(void)
{
    char cmdPath[PATH_MAX];

    /*
     * Don't run anything that's not in SaPRIVCMDDIR.  This check
     * protects against "runpriv ../../usr/people/rogerc/testpriv/priv"
     */
    if (strchr(gPrivilege, '/') != NULL) {
	exit(SaPRIVERROR);
    }

    /*
     * Protect against someone trying to get us to overwrite our stack.
     */
    if (strlen(SaPRIVCMDDIR) + strlen(gPrivilege) + 2 >
	sizeof cmdPath) {
	exit(SaPRIVERROR);
    }

    (void)SaStringFormat(cmdPath, sizeof cmdPath, "%s/%s", SaPRIVCMDDIR, gPrivilege);

    SaPrivCheckFile(cmdPath);

    /*
     * Don't log default privileges; otherwise logs will grow really
     * fast.
     */
    if (!SaIsDefaultPrivilege(gPrivilege)) {
	LogCommand();
    }

    (void)execve(cmdPath, gCmdArgs, gNewEnviron);
    perror(cmdPath);
    exit(SaPRIVEXECFAIL);
}

#ifdef RUNPRIV_TEST
/*
 * Alarm handler.  This is here so runpriv can be run setuid root, and
 * then attached to with a debugger when it sleeps in pause().  Once
 * runpriv has been attached to, it can be sent SIGALRM to wake it up.
 * Then runpriv will be running in the debugger and can be single
 * stepped, etc.
 */
__sigret_t AlarmHandler(_sigargs)
{
}
#endif

/*
 * runpriv main program.
 */
int main(int argc, char *argv[])
{
#ifdef RUNPRIV_TEST
    sigaction_t act;

    act.sa_flags = 0;
    act.sa_handler = AlarmHandler;
    (void)sigemptyset(&act.sa_mask);
    (void)sigaction(SIGALRM, &act, NULL);

    pause();
#endif

    SaPrivFixUmask();


    /*
     * Don't even try if we're not setuid root.
     */
    if (geteuid() != 0) {
	fprintf(stderr, "%s\n", i18n("setuid root bit is off"));
	exit(SaPRIVNOTSETUID);
    }

    ParseArgs(argc, argv);

    SaPrivCheckFile(SaPRIVHOME);

    if (chdir(SaPRIVHOME) == -1) {
	perror(SaPRIVHOME);
	exit(SaPRIVNOHOME);
    }

    SetupEnvironment();
    RunCommand();
}
