//
// SysadmListener.c++
//
//	PacketListener that start sysadm services.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.18 $"

#include <dlfcn.h>
#include <unistd.h>
#include <limits.h>
#include <syslog.h>
#include <string.h>
#include <sys/param.h>
#include <errno.h>

#include <sysadm/Log.h>
#include <sysadm/i18n.h>
#include <sysadm/Authenticator.h>
#include <sysadm/AppContext.h>
#include <sysadm/format.h>

#include "FileLogListener.h"
#include "SysadmListener.h"
#include "SyslogListener.h"

// This array must contain strings representing all of the versions of
// the Rhino protocol that the server will support.  Every time the
// product is re-released, add the new version here, and in the GUI.
//
// Version changes:
// 1.1 -> 1.2  Add support for versioning via the SYSADM_VERSION request.  In
// order to make sure that clients version themselves, we no longer
// talk to 1.1 clients.
//
// 1.2 -> 1.3  Add support for product versioning via the
// PRODUCT_VERSION request.    In order to make sure that clients
// version themselves, we no longer talk to 1.2 clients.
//
// 1.3 -> 1.4  Minor changes for IRIX 6.5.9 and CXFS/FailSafe coexecution.
//
static const char * versionsSupported[]  = {"1.4"};
//
// The Service structure is used to keep track of services that are
// loaded at runtime.
//
class Service {
  public:
    Service(void* dso)
    :  _dso(dso), _refCount(0)
    {
    }
    ~Service()
    {
	dlclose(_dso);
    }

    void* _dso;
    int _refCount;
};

//
// Get notified of the results of our attempts to authenticate the
// client.
//
class SysadmAuthResultListener : public ResultListener {
  public:
    SysadmAuthResultListener(SysadmListener& listener)
    : _listener(listener)
    {
    }

    virtual void succeeded(const ResultEvent&)
    {
	_listener._authenticated = true;
    }

    virtual void failed(const ResultEvent&)
    {
	Log::fatal(SysadmListener::SYSADM_SERVICE,
		   i18n("Login attempt from host \"%s\" failed."),
		   (const char*)_listener._conn.remoteHostName());
	AppContext::getAppContext().exit(1);
    }

    SysadmListener& _listener;
};

const char* SysadmListener::SYSADM_SERVICE = "sysadmd";

const char* SysadmListener::REMOTE_HOST = "remoteHostname";

const char* SysadmListener::AUTH_SCHEME = "authScheme";
const char* SysadmListener::LOAD = "load";
const char* SysadmListener::UNLOAD = "unload";
const char* SysadmListener::LOAD_AUTH = "loadAuth";
const char* SysadmListener::SET_REMOTE_HOST = "setRemoteHost";
const char* SysadmListener::GET_REMOTE_HOST = "getRemoteHost";
const char* SysadmListener::PING = "ping";
const char* SysadmListener::SYSADM_VERSION = "version";

const char* SysadmListener::COOKIE = "cookie";
const char* SysadmListener::RESULT = "result";
const char* SysadmListener::REASON = "reason";

const char* SysadmListener::PRODUCT = "product";
const char* SysadmListener::PRODUCT_VERSION = "productVersion";
const char* SysadmListener::SERVER_PRODUCT_VERSION = "serverProductVersion";
const char* SysadmListener::CLIENT_PRODUCT_VERSION = "clientProductVersion";

//
// SysadmListener constructor.
//
SysadmListener::SysadmListener(Connection& conn)
: _conn(conn), _authenticated(false), _auth(NULL), _versionOk(false)
{
    _authSchemeList = new CollectionOf<String>;
    _authSchemeList->add(new String("unix"));
}

//
// SysadmListener destructor.
//
SysadmListener::~SysadmListener()
{
    RemoveAndDestroyContentsOf(_authSchemeList);
    delete _authSchemeList;
    RemoveAndDestroyContentsOf(&_services);
    delete _auth;
}

//
//  void SysadmListener::adoptAuthSchemeList(CollectionOf<String>* authSchemeList)
//
//  Description:
//      Set the list of allowable authentication schemes.
//
//  Parameters:
//      authSchemeList  List of allowable authentication schemes.
//
void SysadmListener::adoptAuthSchemeList(CollectionOf<String>* authSchemeList)
{
    RemoveAndDestroyContentsOf(_authSchemeList);
    delete _authSchemeList;
    _authSchemeList = authSchemeList;
}

//
//  void SysadmListener::receivePacket(const Packet& packet)
//
//  Description:
//      Called by Connection when a packet is received.  Dispatch the
//      packet to the appropriate method.  Note that the only request
//      we will respond to is AUTH_SCHEME until the client has been
//      authenticated.
//
//  Parameters:
//      packet  Request from client.
//
void SysadmListener::receivePacket(const Packet& packet)
{
    assert(packet.getType() == SYSADM_SERVICE);

    if (packet.getSelector() == PING) {
	return;
    }

    // This operation is valid before authentication
    if (packet.getSelector() == SET_REMOTE_HOST) {
	setRemoteHost(packet);
	return;
    }

    // This operation is valid before authentication
    if (packet.getSelector() == GET_REMOTE_HOST) {
	getRemoteHost(packet);
	return;
    }

    if (packet.getSelector() == LOAD_AUTH) {
	loadAuthScheme(packet);
	return;
    }

    // Do not accept any requests until the client has logged in.
    if (!_authenticated) {
	error(packet, "sysadmd.notAuthenticated");
	return;
    }
    
    if (packet.getSelector() == SYSADM_VERSION) {
	_versionOk = checkVersion(packet);
	return;
    }

    if (!_versionOk) {
	// This is the case where the client (probably an old
	// pre-versioning client) skipped the versioning step.  We'll
	// send them back a message and refuse to go further.
	error(packet, "Version mismatch between the versions of "
	      "sysadm_base on the server and the client.  Please update "
	      "the client.");
	return;
    }

    if (packet.getSelector() == LOAD) {
	load(packet);
    } else if (packet.getSelector() == UNLOAD) {
	unload(packet);
    } else if (packet.getSelector() == PRODUCT_VERSION) {
	checkProductVersion(packet);
    } else {
	Log::error(SYSADM_SERVICE, i18n("Unknown request: %s."),
		   (const char*)packet.getSelector());
	error(packet, "sysadmd.unknownRequest");
    }
}

//
//  void SysadmListener::loadAuthScheme(const Packet& packet)
//
//  Description:
//      Load an authentication scheme.  This is the first step in
//      authenticating the user; the authentication scheme we load
//      will get authentication information from the client.
//
//  Parameters:
//      packet  Specifies the authScheme to load.
//
void SysadmListener::loadAuthScheme(const Packet& packet)
{
    String authScheme = packet.getAttr(AUTH_SCHEME).stringValue();
    if (_auth != NULL) {
	assert(authScheme == _authScheme);
    } else {
	checkAuthSchemeName(authScheme);

	char dsoName[PATH_MAX];
	SaStringFormat(dsoName, sizeof dsoName,
		 "%s/%sServer.so",
		 CONFIG_SYSADM_AUTHSCHEME_DIR, (const char*)authScheme);
	void* dso = dlopen(dsoName, RTLD_NOW);
	if (dso == NULL) {
	    error(packet, "sysadmd.noAuthDSO");
	    return;
	}

	Authenticator* (*createAuthenticator)() =
	    (Authenticator* (*)())dlsym(dso, "CreateAuthenticator");
	if (createAuthenticator == NULL) {
	    error(packet, "sysadmd.noAuthCreator");
	    dlclose(dso);
	    return;
	}

	Authenticator* auth = createAuthenticator();
	auth->setConnection(&_conn);
	auth->authenticate(new SysadmAuthResultListener(*this));
	_auth = new Service(dso);
	_authScheme = authScheme;
    }
    success(packet);
}

//
//  void SysadmListener::load(const Packet& packet)
//
//  Description:
//      Dynamically load a service on behalf of the client.  We
//      reference count loads so that a module in the client
//      can unload a service without having to worry about
//      whether another module is using it.
//
//  Parameters:
//      packet  Specifies the service to load.
//
void SysadmListener::load(const Packet& packet)
{
    String serviceName = packet.getAttr("service").stringValue();
    Service* service = _services.lookupByKey(serviceName);
    if (service == NULL) {
	checkServiceName(serviceName);

	Log::debug(SYSADM_SERVICE, i18n("Load request for service: %s"),
		   (const char*)serviceName);

	char dsoName[PATH_MAX];
	sprintf(dsoName, "%s/%s.so",
		CONFIG_SYSADM_SERVICE_DIR, (const char*)serviceName);
	void* dso = dlopen(dsoName, RTLD_NOW);
	if (dso == NULL) {
	    Log::error(SYSADM_SERVICE, i18n("Load request for %s failed."),
		       (const char*)serviceName);
	    error(packet, dlerror());
	    return;
	}
	PacketListener* (*createPacketListener)(Connection&, const String&) =
	    (PacketListener* (*)(Connection&, const String&))
	    dlsym(dso, "CreatePacketListener");

	if (createPacketListener == NULL) {
	    Log::error(SYSADM_SERVICE, i18n("Load request for %s failed."),
		       (const char*)serviceName);
	    error(packet, dlerror());
	    dlclose(dso);
	    return;
	}

	PacketListener* listener = createPacketListener(_conn, serviceName);
	if (listener == NULL) {
	    Log::error(SYSADM_SERVICE, i18n("Load request for %s failed."),
		       (const char*)serviceName);
	    error(packet, "sysadmd.createPacketListenerFailed");
	    dlclose(dso);
	    return;
	}

	_conn.adoptPacketListener(serviceName, listener);
	service = new Service(dso);
	_services.add(service, new String(serviceName));
	Log::debug(SYSADM_SERVICE, i18n("Load request for %s succeeded."),
		   (const char*)serviceName);
    }
    service->_refCount++;
    success(packet);
}

//
//  void SysadmListener::unload(const Packet& packet)
//
//  Description:
//      Decrement a service's reference count.  Unload it if the
//      reference count goes to 0.
//
//  Parameters:
//      packet
//
void SysadmListener::unload(const Packet& packet)
{
    String serviceName = packet.getAttr("service").stringValue();
    Service* service = _services.lookupByKey(serviceName);
    if (service == NULL) {
	error(packet, "sysadmd.noSuchService");
	return;
    }
    service->_refCount--;
    if (service->_refCount == 0) {
	delete _conn.orphanPacketListener(serviceName);
	delete _services.remove(serviceName);
    }
    success(packet);
}

//
//  void SysadmListener::setRemoteHost(const Packet& packet)
//
//  Description:
//      Set the name of the remote host.  This is required
//      when the socket peer of _conn is not the actual client.
//
//  Parameters:
//      packet
//
void SysadmListener::setRemoteHost(const Packet& packet)
{
    String hostName = packet.getAttr(REMOTE_HOST).stringValue();
    if (hostName.getLength(true) == 0) {
	error(packet,"sysadmd.invalidHostName");
	return;
    }

    Log::trace(SYSADM_SERVICE, "Setting host name to: %s",
	       (const char*)hostName);
    _conn.setRemoteHostName(hostName);
    SyslogListener::setRemoteHost(hostName);
    FileLogListener::setRemoteHost(hostName);

    // Don't i18n this syslog message; it's parsed by the listclients
    // command.
    syslog(LOG_INFO, "changed to host: %s",
	   (const char*)_conn.remoteHostName());

    success(packet);
}

bool SysadmListener::checkVersion(const Packet& packet) {
    const char * versionRequested = packet.getAttr(SYSADM_VERSION).stringValue();

    bool versionOk = false;
    Packet reply(packet.getType(), packet.getSelector());
    reply.setAttr(packet.getAttr(COOKIE));
    for (int ii = 0;
	 ii < sizeof(versionsSupported)/sizeof(versionsSupported[0]);
	 ii++ ) {
	reply.setAttr(Attribute(String(SYSADM_VERSION) + StringFromLong(ii), 
				versionsSupported[ii]));
	if (!strcmp(versionRequested,versionsSupported[ii])) {
	    versionOk = true;
	}
    }

    if (versionOk) {
	reply.setAttr(Attribute(RESULT, true));
    } else {
	reply.setAttr(Attribute(RESULT, false));
    }
    _conn.sendPacket(reply);
    return versionOk;
}

void SysadmListener::checkProductVersion(const Packet& packet) {
    char versionFile[PATH_MAX];
    char versionInstalled[PATH_MAX];

    String product = packet.getAttr(PRODUCT).stringValue();
    String versionRequired =
	packet.getAttr(CLIENT_PRODUCT_VERSION).stringValue();

    Packet reply(packet.getType(), packet.getSelector());
    reply.setAttr(packet.getAttr(COOKIE));
    reply.setAttr(packet.getAttr(PRODUCT));
    reply.setAttr(packet.getAttr(CLIENT_PRODUCT_VERSION));

    Log::debug(SYSADM_SERVICE,
	       "Checking version match for product %s, version %s",
	       (const char*)product, (const char*)versionRequired);
    SaStringFormat(versionFile, sizeof versionFile,
		   "%s/%s/version",
		   CONFIG_SYSADM_PRODUCTS_DIR, (const char*) product);
    int versionLen;
    if ((versionLen = readlink(versionFile, versionInstalled,
			       sizeof(versionInstalled))) == -1) {

	Log::debug(SYSADM_SERVICE, "Error checking version file %s.\n"
		   "The error was %s", versionFile,
		   strerror(errno));
	versionInstalled[0] = 0;
	reply.setAttr(Attribute(RESULT, false));	
    } else {
	versionInstalled[versionLen] = 0;
	Log::debug(SYSADM_SERVICE,
		   "The version found is %s", versionInstalled);
	if (strcmp(versionInstalled, versionRequired)) {
	    reply.setAttr(Attribute(RESULT, false));
	} else {
	    reply.setAttr(Attribute(RESULT, true));
	}
    }
    reply.setAttr(Attribute(SERVER_PRODUCT_VERSION, versionInstalled));
    _conn.sendPacket(reply);
}

//
//  void SysadmListener::getRemoteHost(const Packet& packet)
//
//  Description:
//      Get the name of this host.  This is required
//      when the socket peer of the client connection is
//      not connected directly to the server (this host)
//
//  Parameters:
//      packet
//

void SysadmListener::getRemoteHost(const Packet& packet)
{
    char host[MAXHOSTNAMELEN];
    if (gethostname(host, sizeof(host)) < 0) {
	error(packet,"sysadmd.unknownHostName");
	return;
    }

    Packet reply(packet.getType(), packet.getSelector());
    reply.setAttr(Attribute(REMOTE_HOST, host));
    reply.setAttr(packet.getAttr(COOKIE));
    reply.setAttr(Attribute(RESULT, true));
    _conn.sendPacket(reply);
}

//
//  void SysadmListener::success(const Packet& packet)
//
//  Description:
//      Return a successful result for the operation requested by
//      "packet".
//
//  Parameters:
//      packet  The request that succeeded.
//
void SysadmListener::success(const Packet& packet)
{
    Packet reply(packet.getType(), packet.getSelector());
    reply.setAttr(packet.getAttr(COOKIE));
    reply.setAttr(Attribute(RESULT, true));
    _conn.sendPacket(reply);
}

//
//  void SysadmListener::error(const Packet& packet, const String& reason)
//
//  Description:
//      Return an error for the operation requested by "packet".
//
//  Parameters:
//      packet  The request that failed.
//      reason  The reason for the failure.
//
void SysadmListener::error(const Packet& packet, const String& reason)
{
    Packet reply(packet.getType(), packet.getSelector());
    reply.setAttr(packet.getAttr(COOKIE));
    reply.setAttr(Attribute(RESULT, false));
    reply.setAttr(Attribute(REASON, reason));
    _conn.sendPacket(reply);
}

//
//  void SysadmListener::checkServiceName(const String& serviceName)
//
//  Description:
//      Make sure it's safe to load a service.  We restrict services
//      that we load to residing in the directories we think they
//      should be in.  The check for "/" prevents us from loading
//      services like "../../../usr/tmp/hackerService.so" and possibly
//      having a security problem.
//
//  Parameters:
//      serviceName  The service name to check.
//
void SysadmListener::checkServiceName(const String& serviceName)
{
    if (strchr(serviceName, '/') != NULL) {
	syslog(LOG_WARNING, i18n("Attempt by someone running on host \"%s\" "
	       "to get sysadmd(1M) to dlopen(\"%s\")."),
	       (const char *)_conn.remoteHostName(), (const char*)serviceName);
	// Don't use AppContext exit in this case; a hacker is messing
	// with us, so let's just bail immediately.
	exit(1);
    }
}

//
//  void SysadmListener::checkAuthSchemeName(const String& authSchemeName)
//
//  Description:
//      Make sure it's safe to load an auth scheme.  We only allow
//      auth schemes that were specified on the sysadmd command line,
//      or "unix" if none were specified.
//
//  Parameters:
//      authSchemeName  Name of auth scheme to check.
//
void SysadmListener::checkAuthSchemeName(const String& authSchemeName)
{
    checkServiceName(authSchemeName);
    IteratorOver<String> iter(_authSchemeList);
    String* scheme;
    while ((scheme = iter()) != NULL) {
	if (authSchemeName == *scheme) {
	    return;
	}
    }
    fprintf(stderr, "%s\n", (const char*)authSchemeName);
    char dsoName[PATH_MAX];
    SaStringFormat(dsoName, sizeof dsoName,
		   "%s/%sServer.so",
		   CONFIG_SYSADM_AUTHSCHEME_DIR, (const char*)authSchemeName);
    syslog(LOG_WARNING, i18n("Attempt by someone running on host \"%s\" "
	   "to get sysadmd(1M) to dlopen(\"%s\")."),
	   (const char *)_conn.remoteHostName(), dsoName);
    exit(1);
}
