//
// sysadmd.c++
//
//	Daemon for sysadm service.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.23 $"

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>

#include <signal.h>
#include <unistd.h>
#include <locale.h>
#include <fcntl.h>
#include <syslog.h>
#include <libgen.h>
#include <string.h>
#include <limits.h>
#include <dlfcn.h>
#include <assert.h>
#include <ctype.h>

#include <sysadm/AppContext.h>
#include <sysadm/Packet.h>
#include <sysadm/DictionaryOf.h>
#include <sysadm/LogFilter.h>
#include <sysadm/i18n.h>
#include <sysadm/format.h>

#include "SysadmListener.h"
#include "FileLogListener.h"
#include "SyslogListener.h"

USING_NAMESPACE(sysadm);

static const char* gConfigFile = CONFIG_SYSADM_VAR_DIR "/sysadmd.conf";
static const char* gProtocol = "tcpmux";
static const char* gProtocolArg = "";
static void* gConnDSO = NULL;
static Connection* gConn = NULL;
static CollectionOf<String>* gAuthSchemeList = NULL;

//
//  static LogFilter* ParseLogFilter(const char* arg)
//
//  Description:
//      Parse a log filter argument.  The format of a log filter
//      argument is a specification of the log mask followed by a
//      colon, followed by a comma-separated list of modules.  A
//      module is a string used by a component when calling one of the
//      Log methods.
//
//  Parameters:
//      arg  A log filter argument.
//
//  Returns:
//	A LogFilter object.
//
static LogFilter* ParseLogFilter(const char* arg)
{
    assert(arg != NULL);

    LogFilter* filter = new LogFilter;

    const char* pc = arg;
    int level = 0;
    while (*pc != '\0' && *pc != ':') {
	switch (*pc) {
	case 'f':
	case 'F':
	    level |= Log::FATAL;
	    break;
	case 'e':
	case 'E':
	    level |= Log::ERROR;
	    break;
	case 'w':
	case 'W':
	    level |= Log::WARNING;
	    break;
	case 'i':
	case 'I':
	    level |= Log::INFO;
	    break;
	case 'd':
	case 'D':
	    level |= Log::DEBUG;
	    break;
	case 't':
	case 'T':
	    level |= Log::TRACE;
	    break;
	}
	pc++;
    }
    filter->levelOn(level);
    if (*pc == ':') {
	filter->addModuleList(pc + 1);
    }
    return filter;
}

//
//  static void Usage()
//
//  Description:
//      Print a usage message.
//
static void Usage()
{
    fputs(i18n("usage: sysadmd [ -p protocol[:args] ] [ -s ] [ -f file ] "
          "[ -l f|e|w|i|d|t[:module[,module...]] ]\n"), stderr);
    exit(1);
}

//
//  static void ReadOptionsFile()
//
//  Description:
//      Read /var/sysadm/sysadmd.conf
//
static void ReadOptionsFile()
{
    int fd = open(gConfigFile, O_RDONLY);
    if (fd == -1) {
	return;
    }
    struct stat st;
    if (fstat(fd, &st) == -1) {
	close(fd);
	return;
    }
    const char* buf = (const char*)
	mmap(NULL, st.st_size, PROT_READ, MAP_SHARED, fd, 0);
    close(fd);
    if (buf == MAP_FAILED) {
	return;
    }
    int start = 0;
    int stop = 0;
    int line = 0;
    AppContext& app(AppContext::getAppContext());
    while (start < st.st_size) {
	line++;
	while (stop < st.st_size && buf[stop] != '\n') {
	    stop++;
	}
	if (buf[start] != '#') {
	    int colon = start;
	    while (colon < stop && buf[colon] != ':') {
		colon++;
	    }
	    if (colon < stop) {
		int keyStart = start;
		while (keyStart < colon && isspace(buf[keyStart])) {
		    keyStart++;
		}
		int keyStop = colon - 1;
		while (keyStop > keyStart && isspace(buf[keyStop])) {
		    keyStop--;
		}
		if (keyStop > keyStart) {
		    String key(buf + keyStart, keyStop - keyStart + 1);
		    if (app.getProperty(key) != String::NUL) {
			syslog(LOG_WARNING,
			    i18n("Ignoring duplicate key on line %d of %s"),
			    line, gConfigFile);
		    } else {
			int valStart = colon + 1;
			while (valStart < stop && isspace(buf[valStart])) {
			    valStart++;
			}
			int valStop = stop;
			while (valStop >= valStart && isspace(buf[valStop])) {
			    valStop--;
			}
			String value(buf + valStart, valStop - valStart + 1);
			app.setProperty(key, value);
		    }
		}
	    }
	}
	start = ++stop;
    }
    munmap((void*)buf, st.st_size);
}

//
//  static void ParseArgs(int argc, char* argv[])
//
//  Description:
//      Parse command line arguments.
//
//  Parameters:
//      argc   Number of command line arguments.
//      argv   Vector of command line arguments.
//
static void ParseArgs(int argc, char* argv[])
{
    LogListener* listener = NULL;
    bool sflag = false;
    bool fflag = false;

    int c;
    while ((c = getopt(argc, argv, "a:f:l:p:s")) != EOF) {
	switch (c) {
	case 'a':
	    if (gAuthSchemeList == NULL) {
		gAuthSchemeList = new CollectionOf<String>;
	    }
	    gAuthSchemeList->add(new String(optarg));
	    break;
	case 'f':
	    {
		FILE* fp = fopen(optarg, "a");
		if (fp == NULL) {
		    perror(optarg);
		} else {
		    listener = new FileLogListener(fp);
		    Log::getLog().adoptLogListener(listener);
		    fflag = true;
		}
	    }
	    break;
	case 'l':
	    if (listener == NULL) {
		Usage();
	    } else {
		listener->adoptFilter(ParseLogFilter(optarg));
	    }
	    break;
	case 'p':
	    gProtocol = optarg;
	    {
		char* colon = strchr(gProtocol, ':');
		if (colon != NULL) {
		    *colon = '\0';
		    gProtocolArg = colon + 1;
		}
	    }
	    break;
	case 's':
	    if (sflag) {
		Usage();
	    } else {
		sflag = true;
		listener = new SyslogListener;
		Log::getLog().adoptLogListener(listener);
	    }
	    break;
	default:
	    Usage();
	    break;
	}
    }
    // If log file was not specified on the command line, check config
    // file.
    AppContext& app(AppContext::getAppContext());
    if (!fflag) {
	String logFile = app.getProperty("logFile");
	if (logFile != String::NUL) {
	    FILE* fp = fopen(logFile, "a");
	    if (fp == NULL) {
		perror(logFile);
	    } else {
		listener = new FileLogListener(fp);
		Log::getLog().adoptLogListener(listener);
		String filter = app.getProperty("logFile.filter");
		if (filter != String::NUL) {
		    listener->adoptFilter(ParseLogFilter(filter));
		}
	    }
	}
    }
    // If syslog filter was not specified on the command line, check
    // config file.
    if (!sflag) {
	String filter = app.getProperty("syslog.filter");
	if (filter != String::NUL) {
	    listener = new SyslogListener;
	    listener->adoptFilter(ParseLogFilter(filter));
	    Log::getLog().adoptLogListener(listener);
	}
    }
}

//
//  static void CreateConnection()
//
//  Description:
//      Create a connection with the client.  We do this by dlopening
//      the dso for the protocol we've been told to use.  This allows
//      for the plugging in of different protocols.
//
static void CreateConnection()
{
    char dsoName[PATH_MAX];
    SaStringFormat(dsoName, sizeof dsoName,
	     "%s/%sServer.so", CONFIG_SYSADM_PROTOCOLS_DIR, gProtocol);
    gConnDSO = dlopen(dsoName, RTLD_NOW);
    if (gConnDSO == NULL) {
	Log::fatal(SysadmListener::SYSADM_SERVICE,
		   i18n("Unable to load protocol %s: %s"),
		   gProtocol, dlerror());
	exit(1);
    }

    Connection* (*createClientConnection)(const char*) =
	(Connection* (*)(const char*))dlsym(gConnDSO, "CreateClientConnection");
    if (createClientConnection == NULL) {
	Log::fatal(SysadmListener::SYSADM_SERVICE,
		   i18n("Unable to load protocol %s: %s"),
		   gProtocol, dlerror());
	exit(1);
    }
	
    gConn = createClientConnection(gProtocolArg);
    if (gConn == NULL) {
	Log::fatal(SysadmListener::SYSADM_SERVICE,
		   i18n("protocol %s CreateClientConnection failed."), gProtocol);
	exit(1);
    }

    SyslogListener::setRemoteHost(gConn->remoteHostName());
    FileLogListener::setRemoteHost(gConn->remoteHostName());

    SysadmListener* listener = new SysadmListener(*gConn);
    if (gAuthSchemeList != NULL) {
	listener->adoptAuthSchemeList(gAuthSchemeList);
	gAuthSchemeList = NULL;
    }
    gConn->adoptPacketListener(SysadmListener::SYSADM_SERVICE,
			       listener);
    // Don't i18n this syslog message; it's parsed by the listclients
    // command.
    syslog(LOG_INFO, "started for host: %s", (const char*)
	   gConn->remoteHostName());
}

//
// server for remote system admin.  This guy's job is to set up the
// Connection and the sysadmd service and go into AppContext's event
// loop.
//
int main(int argc, char* argv[])
{
    // Close stderr inherited from inetd.  Use /dev/console for
    // stderr.
    close(2);
    open("/dev/console", O_WRONLY);

    // Call openlog regardless of whether we're logging Log:: messages
    // to syslog, because we also log to syslog directly.
    openlog("sysadmd", LOG_PID, LOG_USER);

    ReadOptionsFile();
    ParseArgs(argc, argv);

    // We want default signal handling for SIGCHLD, because we always
    // waitpid to find out what happened to them.
    struct sigaction act;
    act.sa_handler = SIG_DFL;
    act.sa_flags = 0;
    sigemptyset(&act.sa_mask);
    (void)sigaction(SIGCHLD, &act, NULL);

    CreateConnection();

    Log::info(SysadmListener::SYSADM_SERVICE, i18n("sysadmd started."));

    int exitCode = AppContext::getAppContext().run();

    // We can't let gConn delete gListener, because gListener's
    // destructor will dlcose the dsos for all the other listeners
    // which would make their destructors uncallable.  gConn does not
    // delete its listeners in any predictable order.
    PacketListener* listener
	= gConn->orphanPacketListener(SysadmListener::SYSADM_SERVICE);
    delete gConn;
    dlclose(gConnDSO);

    // Now that all the other listeners are gone, we can delete
    // gListener and close all the dsos.
    delete listener;

    exit(exitCode);
}
