//  tcpmux - standalone TCP port service multiplexer (implements RFC 1078)
//  Copyright (c) 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//


//  Note that you don't need this on IRIX, as its inetd has tcpmux built in.
//
//  This implements RFC 1078, which describes the TCPMUX protocol.  Mark
//  Lottor, the author of RFC 1078, also wrote a sample implementation,
//  available at ftp://ftp.nw.com/nw/software/tcpmux.c.  (His version is
//  older than Linux, yet it compiles and runs on Linux without modification!
//  That is so cool.)
//
//  The difference between this version and Mark's is that this provides some
//  of the features of inetd: it can run services as users other than the one
//  it was started as (so your tcpmux services don't have to run as root),
//  and its services can be started with additional arguments.
//
//  You will need to have a configuration file; other than comments, each
//  line should contain the following fields:
//      service name
//      user[.group]
//      server program
//      server program arguments
//
//  Here's a sample tcpmux configuration file containing one service.
//
//      #
//      #  Sample tcpmux config file.
//      #
//      +tcpmux_echo nobody /bin/cat tcpmux_echo
//
//  Make sure your /etc/services has a line like the following:
//
//      tcpmux        1/tcp               # rfc-1078
//
//  You will also need to add the following lines to /etc/inetd.conf.  If you
//  put tcpmux in /usr/local/bin:
//
//      # Additional services may be provided through tcpmux.
//      tcpmux stream tcp nowait root /usr/local/bin/tcpmux tcpmux /path/to/tcpmux.conf
//
//  RFC 1078 says that the + or - line should be terminated with a <CRLF>,
//  so that's why the cout << "\r\n" stuff in the code below.

#include <iostream.h>
#include <fstream.h>
#include <sys/types.h>
#include <regex.h>
#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

//  You can put the configuration file wherever you want it, and pass it as
//  an argument from inetd.
#define CFG_FILE "/etc/tcpmux.conf"
#define MAX_LINE 2000
#define MAX_SERVER_ARGS 10

//                         +     service                    user           [.group]                       server                       server args
#define REGEX_SERVICE   "^(\\+)?([[:alnum:]_]+)[[:space:]]+([[:alnum:]_]+)(\\.[[:alnum:]_]+)?[[:space:]]+([[:alnum:]/._]+)[[:space:]]+([[:alnum:]/._].*)"
//  6 fields, including initial + and counting user[.group] as 2, plus one for
//  element 0 (entire match)
#define REGEX_SERVICE_FIELDS 7
#define FIELD_PLUS     1
#define FIELD_SERVICE  2
#define FIELD_USER     3
#define FIELD_GROUP    4
#define FIELD_SERVER   5
#define FIELD_SERVARGS 6

//  This doesn't handle embedded double quotes.  Sorry.
#define REGEX_ARGS   "([^[:space:]]+|\"[^\"]+\")"
//  1 field, plus one for element 0 (entire match)
#define REGEX_ARGS_FIELDS 2

void
getfield(char *dest, const char *src, const regmatch_t &fld)
{
    int len = fld.rm_eo - fld.rm_so;
    strncpy(dest, src + fld.rm_so, len);
    dest[len] = '\0';
}

int
main(int argc, char **argv)
{
    char linebuf[MAX_LINE + 1];
    char tmpbuf[MAX_LINE + 1];
    char service[MAX_LINE + 1];
    char server[MAX_LINE + 1];
    char *p;
    char *cfg_file = CFG_FILE;

    //  We only take one argument--the config file.
    if(argc > 1) cfg_file = argv[1];

    //  Mark's version does this... I don't know what happens if you
    //  don't do this.
    dup2(0, 1);

    //  Read the name of the service being requested.
    //  I don't know whether this handles \r\n.
    cin.getline(service, MAX_LINE);
    if(!cin.good())
    {
	cout << "-I couldn't read the service name.";
	return 1;
    }

    //  Remove trailing \r and/or \n
    p = service + strlen(service);
    while((--p >= service) && ((*p == '\r') || (*p == '\n'))) *p = '\0';
    if(p < service)
    {
	cout << "-I couldn't read the service name.\r\n";
	return 1;
    }

    ifstream cfg(cfg_file);
    if(!cfg.good())
    {
	cout << "-Sorry, no services today.\r\n";
	return 1;
    }

    regex_t preg;
    regmatch_t match[REGEX_SERVICE_FIELDS > REGEX_ARGS_FIELDS ? REGEX_SERVICE_FIELDS : REGEX_ARGS_FIELDS];
    if(regcomp(&preg, REGEX_SERVICE, REG_EXTENDED | REG_ICASE))
    {
	cout << "-gahh, dain bramage\r\n";
	return 1;
    }

    //  "HELP" is a reserved service name.
    if(strcasecmp(service, "help") == 0)
    {
	//  Mark's version doesn't do this, but the RFC kind of seems like
	//  it should.
	//cout << "+Services:\r\n";

	//  Run through the list of services, dumping them out
	for(cfg.getline(linebuf, MAX_LINE); cfg.good(); cfg.getline(linebuf, MAX_LINE))
	{
	    if((*linebuf == '\0') || (*linebuf == '#')) continue;
	    if(regexec(&preg, linebuf, REGEX_SERVICE_FIELDS, match, 0) == 0)
	    {
    	    getfield(tmpbuf, linebuf, match[FIELD_SERVICE]);
    	    cout << tmpbuf << "\n";
	    }
	    //  else the line didn't match our regex... weird.
	}
	cfg.close();
	return 0;
    }

    //  Run through the list of services, looking for a match
    int servlen = strlen(service);
    for(cfg.getline(linebuf, MAX_LINE); cfg.good(); cfg.getline(linebuf, MAX_LINE))
    {
	if((*linebuf == '\0') || (*linebuf == '#')) continue;
	if(regexec(&preg, linebuf, REGEX_SERVICE_FIELDS, match, 0) != 0) continue;

	if((servlen == (match[FIELD_SERVICE].rm_eo - match[FIELD_SERVICE].rm_so)) &&
	   (strncmp(service, linebuf + match[FIELD_SERVICE].rm_so, servlen) == 0))
	{
	    cfg.close();

	    //  Set our GID & UID.
	    getfield(tmpbuf, linebuf, match[FIELD_USER]);
	    struct passwd *pw = getpwnam(tmpbuf);
	    if(pw == NULL)
	    {
		cout << "-User for service not available\r\n";
		return 1;
	    }
	    uid_t uid = pw->pw_uid;
	    gid_t gid = pw->pw_gid;
	    //  free pw??
	    getfield(tmpbuf, linebuf, match[FIELD_GROUP]);
	    if(tmpbuf[0] == '.')
	    {
		struct group *gr = getgrnam(tmpbuf + 1);  // skip leading "."
		if(gr == NULL)
		{
		    cout << "-Group for service not available\r\n";
		    return 1;
		}
		gid = gr->gr_gid;
		//  free gr??
	    }
	    if(setgid(gid) != 0)
	    {
		cout << "-Could not set gid\r\n";
		return 1;
	    }
	    if(setuid(uid) != 0)
	    {
		cout << "-Could not set uid\r\n";
		return 1;
	    }

	    getfield(server, linebuf, match[FIELD_SERVER]);

	    if(match[FIELD_PLUS].rm_so != match[FIELD_PLUS].rm_eo)
	    {
		//  That means our "+" field has nonzero length, which means
		//  we want to print a + so that the service itself doesn't
		//  have to.  Just to keep from printing the + and then
		//  failing on the exec (and printing a -), let's make sure
		//  the server executable exists.
		struct stat st;
		if(stat(server, &st) == -1) // || (!(st.st_mode & S_IXUSR)) etc.
		{
		    cout << "-Server executable not available\r\n";
		    return 1;
		}

		cout << "+\r\n";
		cout.flush();
	    }

	    regfree(&preg);
	    getfield(tmpbuf, linebuf, match[FIELD_SERVARGS]);

	    //  This nasty & fragile chunk splits tmpbuf (containing the
	    //  server arguments) into individual strings for use with execv.
	    //  It might have been faster & safer (& simpler) to do this part
	    //  without regexec.  Note that the current regex used for this
	    //  is broken; it doesn't handle embedded double quotes, even if
	    //  they're escaped.
	    regex_t areg;
	    if(regcomp(&areg, REGEX_ARGS, REG_EXTENDED | REG_ICASE))
	    {
		cout << "-gahh, dain bramage compiling REGEX_ARGS\r\n";
		return 1;
	    }
	    char *sargv[MAX_SERVER_ARGS + 1];
	    int sargc = 0;
	    int offset = 0;
	    int arglen = strlen(tmpbuf);
	    while((offset < arglen) &&
    	      (sargc < MAX_SERVER_ARGS) &&
    	      (regexec(&areg, tmpbuf + offset, REGEX_ARGS_FIELDS, match, 0) == 0) &&
    	      (match[0].rm_so != -1))
	    {
		//  horrible, horrible.  If this argument is bounded by
		//  quotes, remove them.
		if((tmpbuf[offset + match[0].rm_so] == '"') &&
		   (tmpbuf[offset + match[0].rm_eo - 1] == '"'))
		{
			++match[0].rm_so;
			--match[0].rm_eo;
		}
		tmpbuf[offset + match[0].rm_eo] = '\0';
		sargv[sargc++] = tmpbuf + offset + match[0].rm_so;
		offset += match[0].rm_eo + 1;
	    }
	    regfree(&areg);
	    if(sargc == MAX_SERVER_ARGS)
	    {
		cout << "-More than " << dec << MAX_SERVER_ARGS << " server args!\r\n";
		return 1;
	    }
	    if(sargc == 0) sargv[sargc++] = server;
	    sargv[sargc] = NULL;

	    //cout << "uid " << dec << getuid() << ", gid " << getgid() << "\n";
	    //cout << "execv(\"" << server << "\", sargc == " << dec << sargc << ")...\n";
	    //for(int ii = 0; ii < sargc; ++ii) cout << "  sargv[" << dec << ii << "] == \"" << sargv[ii] << "\"" << endl;
	    //cout.flush();
	    //return 0;

	    execv(server, sargv);

	    //  If we're still here, the exec failed!
	    cout << "-Could not exec server.\r\n";
	    return 1;
	}

    }
    cout << "-No such service.\r\n";
    return 1;
}
