/*
 * auth.c
 *
 *	Authentication handshaking.
 *
 *
 *  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.10 $"

#include <limits.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <syslog.h>
#include <sysadm/i18n.h>
#include <sysadm/privilege.h>
#include <sysadm/auth.h>
#include <sysadm/format.h>

/*
 *  void SaPrivCheckFile(const char *file)
 *
 *  Description:
 *      Check a file to see if it's OK to open, dlopen, or exec it.
 *      It has to be owned by root and writable by no one else in
 *      order to be acceptable.
 *
 *  Parameters:
 *      file  Name of the file to check
 */
void SaPrivCheckFile(const char *file)
{
    struct stat st;
    char *dir, *slash;

    /*
     * Check the file...
     */
    if (stat(file, &st) == -1) {
	perror(file);
	exit(SaPRIVERROR);
    }

    if (st.st_uid != 0) {
	const char *msg = i18n("setuid root access to \"%s\" denied\n"
	        "  because it is not owned by root.\n"
	        "  Use the chown(1) command to fix ownership.\n");
	fprintf(stderr, msg, file);
	syslog(LOG_ERR, msg, file);
	exit(SaPRIVERROR);
    }

    if ((st.st_mode & 022) != 0) {
	const char *msg = i18n("setuid root access to \"%s\" denied\n"
	        "  because it is writable by group or others.\n"
	        "  Use the chmod(1) command fix access.\n");
	fprintf(stderr, msg, file);
	syslog(LOG_ERR, msg, file);
	exit(SaPRIVERROR);
    }	

    /*
     * Furthermore, make sure the directory the file is in is owned by
     * root and not writable by anyone else.
     */
    dir = strdup(file);
    slash = strrchr(dir, '/');
    if (!slash) {
	const char *msg = i18n("setuid root access to \"%s\" denied "
	        "because it is not an absolute path.\n");
	fprintf(stderr, msg, file);
	syslog(LOG_ERR, msg, file);
	exit(SaPRIVERROR);
    }

    *slash = '\0';

    if (stat(dir, &st) == -1) {
	perror(dir);
	exit(SaPRIVERROR);
    }

    if (st.st_uid != 0) {
	const char *msg = i18n("setuid root access to \"%s\" denied\n"
	        "  because it is not owned by root.\n"
	        "  Use the chown(1) command to fix ownership.\n");
	fprintf(stderr, msg, dir);
	syslog(LOG_ERR, msg, dir);
	exit(SaPRIVERROR);
    }

    if ((st.st_mode & 022) != 0) {
	const char *msg = i18n("setuid root access to \"%s\" denied\n"
	        "  because it is writable by group or others.\n"
	        "  Use the chmod(1) command fix access.\n");
	fprintf(stderr, msg, dir);
	syslog(LOG_ERR, msg, dir);
	exit(SaPRIVERROR);
    }

    free(dir);
}

/*
 *  bool SaPrivAuthenticate(const char *authScheme)
 *
 *  Description:
 *      Authenticate the user based on the specified authentication
 *      scheme.  This involves using "authScheme" to construct the
 *      name of a dso to open, and getting the authenticate function
 *      out of the dso, and calling it.  For UNIX, the authenticate
 *      function reads a line from stdin and checks to see if it's the
 *      root password.
 *
 *  Parameters:
 *      authScheme  Authentication scheme to use.  UNIX checks for
 *                  root password.  This is extensible for
 *                  environments where other authentication schemes
 *                  are desirable.
 *
 *  Returns:
 *	true if user is authenticated, false otherwise.
 */
bool SaPrivAuthenticate(const char *authScheme)
{
    char authdso[PATH_MAX];
    bool isAuth;
    SaAuth *auth;

    /*
     * Watch out for people trying to overflow our stack with
     * ridiculously long authentication scheme names.
     */
    if (strlen(SaPRIVAUTHDIR) + strlen(authScheme) + 13 > sizeof
	authdso) {
	exit(SaPRIVERROR);
    }

    (void)SaStringFormat(authdso, sizeof authdso,
		   "%s/%sReceiver.so", SaPRIVAUTHDIR, authScheme);
    SaPrivCheckFile(authdso);

    auth = SaAuthOpen(authScheme, SaAuthReceiver);
    if (auth == NULL) {
	fprintf(stderr, i18n("%s is not a valid authentication scheme.\n"),
	        authScheme);
	exit(SaPRIVNOTPRIV);
    }
    isAuth = SaAuthCheckParams(auth, 0, 1);

    SaAuthClose(auth);

    return isAuth;
}

/*
 *  void SaPrivFixUmask(void)
 *
 *  Description:
 *      Set the umask for a setuid program or privileged command.  We
 *      make sure that only the owner can write a file, and that the
 *      owner can read and write the file.  This gives the caller the
 *      flexibility to decide whether group and other can read or
 *      execute files that we create.
 */
void SaPrivFixUmask(void)
{
    (void)umask((umask(0) | 022) & ~0600);
}

/*
 *  void SaPrivValidatePrivName(const char* privName)
 *
 *  Description:
 *      Make sure privName is an acceptable privilege name.  Make sure
 *      it's not too long, and that the privilege actually exists.
 *
 *  Parameters:
 *      privName  Name of the privilege to check
 */
void SaPrivValidatePrivName(const char* privName)
{
    char privPath[PATH_MAX];

    if (strchr(privName, '/') != NULL || strlen(privName) > NAME_MAX) {
	exit(SaPRIVERROR);
    }
    SaStringFormat(privPath, sizeof privPath, "%s/%s", SaPRIVCMDDIR, privName);
    if (access(privPath, F_OK | X_OK) == -1) {
	perror(privPath);
	exit(1);
    }
}
