/*
 * checkpriv.c
 *
 *	Determine whether the user should be allowed to perform the
 *	requested privileged operation.
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.17 $"

#include <pwd.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>
#include <stdio.h>

#include <sysadm/privilege.h>
#include <sysadm/format.h>
#ifndef HAVE__GETPWENT_NO_SHADOW
#include <shadow.h>
#endif /* !HAVE__GETPWENT_NO_SHADOW */

/*
 * This is globally visible for testing purposes.
 */
char *_SaPrivChkConfig = SaPRIVCHKCONFIG;

/*
 *  bool SaIsDefaultPrivilege(const char *privilege)
 *
 *  Description:
 *      Determine whether "privilege" is a default privilege.  To be a
 *      default privilege, the first line of the file
 *      /var/sysadm/defaultPrivileges/privilege must be "on".
 *
 *  Parameters:
 *      privilege  The privilege to check
 *
 *  Returns:
 *	true if "privilege" is default, false otherwise
 */
bool SaIsDefaultPrivilege(const char *privilege)
{
    char pathName[PATH_MAX];
    char buf[4], *pc;
    FILE *fp;

    /*
     * Bail if someone is messing with us.
     */
    if (strlen(privilege) > NAME_MAX) {
	exit(SaPRIVERROR);
    }

    (void)SaStringFormat(pathName, sizeof pathName, "%s/%s", SaPRIVDEFDIR, privilege);
    fp = fopen(pathName, "r");
    if (fp == NULL) {
	return false;
    }

    if (fgets(buf, sizeof buf, fp) == NULL
	|| ferror(fp)) {
	return false;
    }

    fclose(fp);

    pc = strchr(buf, '\n');
    if (pc) {
	*pc = '\0';
    }

    return strcmp(buf, "on") == 0 ? true : false;
}

/*
 *  static bool PrivilegesAreDefault(const char * const *privs,
 *  				          unsigned int nPrivs)
 *
 *  Description:
 *      Check to see whether all the privileges in an array are
 *      default privileges.  Default privileges are privileges that
 *      all users have.
 *
 *  Parameters:
 *      privs   Privileges to check for defaultness
 *      nPrivs  Number of privileges in "privs".
 *
 *  Returns:
 *	true if all "privs" are default, false if at least one of
 *      them isn't.
 */
static bool PrivilegesAreDefault(const char * const *privs,
				      unsigned int nPrivs)
{
    int i;

    /*
     * If all privileges are default, then the user is privileged.
     */
    for (i = 0; i < nPrivs; i++) {
	if (!SaIsDefaultPrivilege(privs[i])) {
	    return false;
	}
    }

    return true;
}

/*
 *  bool SaIsPrivilegedUser(const char *loginName)
 *
 *  Description:
 *      Check to see if "loginName" is a privileged user.  A
 *      privileged user has all privileges.
 *
 *  Parameters:
 *      loginName  login to check
 *
 *  Returns:
 *	true if "loginName" is a privileged user, false otherwise.
 */
bool SaIsPrivilegedUser(const char *loginName)
{
    char privUser[20], *eol;
    FILE *fp;
    bool isPrivUser = false;

    fp = fopen(SaPRIVUSERDB, "r");
    if (fp == NULL) {
	return false;
    }
	    
    while (fgets(privUser, sizeof privUser, fp) != NULL) {
	eol = strchr(privUser, '\n');
	if (eol) {
	    *eol = '\0';
	}
	if (strcmp(privUser, loginName) == 0) {
	    isPrivUser = true;
	}
    }
    fclose(fp);
    return isPrivUser;
}

/*
 *  bool SaPrivilegesAreEnabled(void)
 *
 *  Returns:
 *      true if privileges are chkconfig'd on, false
 *      otherwise.
 */
bool SaPrivilegesAreEnabled(void)
{
    char buf[20];
    bool returnValue = false;
    FILE *fp = fopen(_SaPrivChkConfig, "r");

    if (fp == NULL) {
	return returnValue;
    }
    
    if (fgets(buf, sizeof buf, fp) != NULL
	&& strncmp(buf, "on", 2) == 0) {
	returnValue = true;
    }

    fclose(fp);
    return returnValue;
}

/*
 *  SaPrivResult SaCheckPriv(uid_t uid, const char * const *privs,
 *                           unsigned int nPrivs)
 *
 *  Description:
 *      Check to see if the user represented by uid should be allowed
 *      to perform the privileged operation represented by "privilege".
 *
 *  Parameters:
 *      uid        User to check
 *      privs      Privileges user wants to perform
 *	nPrivs     The number of privileges in "privs".
 *
 *  Returns:
 *	SaNotPriv          If user does not have this privilege.
 *	SaIsRoot           If user is root.
 *	SaNoRootPassword   If there is no root password on the system.
 *	SaHasPriv          If user has entry in privilege database for
 *	                   this privilege.
 */
SaPrivResult SaCheckPriv(uid_t uid, const char * const *privs,
			 unsigned int nPrivs)
{
    struct passwd *pwd;

    /*
     * Root is automatically privileged
     */
    if (uid == 0) {
	return SaIsRoot;
    }

    /*
     * If there's no root password, all users are automatically
     * privileged.
     */
    pwd = getpwuid(0);
    if (pwd && strcmp(pwd->pw_passwd, "") == 0) {
	return SaNoRootPassword;
    }
#ifndef HAVE__GETPWENT_NO_SHADOW
    /*
     * We have to check the shadow password ourselves.
     */
    if (strcmp(pwd->pw_passwd, "x") == 0) {
        struct spwd *sp = getspnam(pwd->pw_name);
        if (sp == NULL) return SaNotPriv;  /* brain damage */
        if (strcmp(sp->sp_pwdp, "") == 0) {
	    return SaNoRootPassword;
        }
    }
#endif /* HAVE__GETPWENT_NO_SHADOW */

    /*
     * If privileges are chkconfig'd off, we ignore privileges for
     * non-root users when there's a root password.
     */
    if (!SaPrivilegesAreEnabled()) {
	return SaNotPriv;
    }

    /*
     * See if these are all default privileges.
     */
    if (PrivilegesAreDefault(privs, nPrivs)) {
	return SaHasPriv;
    }

    pwd = getpwuid(uid);
    if (pwd == NULL) {
	return SaNotPriv;
    }

    /*
     * Check to see if this is a privileged user.
     */
    if (SaIsPrivilegedUser(pwd->pw_name)) {
	return SaHasPriv;
    }

    /*
     * Otherwise, check the privilege database.
     */
    return SaHasPrivInDB(pwd->pw_name, privs, nPrivs) ? SaHasPriv : SaNotPriv;
}
