/*
 * privent.c
 *
 *	Privilege database access
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.11 $"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <assert.h>

#include <sysadm/i18n.h>
#include <sysadm/privilege.h>

#define MAXLINE 4096

struct _SaPrivFile {
    FILE *fp;
    SaPrivEnt *ent;
};

/*
 * Struct used by SaHasPrivInDB for keeping track of privileges that have
 * been seen.
 */
typedef struct _PrivInfo {
    const char *privilege;
    bool has;
} PrivInfo;

/*
 * This is globally accessible for testing purposes
 */
char *_SaPrivDB = SaPRIVDB;

/*
 *  static void *SafeMalloc(size_t size)
 *
 *  Description:
 *      A version of malloc that can't return NULL
 *
 *  Parameters:
 *      size  amount of memory to allocate
 *
 *  Returns:
 *      allocated memory
 */
static void *SafeMalloc(size_t size)
{
    void *mem = malloc(size);

    if (mem == NULL) {
	syslog(LOG_ERR, i18n("Out of memory"));
	exit(SaPRIVNOMEM);
    }

    return mem;
}

/*
 *  static void *SafeRealloc(void *ptr, size_t size)
 *
 *  Description:
 *      A version of realloc that can't return NULL
 *
 *  Parameters:
 *      ptr   Previously allocated memory
 *      size  new size
 *
 *  Returns:
 *      Reallocated memory
 */
static void *SafeRealloc(void *ptr, size_t size)
{
    void *mem = realloc(ptr, size);

    if (mem == NULL) {
	syslog(LOG_ERR, i18n("Out of memory"));
	exit(SaPRIVNOMEM);
    }

    return mem;
}

/*
 *  static char *SafeStrDup(const char *str)
 *
 *  Description:
 *      A version of strdup that can't return NULL
 *
 *  Parameters:
 *      str  the string to duplicate
 *
 *  Returns:
 *	The duplicated string
 */
static char *SafeStrDup(const char *str)
{
    char *newStr = strdup(str);

    if (newStr == NULL) {
	syslog(LOG_ERR, i18n("Out of memory"));
	exit(SaPRIVNOMEM);
    }

    return newStr;
}

/*
 *  static SaPrivEnt *AllocPrivEnt(const char *privilege)
 *
 *  Description:
 *      Allocate a SaPrivEnt structure, and initialize its fields.
 *
 *  Parameters:
 *      privilege  The privilege string for this structure
 *
 *  Returns:
 *	A pointer to a SaPrivEnt structure
 */
static SaPrivEnt *AllocPrivEnt(const char *privilege)
{
    SaPrivEnt *ent = SafeMalloc(sizeof *ent);
    
    ent->privilege = SafeStrDup(privilege);
    ent->numUsers = 0;
    return ent;
}

/*
 *  static void AddUserToPrivEnt(SaPrivEnt *ent, const char *user)
 *
 *  Description:
 *      Add a user to the array of users in a SaPrivEnt structure.
 *
 *  Parameters:
 *      ent   SaPrivEnt to add a user to 
 *      user  The user id to add
 */
static void AddUserToPrivEnt(SaPrivEnt *ent, const char *user)
{
    ent->users = ent->numUsers ?
	    SafeRealloc(ent->users,
			(size_t)(ent->numUsers + 1) * sizeof *ent->users) :
				SafeMalloc(sizeof *ent->users);
    
    ent->users[ent->numUsers++] = strdup(user);
}

/*
 *  static void FreePrivEnt(SaPrivEnt *ent)
 *
 *  Description:
 *      Free a SaPrivEnt structure
 *
 *  Parameters:
 *      ent  memory to free
 */
static void FreePrivEnt(SaPrivEnt *ent)
{
    int i;

    free(ent->privilege);
    if (ent->numUsers) {
	for (i = 0; i < ent->numUsers; i++) {
	    free((char *)ent->users[i]);
	}
	free(ent->users);
    }
    free(ent);
}

/*
 *  static PrivInfo *CreatePrivInfo(const char * const *privs,
 *  				unsigned int nPrivs)
 *
 *  Description:
 *      Create an array of PrivInfo structs that SaHasPrivInDB can use to
 *      keep track of which privileges a user has.
 *
 *  Parameters:
 *      privs   privileges to make PrivInfo for.
 *      nPrivs  number of privileges passed in.
 *
 *  Returns:
 *	PrivInfo array.
 */
static PrivInfo *CreatePrivInfo(const char * const *privs,
				unsigned int nPrivs)
{
    PrivInfo *privInfo;
    int i;

    privInfo = SafeMalloc(sizeof *privInfo * nPrivs);
    for (i = 0; i < nPrivs; i++) {
	privInfo[i].privilege = privs[i];
	privInfo[i].has = false;
    }

    return privInfo;
}

/*
 *  static void DestroyPrivInfo(PrivInfo *privInfo)
 *
 *  Description:
 *      Destroy an array of PrivInfo created by CreatePrivInfo.
 *
 *  Parameters:
 *      privInfo  The PrivInfo to destroy.
 */
static void DestroyPrivInfo(PrivInfo *privInfo)
{
    free(privInfo);
}

/*
 *  SaPrivFile *SaOpenPrivDB(void)
 *
 *  Description:
 *      Open the privilege database, in preperation for getting
 *      entries out of it.
 *
 *  Returns:
 *	A pointer to a SaPrivFile if successful, NULL if error
 */
SaPrivFile *SaOpenPrivDB(void)
{
    SaPrivFile *privFile;

    privFile = SafeMalloc(sizeof *privFile);
    privFile->fp = fopen(_SaPrivDB, "r");
    if (privFile->fp == NULL) {
	free(privFile);
	return NULL;
    }

    privFile->ent = NULL;
    return privFile;
}

/*
 *  const SaPrivEnt *SaGetPrivEnt(SaPrivFile *privFile)
 *
 *  Description:
 *      Get the next SaPrivEnt structure from privFile.  Memory returned
 *      is owned by this module and should not be assumed to be valid
 *      after subsequent calls to any of these functions using the
 *      same privFile pointer.
 *
 *      This function is meant to be called repeatedly until it
 *      returns NULL to iterate through the contents of the privilege
 *      database.
 *      
 *  Parameters:
 *      privFile  Handle to the privilege database
 *
 *  Returns:
 *	SaPrivEnt pointer if there's another one, NULL if there aren't
 *      any more.
 */
const SaPrivEnt *SaGetPrivEnt(SaPrivFile *privFile)
{
    char buf[MAXLINE];
    char *s, *lasts, *priv, *user;
    
    /*
     * Skip over comments, which are lines beginning with '#'
     */
    while ((s = fgets(buf, sizeof buf - 1, privFile->fp)) != NULL
	   && buf[0] == '#') {
    }

    /*
     * No more entries!
     */
    if (s == NULL) {
	return NULL;
    }

    /*
     * Parse a line from the database into a SaPrivEnt structure.
     */

    /*
     * Separate the line into the "privilege" part (pointed to by buf
     * after the following code) and the "user" part (pointed to by
     * "user" after the following code.
     */
    user = strchr(buf, ':');
    if (user == NULL) {
	return NULL;
    }
    *user++ = '\0';

    lasts = NULL;
    priv = strtok_r(buf, ": \t", &lasts);
    if (priv == NULL) {
	return NULL;
    }

    if (privFile->ent) {
	FreePrivEnt(privFile->ent);
    }

    privFile->ent = AllocPrivEnt(priv);

    lasts = NULL;
    for (user = strtok_r(user, ", \t\n", &lasts);
	 user;
	 user = strtok_r(NULL, ", \t\n", &lasts)) {
	AddUserToPrivEnt(privFile->ent, user);
    }

    return privFile->ent;
}

/*
 *  const SaPrivEnt *SaLookupPrivEnt(SaPrivFile *privFile,
 *                                   const char *privilege)
 *
 *  Description:
 *      Look up "privilege" in the privilege database.
 *
 *  Parameters:
 *      privFile   The privilege database
 *      privilege  The privilege to look up
 *
 *  Returns:
 *	SaPrivEnt for the requested privilege, or NULL if there's no
 *      entry in the database for the requested privilege
 */
const SaPrivEnt *SaLookupPrivEnt(SaPrivFile *privFile, const char *privilege)
{
    const SaPrivEnt *ent;
    long offset;

    offset = ftell(privFile->fp);
    (void)fseek(privFile->fp, 0, SEEK_SET);

    while ((ent = SaGetPrivEnt(privFile)) != NULL) {
	if (strcmp(ent->privilege, privilege) == 0) {
	    break;
	}
    }

    (void)fseek(privFile->fp, offset, SEEK_SET);
    return ent;
}

/*
 *  void SaClosePrivDB(SaPrivFile *privFile)
 *
 *  Description:
 *      Close the privilege database.
 *
 *  Parameters:
 *      privFile  The privilege database to close.
 */
void SaClosePrivDB(SaPrivFile *privFile)
{
    (void)fclose(privFile->fp);
    if (privFile->ent) {
	FreePrivEnt(privFile->ent);
    }
    free(privFile);
}

/*
 *  bool SaHasPrivInDB(const char *user, const char * const *privs,
 *                          unsigned int nPrivs)
 *
 *  Description:
 *      Answer the question: "Does this user have these privileges?"
 *
 *  Parameters:
 *      user       user id of user whose privileges we care about
 *      privilege  the privilege in question
 *
 *  Returns:
 *	true if "user" has "privilege", false otherwise.
 */
bool SaHasPrivInDB(const char *user, const char * const *privs,
			unsigned int nPrivs)
{
    SaPrivFile *privFile;
    const SaPrivEnt *ent;
    unsigned int i, priv, nPrivsHas;
    PrivInfo *privInfo;

    /*
     * You must pass in at least one privilege.
     */
    assert(nPrivs > 0);

    privFile = SaOpenPrivDB();

    if (privFile == NULL) {
	return false;
    }

    privInfo = CreatePrivInfo(privs, nPrivs);
    nPrivsHas = 0;

    /*
     * For each entry in the privilege database, see if that privilege
     * is in the priv list that was passed in.  If so, check all the
     * users in the entry to see if the "user" passed in matches.
     * Mark this privInfo entry and update our counter.
     */
    while ((ent = SaGetPrivEnt(privFile)) != NULL) {
	for (priv = 0; priv < nPrivs; priv++) {
	    if (strcmp(privInfo[priv].privilege, ent->privilege) == 0) {
		for (i = 0; i < ent->numUsers; i++) {
		    if (strcmp(ent->users[i], user) == 0) {
			if (!privInfo[priv].has) {
			    privInfo[priv].has = true;
			    nPrivsHas++;
			}
			break;
		    }
		}
		break;
	    }
	}
    }

    DestroyPrivInfo(privInfo);
    SaClosePrivDB(privFile);

    /*
     * return true iff we found as many privileges in the database as
     * were passed in.
     */
    return nPrivsHas == nPrivs;
}
