/*
 * auth.c
 *
 *	authentication stuff.
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.15 $"

#include <assert.h>
#include <dlfcn.h>
#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>

#include <sysadm/auth.h>
#include <sysadm/authproto.h>
#include <sysadm/format.h>

struct _SaAuth {
    char *authScheme;		/* Name of this authentication scheme */
    void *dso;			/* dso implementing this */
				/* authentication scheme */
    AUTH *dsoAuth;		/* dso's auth stuff */

    /*
     * Functions exported by authentication dsos.
     */
    AUTH *(*open)(const char *authScheme, AuthType authType);
    struct _WidgetRec *(*createWidgets)(AUTH *auth, struct _WidgetRec *parent);
    void (*initWidgets)(AUTH *auth);
    void (*sendParams)(AUTH *auth, int infd, int outfd);
    bool (*checkParams)(AUTH *auth, int infd, int outfd);
    void (*close)(AUTH *auth);
};

/*
 *  SaAuth *SaAuthOpen(const char *authScheme, SaAuthType authType)
 *
 *  Description:
 *      Call dlopen to open the dso corresponding to authScheme, and
 *      get the symbols we need out of it.  Call the dso's open
 *      routine to initialize it.
 *
 *  Parameters:
 *      authScheme  Name of authentication scheme.
 *      authType    Whether we're the sender of receiver of
 *                  authentication parameters.
 *
 *  Returns:
 *	SaAuth pointer which can be used for authentication
 *      operations.
 */

#define GETSYM(field, string) { auth->field = dlsym(auth->dso, string); \
					if (!auth->field) { \
					    (void)fprintf(stderr, \
							  "%s is missing " \
							  "function %s\n", \
							  authdso, \
							  string); \
					    dlclose(auth->dso); \
					    free(auth); \
					    return NULL; }}

SaAuth *SaAuthOpen(const char *authScheme, SaAuthType authType)
{
    char authdso[PATH_MAX];
    SaAuth *auth;

    /*
     * This protects against "runpriv -auth ../../../tmp/hack/hack".
     */
    if (strchr(authScheme, '/') != NULL) {
	exit(2);		/* SaPRIVERROR */ 
    }

    auth = malloc(sizeof *auth);
    if (auth == NULL) {
	exit(1);		/* SaPRIVNOMEM */ 
    }

    if (authType == SaAuthSender) {
	(void)SaStringFormat(authdso, sizeof authdso,
		       "%s/%sSender.so", SaPRIVAUTHDIR, authScheme);
    } else {
	(void)SaStringFormat(authdso, sizeof authdso,
		       "%s/%sReceiver.so", SaPRIVAUTHDIR, authScheme);
    }

    auth->dso = dlopen(authdso, RTLD_LAZY);
    if (auth->dso == NULL) {
	(void)fprintf(stderr, "%s: %s\n", authdso, dlerror());
	free(auth);
	return NULL;
    }

    GETSYM(open, "Open");
    GETSYM(close, "Close");

    if (authType == SaAuthSender) {
	GETSYM(createWidgets, "CreateWidgets");
	GETSYM(initWidgets, "InitWidgets");
	GETSYM(sendParams, "SendParams");
	auth->checkParams = NULL;
    } else {
	auth->createWidgets = NULL;
	auth->initWidgets = NULL;
	auth->sendParams = NULL;
	GETSYM(checkParams, "CheckParams");
    }

    auth->authScheme = strdup(authScheme);
    auth->dsoAuth = auth->open(authScheme, (AuthType)authType);

    return auth;
}

/*
 *  void SaAuthClose(SaAuth *auth)
 *
 *  Description:
 *      Free stuff associated with an SaAuth pointer.  Call the dso's
 *      close routine, dlclose the dso, and free memory that we
 *      allocated in SaAuthOpen.
 *
 *  Parameters:
 *      auth  authentication stuff.
 */
void SaAuthClose(SaAuth *auth)
{
    auth->close(auth->dsoAuth);
    dlclose(auth->dso);
    free(auth->authScheme);
    free(auth);
}

/*
 *  const char *SaAuthGetName(SaAuth *auth)
 *
 *  Description:
 *      Get the name of the authentication scheme corresponding to
 *      "auth".
 *
 *  Parameters:
 *      auth  authenticaton stuff.
 *
 *  Returns:
 *	Name of the authentication scheme; this is the same string
 *      (meaning strcmp compares the same) that was passed to
 *      SaAuthOpen.
 */
const char *SaAuthGetName(SaAuth *auth)
{
    return auth->authScheme;
}

/*
 * The rest of these functions just call their dso equivalents in the
 * authentication dso.
 */
struct _WidgetRec *SaAuthCreateWidgets(SaAuth *auth, struct _WidgetRec *parent)
{
    /*
     * Make sure we're a sender.
     */
    assert(auth->createWidgets != NULL);
    return auth->createWidgets(auth->dsoAuth, parent);
}

void SaAuthInitWidgets(SaAuth *auth)
{
    /*
     * Make sure we're a sender.
     */
    assert(auth->initWidgets != NULL);
    auth->initWidgets(auth->dsoAuth);
}

void SaAuthSendParams(SaAuth *auth, int infd, int outfd)
{
    struct sigaction act, oact;

    /*
     * Make sure we're a sender.
     */
    assert(auth->sendParams != NULL);

    /*
     * Ignore SIGPIPE while sending auth params.  If something goes
     * wrong, we'll catch the error downstream.
     */
    act.sa_flags = 0;
    act.sa_handler = SIG_IGN;
    (void)sigemptyset(&act.sa_mask);
    (void)sigaction(SIGPIPE, &act, &oact);

    auth->sendParams(auth->dsoAuth, infd, outfd);

    (void)sigaction(SIGPIPE, &oact, NULL);
}

bool SaAuthCheckParams(SaAuth *auth, int infd, int outfd)
{
    /*
     * Make sure we're a receiver.
     */
    assert(auth->checkParams != NULL);
    return auth->checkParams(auth->dsoAuth, infd, outfd);
}
