/*
 * unix.c
 *
 *	UNIX authentication.  Read a line from standard input and see
 *	if it matches the root password.
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.9 $"

#include <sys/types.h>
#include <pwd.h>
#include <crypt.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include <sysadm/authproto.h>
#include <sysadm/i18n.h>
#ifndef HAVE__GETPWENT_NO_SHADOW
#include <shadow.h>
#endif /* !HAVE__GETPWENT_NO_SHADOW */

/*
 *  AUTH *Open(const char *authScheme, AuthType authType)
 *
 *  Description:
 *      Initialize the unix authentication dso.
 *
 *  Parameters:
 *      authScheme  Authentication scheme -- "unix".
 *      authType    AUTH_SENDER or AUTH_RECEIVER.
 *
 *  Returns:
 *	A pointer to our auth stuff.
 */
/*ARGSUSED*/
AUTH *Open(const char *authScheme, AuthType authType)
{
    return NULL;
}

/*
 *  static int getline(int input, char *buf, size_t size)
 *
 *  Description:
 *      Get a line from input.  Avoid use of stdio because we
 *      only want to read from the first line; if we buffered any
 *      chars from subsequent lines software down the road could end
 *      up getting hosed.
 *
 *  Parameters:
 *      buf   buffer to read a line into
 *      size  size of buffer
 *
 *  Returns:
 *	0 if successful, -1 if error
 */
static int getline(int input, char *buf, size_t size)
{
    char *pc;
    int n;

    /*
     * If runpriv is being run from the command line, use getpass(3)
     * so that a prompt is entered and the password does not appear on
     * the terminal.
     */
    if (isatty(input)) {
	pc = getpass(i18n("Password:"));
	if (!pc || strlen(pc) > size) {
	    return -1;
	}
	(void)strcpy(buf, pc);
	return 0;
    }

    /*
     * Read from standard input one character at a time.
     */
    pc = buf;
    while (pc - buf < size && (n = read(input, pc, 1)) == 1
	   && *pc != '\n') {
	pc++;
    }

    /*
     * Make sure the loop exited for the right reason
     */
    if (pc - buf == size || n == -1) {
	return -1;
    }

    /*
     * Replace carriage return with a NULL.
     */
    *pc = '\0';
    return 0;
}

/*
 *  bool CheckParams(AUTH *auth, int infd, int outfd)
 *
 *  Description:
 *      Authenticate user for privileged operations.  This scheme
 *      equates authentication with capability of providing the root
 *      password.
 *
 *  Parameters:
 *      auth      our authentication stuff.
 *      infd      Input from SendParams or terminal.
 *      outfd     Output to SendParams.  Not used for unix authentication.
 *
 *  Returns:
 *	true if user gets the password right, false otherwise.
 */
/*ARGSUSED*/
bool CheckParams(AUTH *auth, int infd, int outfd)
{
    char buf[256];
    char pwbuf[256];
    struct passwd *pwd;
    char *encrypted;
    
    pwd = getpwuid(0);

    if (!pwd) {
	return false;
    }

    if (getline(infd, buf, sizeof buf) == -1) {
	return false;
    }

#ifdef HAVE__GETPWENT_NO_SHADOW
    strncpy(pwbuf, pwd->pw_passwd, sizeof(pwbuf));
#else
    /*  Check for shadow password */
    if (strcmp(pwd->pw_passwd, "x") == 0) {
        struct spwd *sp = getspnam(pwd->pw_name);
        if(sp == NULL) return false;
        strncpy(pwbuf, sp->sp_pwdp, sizeof(pwbuf));
    }
#endif /* HAVE__GETPWENT_NO_SHADOW */

    encrypted = crypt(buf, pwbuf);

    /*
     * Wipe out plain-text password.
     */
    memset(buf, 0, sizeof buf);

    return strcmp(encrypted, pwbuf) == 0;
}

/*
 *  void Close(AUTH *auth)
 *
 *  Description:
 *      Free authentication stuff that we've allocated.  This gets
 *      called in preperation for a call to dlclose.
 *
 *  Parameters:
 *      auth  Our authentication stuff.
 */
void Close(AUTH *auth)
{
}
