/*
 * logopen.c
 *
 *	Implement opening and closing of the system admin log.
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.11 $"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/param.h>

#include <sysadm/salogI.h>

/*
 * This is externally visible for testing purposes.
 */
const char *_SaLogFile = SaLOGFILE;
const char *_SaLogFileOld = SaLOGFILEOLD;
const char *_SaLogConfig = SaLOGCONFIG;
const char *_SaLogRotLock = SaLOGROTLOCK;

/*
 * Globally accessible copy of the last SaLog pointer returned by
 * SaLogOpen.  Useful for using SaLogWriteMsg from library routines.
 */
SaLog *theSaLog = NULL;

/*
 *  static unsigned long GetRotSize(void)
 *
 *  Description:
 *      Get the rotation size of the salog.  This is the size that
 *      when exceeded causes us to rotate the log.  We get the number
 *      from our config file if it's in there, and use SaLOGROTSIZE if
 *      it's not.
 *
 *  Returns:
 *	rotation size of the system admin log.
 */
static unsigned long GetRotSize(void)
{
    FILE *fp;
    unsigned long rotSize = SaLOGROTSIZE;
    char buf[256];

    fp = fopen(_SaLogConfig, "r");
    if (fp != NULL) {
	while (fgets(buf, sizeof buf, fp) != NULL) {
	    if (strncmp(buf, "rotsize=", 8) == 0) {
		rotSize = atoi(buf + 8);
		if (rotSize < SaLOGMINROTSIZE) {
		    rotSize = SaLOGMINROTSIZE;
		}
		break;
	    }
	}
	fclose(fp);
    }
		
    return rotSize;
}

/*
 *  static void LockAndOpen(int *lockfd, int *logfd)
 *
 *  Description:
 *      Open the lock file.  Rotate the log if it's too big.  Open the
 *      log file.
 *      
 *      We only rotate the log if we can get a write lock on the log
 *      rotation lock file (_SaLogRotLock).  After we've (perhaps)
 *      rotated the log, we get a read lock on _SaLogRotLock.  We hold
 *      the read lock until SaLogClose is called or until we exit.
 *      Thus, the log can only be rotated if there are no other
 *      processes that have it open.  This simplifies SaLogWrite
 *      because it doesn't have to reopen the log file, but has the
 *      drawback that under pathological conditions (a hung priv
 *      command), the log will never get rotated.
 *
 *  Parameters:
 *      lockfd  Gets file descriptor for the lock.
 *      logfd   Gets file descriptor for the log.
 */
static void LockAndOpen(int *lockfd, int *logfd)
{
    int oflags, statRes;
    struct stat sb;
    unsigned long rotSize;
    struct flock lockBuf;
    mode_t logPerm = 0644 | S_ISGID;

    *logfd = -1;
    *lockfd = open(_SaLogRotLock, O_RDWR | O_CREAT, 0600);
    statRes = stat(_SaLogFile, &sb);
    oflags = O_RDWR;

    if (*lockfd != -1 && statRes != -1) {

	lockBuf.l_whence = SEEK_SET;
	lockBuf.l_start = 0;
	lockBuf.l_len = 0;

	rotSize = GetRotSize();
	if (sb.st_size > rotSize) {
	    lockBuf.l_type = F_WRLCK;
	    if (fcntl(*lockfd, F_SETLK, &lockBuf) != -1) {
		(void)unlink(_SaLogFileOld);
		(void)rename(_SaLogFile, _SaLogFileOld);
		lockBuf.l_type = F_UNLCK;
		(void)fcntl(*lockfd, F_SETLK, &lockBuf);

		/*
		 * The new, empty log should have the same permissions
		 * as the old log, in case the administrator doesn't
		 * want anyone to see what's in the log.  S_ISGID is
		 * to set mandatory locking; we turn the bit on in
		 * case the admin has run "chmod 600 /var/sysadm/salog".
		 */
		oflags |= O_CREAT;
		logPerm = sb.st_mode | S_ISGID;
	    }
	}
    }

    if (*lockfd != -1) {
	lockBuf.l_type = F_RDLCK;
	(void)fcntl(*lockfd, F_SETLKW, &lockBuf);
    }

    if (statRes == -1) {
	oflags |= O_CREAT;
    }

    *logfd = open(_SaLogFile, oflags, logPerm);
}

/*
 *  SaLogErrorCode SaLogOpen(const char *command, SaLog **logReturn)
 *
 *  Description:
 *      Open the system admin log in preparation for writing.
 *
 *  Parameters:
 *	command    The name of the command calling us.
 *      logReturn  Gets the log handle.
 *
 *  Returns:
 *	SaLogErrNoError if log successfully opened, an error code
 *      otherwise.
 */
SaLogErrorCode SaLogOpen(const char *command, SaLog **logReturn)
{
    SaLog *log;
    int logfd, lockfd;
    FILE *fp;
    char hostName[MAXHOSTNAMELEN + 1];

    LockAndOpen(&lockfd, &logfd);

    /*
     * Use open/fdopen instead of fopen because we want to set
     * mandatory locking if we're creating the file.
     */
    if (logfd == -1 || (fp = fdopen(logfd, "r+")) == NULL) {
	if (logfd != -1) {
	    (void)close(logfd);
	}
	if (lockfd != -1) {
	    (void)close(lockfd);
	}
	return SaLogErrSys;
    }
    
    log = malloc(sizeof(*log));
    if (log == NULL) {
	(void)fclose(fp);
	if (lockfd != -1) {
	    (void)close(lockfd);
	}
	return SaLogErrNoMem;
    }

    log->fp = fp;
    log->errorCode = SaLogErrNoError;
    log->errorString = "";
    log->pid = getpid();
    log->uid = getuid();

    if (gethostname(hostName, MAXHOSTNAMELEN) == 0) {
	hostName[MAXHOSTNAMELEN] = '\0';
	log->host = strdup(hostName);
    } else {
	log->host = strdup("");
    }

    log->command = strdup(command);
    log->logToSysLog = false;
    log->perrorFlag = true;
    log->rotLockFd = lockfd;

    *logReturn = log;
    theSaLog = log;

    return SaLogErrNoError;
}

/*
 *  void SaLogClose(SaLog *log)
 *
 *  Description:
 *      Free resources allocated by SaLogOpen.
 *
 *  Parameters:
 *      log  handle of log to close.
 */
void SaLogClose(SaLog *log)
{
    if (log == theSaLog) {
	theSaLog = NULL;
    }

    if (log->rotLockFd != -1) {
	(void)close(log->rotLockFd);
    }

    (void)fclose(log->fp);
    free(log->host);
    free(log->command);
    free(log);
}

/*
 * Set the process id that will get logged.
 */
void SaLogSetPid(SaLog *log, pid_t pid)
{
    log->pid = pid;
}

/*
 * Set the user id that will get logged.
 */
void SaLogSetUid(SaLog *log, uid_t uid)
{
    log->uid = uid;
}

/*
 * Set the host that will get logged.
 */
void SaLogSetHost(SaLog *log, const char *host)
{
    free(log->host);
    log->host = strdup(host);
}

/*
 * Set the command that log messages will be reported as being
 * associated with.
 */
void SaLogSetCommand(SaLog *log, const char *command)
{
    free(log->command);
    log->command = strdup(command);
}

/*
 * Set the flag that tells us whether or not to copy log messages to
 * syslog.
 */
void SaLogSetSysLogFlag(SaLog *log, bool logToSysLog)
{
    log->logToSysLog = logToSysLog;
}

/*
 * Set the flag that tells us whether or not to copy error messages to
 * stderr.
 */
void SaLogSetPerrorFlag(SaLog *log, bool perrorFlag)
{
    log->perrorFlag = perrorFlag;
}

