/*
 * logwrite.c
 *
 *	Writing to the system amdin log.
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.13 $"

#include <stdio.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <assert.h>
#include <stdlib.h>
#include <sysadm/salogI.h>
#include <sysadm/i18n.h>
#include <sysadm/format.h>


/*
 * This is externally visible for testing purposes.
 */
bool _SaTestLock = false;

/*
 * Obtain a lock in preparation for writing to the log file.
 */
static int LockLog(SaLog *log)
{
    int status;
    long offset;

    /*
     * Mandatory lock on the log file itself, to keep readers out
     * while we're updating it.
     */
    offset = ftell(log->fp);
    (void)fseek(log->fp, 0, SEEK_SET);
    status = lockf(fileno(log->fp), F_LOCK, 0);
    (void)fseek(log->fp, offset, SEEK_SET);

    /*
     * This is here for testing purposes.  We grab the lock and go to
     * sleep until we're signalled, so tester can verify that the
     * locking actually works.
     */
    if (_SaTestLock && status == 0) {
	pause();
	sginap(0);
    }

    return status;
}

/*
 * Unlock the log file.
 */
static void UnLockLog(SaLog *log)
{
    long offset;

    offset = ftell(log->fp);
    (void)fseek(log->fp, 0, SEEK_SET);
    (void)lockf(fileno(log->fp), F_ULOCK, 0);
    (void)fseek(log->fp, offset, SEEK_SET);
}

/*
 * Determine whether or not we should copy log messages to syslog.
 * The answer is yes if log->logToSysLog is true or if our config
 * file says we should do it.
 */
static bool ShouldWriteToSysLog(SaLog *log)
{
    char buf[256];
    FILE *fp;
    bool shouldWrite = false;

    if (log->logToSysLog) {
	return true;
    }

    if ((fp = fopen(_SaLogConfig, "r")) == NULL) {
	return false;
    }

    while (fgets(buf, sizeof buf, fp) && !shouldWrite) {
	if (strncmp(buf, "syslog", 6) == 0) {
	    shouldWrite = true;
	}
    }

    (void)fclose(fp);
    return shouldWrite;
}

/*
 *  static void WriteToSysLog(SaLog *log, SaLogMsgType type,
 *  			      const char *timeString,
 *  			      const char *formatString, va_list ap)
 *
 *  Description:
 *      Write an salog message to syslog. This code is based on the
 *      libc code for vsyslog.  We can't use vsyslog because we'd have
 *      to call openlog and we don't want to muck with our caller's
 *      use of syslog.  Also, our formatString could be a lfmt string.
 *      So why not use lfmt?  It's not flexible enough in the control
 *      it gives over the priority.  Sigh.
 *
 *  Parameters:
 *      log           SaLog handle.
 *      type          Type of message.
 *      timeString    String to use for time.
 *      formatString  lfmt-style format string.
 *      ap            Args for formatString.
 */
static void WriteToSysLog(SaLog *log, SaLogMsgType type,
			  const char *timeString,
			  const char *formatString, va_list ap)
{
    char tbuf[SaLOGBUFSIZE], *pc;
    int fd, pri;

    /*
     * Tricky opening code taken from libc openlog.
     */
    fd = open("/dev/log", O_WRONLY | O_NDELAY);
    if (fd == -1) {
	return;
    }
    (void)fcntl(fd, F_SETFL, 0);

    /*
     * Figure out which priority to use.
     */
    switch (type) {
      case SaLogError:
	pri = LOG_ERR;
	break;
      case SaLogWarning:
	pri = LOG_WARNING;
	break;
      default:
	pri = LOG_INFO;
	break;
    }
    
    /*
     * Print a message in the same format that libc syslog does it.
     */
    (void)SaStringFormat(tbuf, sizeof tbuf,
		   "<%d>%.15s ", pri | LOG_USER, timeString + 4);
    (void)strncat(tbuf, log->command, sizeof tbuf - strlen(tbuf));
    pc = tbuf + strlen(tbuf);
    (void)SaStringFormat(pc, sizeof tbuf - (pc - tbuf), "[%d]", log->pid);
    if (*log->command) {
	(void)strncat(tbuf, ": ", sizeof tbuf - strlen(tbuf));
    }
    
    pc = tbuf + strlen(tbuf);
    (void)SaStringFormatV(pc, sizeof tbuf - (pc - tbuf), formatString, ap);

    /*
     * Make sure we didn't overrun our buffer.
     */
    assert(strlen(tbuf) < sizeof tbuf);

    (void)write(fd, tbuf, strlen(tbuf));
    (void)close(fd);
}

/*
 *  SaLogStatus SaLogWriteMsg(SaLog *log, SaLogMsgType type,
 *  			      const char *formatString, ...)
 *
 *  Description:
 *      Write a message to the sysadm log file, and possibly to syslog
 *      as well.
 *
 *  Parameters:
 *      log            log handle.
 *      type           type of message being logged.
 *      formatString   printf-style format string.
 *      ...            args for formatString.
 *
 *  Returns:
 *	Success or failure.
 */
SaLogStatus SaLogWriteMsg(SaLog *log, SaLogMsgType type,
			  const char *formatString, ...)
{
    va_list ap;
    va_start(ap, formatString);
    return SaLogWriteMsgv(log, type, formatString, ap);
}

SaLogStatus SaLogWriteMsgv(SaLog *log, SaLogMsgType type,
			   const char *formatString, va_list args)
{
    time_t now;
    char timeBuf[100], *pc;
    struct tm t;

    if (LockLog(log) == -1) {
	log->errorCode = SaLogErrCantLock;
	return SaLogStatusError;
    }

    (void)time(&now);
    localtime_r(&now, &t);
    /*  Format the time the message was written */
    strftime(timeBuf, sizeof timeBuf, i18n("%b %e %H:%M:%S"), &t);
    pc = strchr(timeBuf, '\n');
    if (pc) {
	*pc = '\0';
    }
    
    (void)fseek(log->fp, 0, SEEK_END);
    (void)fprintf(log->fp, "%s;%d;%d;%d;%s;%s;", timeBuf, type,
		  log->pid, log->uid, log->host, log->command);
    vfprintf(log->fp, formatString, args);

    /*
     * Copy error messages to stderr.
     */
    if (type == SaLogError && log->perrorFlag) {
	vfprintf(stderr, formatString, args);
    }

    if (strchr(formatString, '\n') == NULL) {
	(void)fputc('\n', log->fp);
	if (type == SaLogError && log->perrorFlag) {
	    (void)fputc('\n', stderr);
	}
    }
    (void)fflush(log->fp);

    UnLockLog(log);

    if (ShouldWriteToSysLog(log)) {
	/*
	 * Use ctime for syslog.
	 */
	ctime_r(&now, timeBuf);
	pc = strchr(timeBuf, '\n');
	if (pc) {
	    *pc = '\0';
	}
	WriteToSysLog(log, type, timeBuf, formatString, args);
    }

    if (ferror(log->fp)) {
	log->errorCode = SaLogErrSys;
	log->errorString = strerror(errno);
	return SaLogStatusError;
    }
    return SaLogStatusOK;
}
