/*
 * salog.h
 *
 *	System Administration logging.
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.8 $"

#ifndef _SALOG_H_
#define _SALOG_H_
#ifdef __cplusplus
extern "C" {
#endif

#include <sysadm/sysadm.h>
#include <sys/types.h>
#include <stdarg.h>
#include <syslog.h>

/*
 * Opaque handle for logging API.
 */
typedef struct _SaLog SaLog;

/*
 * Return status of functions
 */
typedef enum { SaLogStatusError = -1, SaLogStatusOK = 0 } SaLogStatus;

/*
 * Error codes
 */
typedef enum {
    SaLogErrNoError,		/* No error */
    SaLogErrSys,		/* System call failed */
    SaLogErrNoMem,		/* Out of memory */
    SaLogErrCantLock		/* Unable to lock system admin log */
} SaLogErrorCode;

/*
 * Type of a log message.
 *
 *   SaLogError       Error message.
 *   SaLogWarning     Warning message.
 *   SaLogCommand     shell commands which were run.  These are
 *                    normally only logged by runpriv(1M).
 *   SaLogMessage     Human-readable message describing what was done.
 */
typedef enum {
    SaLogError,
    SaLogWarning,
    SaLogCommand,
    SaLogMessage
} SaLogMsgType;

/*
 * Open the System Administration log for writing.  "command" should
 * be the name of the command that is calling the log functions.
 */
SaLogErrorCode SaLogOpen(const char *command, SaLog **logReturn);

/*
 * Get the error code of the last SaLog function that returned
 * SaLogStatusError.
 */
SaLogErrorCode SaLogGetErrorCode(SaLog *log);

/*
 * Translate an error code to a string.
 */
const char *SaLogErrorCodeToString(SaLogErrorCode errorCode);

/*
 * Get an error string describing the failure of the last SaLog
 * function that returned SaLogStatusError.  This is equivalent to
 * calling:
 * 
 *   SaLogErrorCodeToString(log, SaLogGetErrorCode(log));
 */
const char *SaLogGetErrorString(SaLog *log);

/*
 * Set the process id that gets logged by SaLogWriteMsg.
 */
void SaLogSetPid(SaLog *log, pid_t pid);

/*
 * Set the process user id that gets logged by SaLogWriteMsg.
 */
void SaLogSetUid(SaLog *log, uid_t uid);

/*
 * Set the host name that gets logged by SaLogWriteMsg.
 */
void SaLogSetHost(SaLog *log, const char *host);

/*
 * Set the command name that gets logged by SaLogWriteMsg.
 */
void SaLogSetCommand(SaLog *log, const char *command);

/*
 * If "logToSysLog" is B_TRUE, log messages will be copied to syslog.
 * Setting "logToSysLog" to B_FALSE will prevent log messages from
 * being copied to syslog unless the "syslog" flag is set in
 * /var/sysadm/salog.conf.  Default value is B_FALSE.
 */
void SaLogSetSysLogFlag(SaLog *log, bool logToSysLog);

/*
 * if "perrorFlag", messages of type "SaLogError" will get copied to
 * stderr.  Default value is B_TRUE.
 */
void SaLogSetPerrorFlag(SaLog *log, bool perrorFlag);

/*
 * Write a message to the log.  formatString and varargs as in printf.
 */
SaLogStatus SaLogWriteMsg(SaLog *log, SaLogMsgType type,
			  const char *formatString, ...);

SaLogStatus SaLogWriteMsgv(SaLog *log, SaLogMsgType type,
			   const char *formatString, va_list args);

/*
 * Close the log.
 */
void SaLogClose(SaLog *log);

/*
 * Globally accessible copy of the last SaLog pointer returned by
 * SaLogOpen.  Useful for using SaLogWriteMsg from library routines.
 */
extern SaLog *theSaLog;

#ifdef __cplusplus
}
#endif
#endif /* _SALOG_H_ */
