/*
 * testlog.c
 *
 *	test libsalog.
 *
 *
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.10 $"

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <sysadm/salog.h>
#include <sysadm/salogI.h>

/*
 * Write a log message, check return code, print a message and exit if
 * there was an error.
 */
void WriteLogMsg(SaLog *log, SaLogMsgType type,
		 const char *formatString, ...)
{
    va_list ap;
    va_start(ap, formatString);
    if (SaLogWriteMsgv(log, type, formatString, ap) != SaLogStatusOK) {
	(void)fprintf(stderr, "%s: %s\n", "testlog",
		      SaLogGetErrorString(log));
	exit(1);
    }
}

/*
 * Test the process id setting feature.
 */
void TestPid(SaLog *log)
{
    SaLogSetPid(log, 1);
    WriteLogMsg(log, SaLogMessage, "Pid test: pid should be 1");
    SaLogSetPid(log, getpid());
}

/*
 * Test the user id setting feature.
 */
void TestUid(SaLog *log)
{
    uid_t uid = getuid();
    SaLogSetUid(log, uid + 1);
    WriteLogMsg(log, SaLogMessage, "Uid test: uid should be %d", uid + 1);
    SaLogSetUid(log, uid);
}

/*
 * Test the host name setting feature.
 */
void TestHost(SaLog *log)
{
    char hostName[MAXHOSTNAMELEN + 1];
    SaLogSetHost(log, "bogushost");
    WriteLogMsg(log, SaLogMessage,
		"Host test: host should be bogushost");
    (void)gethostname(hostName, MAXHOSTNAMELEN);
    SaLogSetHost(log, hostName);
}

/*
 * Test the command name setting feature.
 */
void TestCommand(SaLog *log)
{
    SaLogSetCommand(log, "blah");
    WriteLogMsg(log, SaLogMessage,
		"Command test: command should be blah");
    SaLogSetCommand(log, "testlog");
}

/*
 * Test the copy-to-syslog feature.
 */
void TestSysLog(SaLog *log)
{
    SaLogSetSysLogFlag(log, true);
    WriteLogMsg(log, SaLogMessage,
		"Syslog test: This message should show up in syslog.");
    WriteLogMsg(log, SaLogWarning,
		"Syslog test: This warning should show up in syslog.");
    WriteLogMsg(log, SaLogError,
		"Syslog test: This error should show up in syslog.");
    WriteLogMsg(log, SaLogCommand,
		"Syslog test: This command should show up in syslog.");
    SaLogSetSysLogFlag(log, false);
}

/*
 * Test the i18n support.
 */
void TestI18N(SaLog *log)
{
/*    SaLogSetI18NFlag(log, true);*/
    WriteLogMsg(log, SaLogMessage,
		"bogus:32:i18n test: This should not have two colons"
		" at the beginning");
/*    SaLogSetI18NFlag(log, false);*/
}

/*
 * Make sure printf format strings are working properly.
 */
void TestFormatting(SaLog *log)
{
    WriteLogMsg(log, SaLogMessage,
		"Format test: This should have %d %%, %s", 1, "right?");
    
/*    SaLogSetI18NFlag(log, true);*/
    WriteLogMsg(log, SaLogMessage,
		"bogus:11:i18n Format test: This should have some %c's", ':');
/*    SaLogSetI18NFlag(log, false);*/
}

/*
 * Make sure the "type" argument is processed properly.
 */
void TestType(SaLog *log)
{
    WriteLogMsg(log, SaLogCommand, "Type test: this is a command");
    WriteLogMsg(log, SaLogMessage, "Type test: this is a message");
    WriteLogMsg(log, SaLogWarning, "Type test: this is a warning");
    WriteLogMsg(log, SaLogError, "Type test: this is an error");
}

/*
 * Handle the alarm signal.  When the child process below calls
 * SaLogWriteMsg after setting _SaTestLog to true, SaLogWriteMsg
 * will end up calling pause().  The parent process sends a SIGALRM to
 * wake the child up.
 */
/*ARGSUSED*/
void AlarmHandler(int arg)
{
}

/*
 * Child process for lock testing.  This child process sets _SaTestLock
 * to true before calling SaLogWriteMsg, so that it will go to sleep
 * in LockLog.  the parent will then test to make sure the locking
 * worked.
 */
void child(void)
{
    struct sigaction act;
    SaLog *log;
    SaLogErrorCode code;

    act.sa_flags = 0;
    act.sa_handler = AlarmHandler;
    (void)sigemptyset(&act.sa_mask);
    (void)sigaction(SIGALRM, &act, NULL);
    if ((code = SaLogOpen("testlog", &log)) != SaLogErrNoError) {
	(void)fprintf(stderr, "Error: %s\n",
		      SaLogErrorCodeToString(code));
	exit(1);
    }

/*    SaLogSetI18NFlag(log, false);*/
    
    _SaTestLock = true;
    if (SaLogWriteMsg(log, SaLogMessage, "Lock test") != SaLogStatusOK) {
	(void)fprintf(stderr, "Error: %s\n", SaLogGetErrorString(log));
	exit(1);
    }
    SaLogClose(log);
    exit(0);
}

/*
 * End the child process.
 */
void endchild(pid_t pid)
{
    int status;
    (void)kill(pid, SIGALRM);
    (void)waitpid(pid, &status, 0);
}

/*
 * Make sure locking the log works.  We fork, and the child process
 * gets the lock and then goes to sleep.  The parent makes sure it
 * can't get the lock while the child still has it.
 */
void TestLocking(SaLog *log)
{
    pid_t pid;
    int fd, status;
    char buf[10];

    if ((pid = fork()) == -1) {
	perror("Can't fork");
	exit(1);
    }

    if (pid == 0) {
	child();
    }

    if ((fd = open(_SaLogFile, O_RDWR | O_NONBLOCK)) == -1) {
	perror(_SaLogFile);
	endchild(pid);
	exit(1);
    }

    /*
     * Yield to child, to make sure it gets the lock before we call read.
     */
    sginap(5);

    if (read(fd, buf, sizeof buf) != -1 || errno != EAGAIN) {
	(void)fprintf(stderr, "Error: I can read with the log locked\n");
	endchild(pid);
	exit(1);
    }

    if (lockf(fd, F_TLOCK, 0) != -1 || errno != EACCES) {
	(void)fprintf(stderr, "Error: I can lock the file twice\n");
	endchild(pid);
	exit(1);
    }

    (void)kill(pid, SIGALRM);
    if (SaLogWriteMsg(log, SaLogMessage,
		      "This should come after \"Lock test\"") !=
	SaLogStatusOK) {
	(void)fprintf(stderr, "Error: %s\n", SaLogGetErrorString(log));
	exit(1);
    }	
	    
    if (waitpid(pid, &status, 0) != pid
	|| !WIFEXITED(status)
	|| WEXITSTATUS(status) != 0) {
	(void)fprintf(stderr, "Error: child process exited non-zero\n");
	exit(1);
    }
}

void main(void)
{
    SaLog *log;
    SaLogErrorCode code;

    _SaLogFile = "salog";
    _SaLogFileOld = "oldsalog";
    _SaLogConfig = "salog.conf";
    _SaLogRotLock = ".salogrot.lock";

    code = SaLogOpen("testlog", &log);
    if (code != SaLogErrNoError) {
	(void)fprintf(stderr, "Error: %s\n",
		      SaLogErrorCodeToString(code));
	exit(1);
    }
		      
    if (SaLogGetErrorCode(log) != SaLogErrNoError) {
	(void)fprintf(stderr, "Error: log has wrong error code\n");
	exit(1);
    }

    if (log != theSaLog) {
	(void)fprintf(stderr, "Error: theSaLog did not get set\n");
	exit(1);
    }

    SaLogSetPerrorFlag(log, false);
/*    SaLogSetI18NFlag(log, false);*/
    TestPid(log);
    TestUid(log);
    TestHost(log);
    TestCommand(log);
    TestSysLog(log);
    TestI18N(log);
    TestFormatting(log);
    TestType(log);
    TestLocking(log);

    SaLogClose(log);
    if (theSaLog != NULL) {
	(void)fprintf(stderr, "Error: theSaLog did not get unset\n");
    }
    exit(0);
}
