//
// AssocFactory.c++
//
//	Factory class for Association objects.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.9 $"

#include <stdio.h>
#include <limits.h>
#include <dlfcn.h>
#include <assert.h>

#include <sysadm/Log.h>
#include <sysadm/CategoryFactory.h>
#include <sysadm/AssocFactory.h>
#include <sysadm/format.h>
#include <sysadm/i18n.h>

BEGIN_NAMESPACE(sysadm);

const char* AssocFactory::NAME = "AssocFactory";

DictionaryOf<struct AssocFactory::DictElem> 
    AssocFactory::_assocs;


//
//  static String computeFuncName(const String& catName)
//
//  Description:
//      Computes function name that will return a new instance 
//	of an Association.
//
//  Parameters:
//      catName The name of the Category requested.
//
//  Returns:
//	Function name for creating Category.
//
static String computeFuncName(const String& catName) {
    int len = 3 + catName.getLength() + 1;
    char funcName[len];
    SaStringFormat(funcName, len, "get%s", (const char*)catName);
    return funcName;
}


//
//  static String computeDSOName(const String& assocName) 
//
//  Description:
//      Computes DSO name that contains symbol for creation of
//	a new instance of Association.
//
//  Parameters:
//      assocName The name of the Association requested.
//
//  Returns:
//	DSO name with symbol for creating Association.
//
static String computeDSOName(const String& assocName) {
    char dsoName[PATH_MAX];
    SaStringFormat(dsoName, sizeof dsoName,
		   "%s/%s.so",
		   CONFIG_SYSADM_ASSOCIATION_DIR, (const char*) assocName);
    return dsoName;
}

//
//  static void* loadDSO(const String& dsoName)
//
//  Description:
//      Loads DSO
//
//  Parameters:
//      dsoName Name of DSO
//	errorString  String whose contents are initialized on failure.
//
//  Returns:
//	Return value of dlopen call.
//
static void* loadDSO(const String& dsoName, String& errorString) {
     void* dso = dlopen(dsoName, RTLD_NOW);
     if (dso == NULL) {
	 char str[Log::MAXLEN];
	 SaStringFormat(str, Log::MAXLEN,
			i18n("Can't load association: %s."),
			dlerror());
	 errorString = str;
	 Log::error(AssocFactory::NAME, (const char*) errorString);
     }
     return dso;
}

//
//  Association* AssocFactory::getAssociation(const String& parentCategoryName,
//  					      const String& parentSelector,
//  					      const String&
//  					      childCategoryName,
//					      String& errorString)
//
//  Description:
//      Static get method. 
//
//  Parameters:
//      parentCategoryName Category name of the parent item.
//      parentSelector Selector of the parent item.
//      childCategoryName Category name of the child item(s).
//	errorString  String whose contents are initialized on failure.
//
//  Returns:
//	A reference counted Association instance on success
//	NULL, on failure, errorString is intialized with the
//	localized error string
Association* AssocFactory::getAssociation(const String& parentCategoryName,
					  const String& parentSelector,
					  const String& childCategoryName,
					  String& errorString) {

    String assocName =
        Association::computeAssocName(parentCategoryName, childCategoryName);
    
    DictElem* dictElem = _assocs.lookupByKey(assocName);

    if (dictElem == NULL) {	
	Log::debug(NAME, "No dict elem for assocName %s", 
		   (const char*) assocName);
	String dsoName = computeDSOName(assocName);
	void* dso = loadDSO(dsoName, errorString);
	if (dso == NULL) {
	    return NULL;
	}
	
	String funcName = computeFuncName(assocName);
	GET_ASSOC getAssociation = (GET_ASSOC) dlsym(dso, funcName);
	if (getAssociation == NULL) {
	    char str[Log::MAXLEN];
	    SaStringFormat(str, Log::MAXLEN,
			   i18n("Can't find function %s in %s: %s."),
			       funcName, (const char*)dsoName,
			       dlerror());
	    errorString = str;
	    Log::error(AssocFactory::NAME, (const char*) errorString);
	    return NULL;
	}	

	Category* parentCategory = 
	    CategoryFactory::getCategory(parentCategoryName, errorString);
	if (parentCategory == NULL) {
	    dlclose(dso);
	    return NULL;
	}
	
	Category* childCategory = 
	    CategoryFactory::getCategory(childCategoryName, errorString);

	if (childCategory == NULL) {
	    // Do not lose the "real" error
	    String errorString;
	    CategoryFactory::releaseCategory(parentCategory, errorString);
	    dlclose(dso);
	    return NULL;
	}
	
	dictElem = new DictElem(*parentCategory, *childCategory,
				dso, getAssociation);
	_assocs.add(dictElem, new String(assocName));
    } else {
	Log::debug(NAME, "Dict elem for assocName %s already exists",
		   (const char*) assocName);
    }

    AssocElem* assocElem = dictElem->_assocElems.lookupByKey(parentSelector);
    if (assocElem == NULL) {
	Log::debug(NAME, "No assoc elem for parentSelector %s", 
		   (const char*) parentSelector);
	Association* assoc = (*dictElem->_getAssociation)
	    (dictElem->_parentCategory, parentSelector, 
	     dictElem->_childCategory);
	assocElem = new AssocElem(assoc, 1);
	dictElem->_assocElems.add(assocElem, new String(parentSelector));
    } else {
	Log::debug(NAME, "Assoc elem for parentSelector %s already exists", 
		   (const char*) parentSelector);
	assocElem->_refCount++;
    }
 
    assert(assocElem->_assoc != NULL);
    return assocElem->_assoc;
}

//
//  void AssocFactory::releaseAssociation(Association* assoc,
//	String& errorString)
//
//  Description:
//      Static release method.
//
//  Parameters:
//	assoc  The Association to be released.
//	errorString  String to initialize on failure.
//
//  Returns:
//	true, on success
//	false, on failure, errorString is intialized with the
//	localized error string
bool AssocFactory::releaseAssociation(Association* assoc,
				      String& errorString) {
    String parentCategoryName = assoc->getParentCategoryName();
    String childCategoryName = assoc->getChildCategoryName();
    String parentSelector = assoc->getParentSelector();

    String assocName =
        Association::computeAssocName(parentCategoryName, childCategoryName);
    DictElem* dictElem = _assocs.lookupByKey(assocName);
    assert (dictElem != NULL);

    AssocElem* assocElem =
        dictElem->_assocElems.lookupByKey(parentSelector);
    assert (assocElem != NULL);

    Log::debug(AssocFactory::NAME, "Unloading %s, %s, %s", 
	       (const char*) parentCategoryName, 
	       (const char*) childCategoryName, 
	       (const char*) parentSelector);

    assocElem->_refCount--;
    assert (assocElem->_refCount >= 0);
    if (assocElem->_refCount == 0) {
	dictElem->_assocElems.remove(parentSelector);
	delete assocElem->_assoc;
	delete assocElem;
    }
    
    int status = 0;
    if (dictElem->_assocElems.getSize() == 0) {
	_assocs.remove(assocName);
	if (!CategoryFactory::releaseCategory(&dictElem->_parentCategory, 
					      errorString)) {
	    return false;
	}
	if (!CategoryFactory::releaseCategory(&dictElem->_childCategory,
					      errorString)) {
	    return false;
	}   
	status = dlclose(dictElem->_dso);
	delete dictElem;
    }

    if (status == 0) {
	return true;
    } else {
	char str[Log::MAXLEN];
	SaStringFormat(str, 
		       Log::MAXLEN,
		       i18n("Can't unload association: %s."),
		       dlerror());
	errorString = str;
	Log::error(AssocFactory::NAME, (const char*) errorString);
	return false;
    }
}



END_NAMESPACE(sysadm);
