//
// Association.c++
//
//	Base class for deriving classes to represent relationships between
//	monitored items.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.4 $"

#include <sysadm/Association.h>
#include <sysadm/Log.h>
#include "ParentCategoryListener.h"
#include <string.h>

BEGIN_NAMESPACE(sysadm);

#define SaASSOC_PREPOSITION "sAssocWith";
#define SaASSOC_SEPARATOR "-";

const char* Association::NAME = "Association";

//
//  static String truncate(const String& categoryName)
//
//  Description:
//      Removes the "Category" suffix from the name of a Category.
//
//  Parameters:
//      categoryName Name of Category.
//
//  Returns:
//	Name of Category with suffix "Category" removed.
//
static String truncate(const String& categoryName) {
    int len = categoryName.getLength();
    int catLength = strlen("Category");
    assert(len > catLength);
    return categoryName.getSubString(0, len - catLength);
}

//
// Association implementation.
//
class AssociationImpl {
  public:
    AssociationImpl(Category& parentCategory,
		    const String& parentSelector,
		    const String& childCategoryName);
    ~AssociationImpl();
    const Item* _parent;
    const String* _parentSelector;
    Category& _parentCategory;
    const String* _childCategoryName;
    ParentCategoryListener* _parentListener;
};

//
//  AssociationImpl::AssociationImpl(Category& parentCategory,
//  				     const String& parentSelector)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentCategory Category of the parent item.
//      parentSelector Selector of the parent item.
//	childCategoryName Name of child category.
AssociationImpl::AssociationImpl(Category& parentCategory,
				 const String& parentSelector,
				 const String& childCategoryName)
 : _parentCategory(parentCategory),
   _parentSelector(new String(parentSelector)), _parent(NULL), 
   _parentListener(NULL), _childCategoryName(new String(childCategoryName)) {
}


//
//  AssociationImpl::~AssociationImpl()
//
//  Description:
//      Destructor.
//
AssociationImpl::~AssociationImpl() {
    delete _parentSelector;
    delete _childCategoryName;
    delete _parentListener;
}

//
//  Association::Association(Category& parentCategory, 
//  			     const String& parentSelector,
//  			     Category& parentSelector)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentCategory Category of the parent item.
//      parentSelector Selector of the parent item.
//      childCategory Category of the child item(s).
//
Association::Association(Category& parentCategory, 
			 const String& parentSelector,
			 Category& childCategory)
 : Category(computeSelector(parentCategory.getSelector(), 
			    parentSelector, childCategory.getSelector())),
   _impl(new AssociationImpl(parentCategory, parentSelector, 
			     childCategory.getSelector())) {
    Log::debug(NAME, "Association constructor for selector %s", 
	       (const char*) getSelector());
}


//
//  Association::~Association()
//
//  Description:
//      Destructor.
//
Association::~Association() {
    delete _impl;
}

//
//  void Association::startMonitor()
//
//  Description:
//      Start monitoring the system.
//	Creates ParentCategoryListener for receiving notifications
//	on the parent Item.
//
void Association::startMonitor() {
    Log::debug(NAME, "Started monitoring");

    _impl->_parentListener = 
        new ParentCategoryListener(_impl->_parentCategory,
				   *_impl->_parentSelector,
				   *this);
}

//
//  void Association::parentAdded(const Item& item)
//
//  Description:
//      Called by ParentCategoryListener when the parent item
//	is added or detected at startup.
//	Add the item to our state.
//
//  Parameters:
//      item The parent that was added.
//
void Association::parentAdded(const Item& item) {
    _impl->_parent = &item;
}

//
//  void Association::parentChanged(const Item& oldItem, 
//  				    const Item& newItem)
//
//  Description:
//      Called by ParentCategoryListener when the parent item
//	changes.
//	Change the item we point to.
//
//  Parameters:
//      oldItem The old parent.
//      newItem The new parent.
//
void Association::parentChanged(const Item& /*oldItem*/, 
				const Item& newItem) {
    _impl->_parent = &newItem;
}


//
//  void Association::parentRemoved()
//
//  Description:
//      Called by ParentCategoryListener when the parent item
//	is removed.
//	Remove all items from our list.
//
void Association::parentRemoved() {
    _impl->_parent = NULL;

    replaceItemList(ItemList());
}

//
//  String Association::computeSelector(const String& parentCategoryName, 
//  				        const String& parentSelector,
//  				        const String& childCategoryName)
//
//  Description:
//      Compute the selector for this association.
//
//  Parameters:
//      parentCategoryName Category name of the parent item.
//      parentSelector Selector of the parent item.
//      childCategoryName Selector of the parent item.
//
//  Returns:
//	The selector for this association.
//
String Association::computeSelector(const String& parentCategoryName, 
				    const String& parentSelector,
				    const String& childCategoryName) {
    String selector(truncate(childCategoryName));
    selector += SaASSOC_PREPOSITION;
    selector += truncate(parentCategoryName);
    selector += SaASSOC_SEPARATOR;
    selector += parentSelector;
    return selector;
}

//
//  String Association::computeAssocName(const String& parentCategoryName,
//  			                 const String& childCategoryName)
//
//  Description:
//      Compute the name for this type of association.
//
//  Parameters:
//      parentCategory Category name of the parent item.
//      childCategory Selector of the parent item.
//
//  Returns:
//	The name for this type of association.
//
String Association::computeAssocName(const String& parentCategoryName, 
				     const String& childCategoryName) {
    String selector(truncate(childCategoryName));
    selector += SaASSOC_PREPOSITION;
    selector += truncate(parentCategoryName);
    return selector;
}

//
//  const Item* Association::getParent()
//
//  Description:
//      Get the parent item of this association.
//
//  Returns:
//	The parent item of this association.
//
const Item* Association::getParent() {
    return _impl->_parent;
}

//
//  String Association::getParentSelector()
//
//  Description:
//      Get the selector of the parent item.
//
//  Returns:
//	Selector of the parent item.
//
String Association::getParentSelector() {
    return *_impl->_parentSelector;
}

//
//  String Association::getParentCategoryName()
//
//  Description:
// 	Get name of parent category.
//
//  Returns:
//	Name of parent category.
//
String Association::getParentCategoryName() {
    return _impl->_parentCategory.getSelector();
}

//
//  String Association::getChildCategoryName()
//
//  Description:
//      Get name of child category.
//
//  Returns:
//	Name of child category.
//
String Association::getChildCategoryName() {
    return *_impl->_childCategoryName;
}

END_NAMESPACE(sysadm);

