//
// Association.h
//
//	Base class for deriving classes to represent relationships between
//	monitored items.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.6 $"
#ifndef _SYSADM_ASSOCIATION_H
#define _SYSADM_ASSOCIATION_H

#include <sysadm/Category.h>
#include <sysadm/String.h>

BEGIN_NAMESPACE(sysadm);

typedef CollectionOf<String> StringList;

//
// An instance of a subclass of Association represents a dynamic collection of
// child Item(s) related to a parent Item identified by a
// selector. For example, the user account Item(s) that are related to
// a group account Item with selector "bar" could be represented by an
// Association.  An Association represents an 1-to-n relationship. A
// 1-to-1 relationship is handled as a special case of 1-to-n relationship.
//
// Association derives from Category and supports the same client
// mechanisms for obtaining dynamic information about the Item(s) in an
// Association instance. This class monitors the parent Item and
// subclasses are responsible for determining the child Item(s) that
// added, deleted and changed.  The Association class notifies changes
// to listeners.
//
// If an Item with the specified parent selector does not exist, the
// Association will monitor for the addition of an Item with that
// selector.  When such an Item is added, the subclasses are notified
// and subclasses can do whatever is necessary to determine the child
// Item(s) at that time and monitor the system for future changes. 
//
// If an Item with the specified parent selector existed, but is
// deleted while a client is monitoring the Association, all Items
// are removed from the Association and the Association continues
// monitoring of the system for the addition of an Item with that
// parent selector.
//
// The Association class (Vs. derived classes) is used only when sub-classes
// require logic very specific to the application to determine when
// items are added, changed and removed from the association.  For
// example, the relationship may be calculated by queries on a
// database based on "parentSelector".  The derived classes
// ComputedAssoc, ChildAttrAssoc and ParentAttrAssoc can be used when
// the data requiring for computing the relationship is contained
// within the Item(s) in the Category(s).
//
// See AssocFactory.h for details on the steps required to plug-in
// an Association.
//
class Association : public Category {
  public:
    // Compute the name (selector) for this association instance.
    static String computeSelector(const String& parentCategoryName, 
				  const String& parentSelector,
				  const String& childCategoryName);

    // Used internally for logging.
    static const char* NAME;

  protected:
    // Constructor which initializes Category base class with a unique
    // name based on the input parameters.
    //
    // All subclasses of Association must support a
    // constructor with this signature which will be called by
    // AssocFactory when an instance of Association is requested and
    // one does not exist for the specified combination of input
    // parameters.
    // "parentCategory" is the Category of the parent item.
    // "parentSelector" is the Selector of the parent item.
    // "childCategory" is the Category of the child item(s).
    Association(Category& parentCategory, const String& parentSelector,
		Category& childCategory);

    // Destructor.
    virtual ~Association();

    // Starts monitoring the system by registering for notifications
    // about the parent Item (identified by "parentSelector" in constructor).
    virtual void startMonitor();

    // Called if the parent Item (identified by "parentSelector" in
    // constructor) is determined to exist in the system at the time
    // monitoring is started.  Also called if that Item is added later to
    // the system.  "item" is the parent Item that was added.
    // This class provides a trivial implementation.
    // Subclasses can override this method to, for example, register
    // for notifications on Items in the child Category.
    // None of addItem(), changeItem(), orphanItem(), removeItem() or
    // replaceItemList() should be called prior to the call
    // to parentAdded().
    virtual void parentAdded(const Item& item);

    // Called when parent Item changes.
    // "oldItem" and "newItem" hold the previous and current state of the
    // parent Item respectively.
    // This class provides a trivial implementation.
    // Subclasses can override this method to do whatever is necessary
    // to keep the list of child Item(s) up-to-date.
    virtual void parentChanged(const Item& oldItem, const Item& newItem);
    
    // Called when parentItem is removed. Removes all Items from its
    // list and sends CategoryListener::itemRemoved() messages for
    // each child Item to registered listeners.
    virtual void parentRemoved();

    // Get the parent item of this association.
    virtual const Item* getParent();

    // Allow ParentCategoryListener to invoke our protected methods.
    friend class ParentCategoryListener;

    // Get the selector of the parent item.
    virtual String getParentSelector();

    // Get name of parent category.
    virtual String getParentCategoryName();

    // Get name of child category.
    virtual String getChildCategoryName();

  private:

    // Compute the name for this association
    static String computeAssocName(const String& parentCategoryName, 
				   const String& childCategoryName);

    // Intentionally undefined.	
    Association(const Association&);
    Association& operator=(const Association&);
    
    // Opaque implementation.
    friend class AssociationImpl;
    AssociationImpl* _impl;
    
    friend class AssocFactory;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_ASSOCIATION_H
