//
// Category.c++
//
//	Base class for deriving classes to represent categories of
//	monitored items.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.60 $"

#include <sysadm/Category.h>
#include <string.h>
#include <sysadm/Log.h>

BEGIN_NAMESPACE(sysadm);

const char* Category::NAME = "Category";

// Optimization to determine which list the 
// ListenerMap belongs to
enum ListenerMapState {MASTER, ADDED};

//
// Elements of the list matching listeners to filters.
//
class CategoryListenerHandle {
  public:
    CategoryListenerHandle(CategoryListener* listener, 
			   NotificationFilter* filter) : 
    _listener(listener), _filter(filter), _removed(false), _isAttrListener(false) {
    }

    ~CategoryListenerHandle() {
	delete _listener;
	delete _filter;
    }
    
    CategoryListener* _listener;
    NotificationFilter* _filter;
    bool _removed;
    bool _isAttrListener;

    ListenerMapState _state;
};

// Elements of the list mapping CategoryErrorListenerHandle to
// ErrorCategoryListener
class CategoryErrorListenerHandle {
  public:
    CategoryErrorListenerHandle(CategoryErrorListener* listener) :
    _listener(listener){
    }

    ~CategoryErrorListenerHandle() {
	delete _listener;
    }

    CategoryErrorListener* _listener;
};

typedef CategoryListenerHandle ListenerMap;
typedef CategoryErrorListenerHandle ErrorListenerMap;

//
// Category implementation.
//
class CategoryImpl {

  public:
  
    class ChangedPair {
      public:
	ChangedPair(Item* oldItem,
		    Item* newItem)
	    : _oldItem(oldItem), _newItem(newItem) { }
	Item* _oldItem;
	Item* _newItem;
    };

    CategoryImpl(Category& self) : _self(self), _monitoring(false), 
        _inNotification(false), _inBlockChangesNotification(false),
        _existsEnded(false) { 
    }

    ~CategoryImpl();

    void sortItemList(ItemList& list);
    static int compareItems(const void* lhs, const void* rhs);
    void startNotification();
    void notifyCurrentState(ListenerMap* lMap);
    void notifyAdded(Item& item);
    void notifyChanged(Item& oldItem, Item& newItem);
    void notifyRemoved(Item& item);
    void endNotification();

    OrderedCollectionOf<Item> _items;
    OrderedCollectionOf<ListenerMap> _listenerMaps;
    OrderedCollectionOf<ListenerMap> _addedListenerMaps;
    OrderedCollectionOf<ErrorListenerMap> _errorListenerMaps;

    Category& _self;

    // Indicate whether monitoring has been started or not
    // Monitoring is started on the first adoptNotificationFilter method
    // invocation.
    bool _monitoring;

    bool _inNotification;
    bool _inBlockChangesNotification;
    bool _existsEnded;
};

typedef class CategoryImpl::ChangedPair ChangedPair;

// CategoryImpl::~CategoryImpl
//
// Destructor.
//
CategoryImpl::~CategoryImpl() {
    RemoveAndDestroyContentsOf(&_items);
    RemoveAndDestroyContentsOf(&_listenerMaps);
    RemoveAndDestroyContentsOf(&_addedListenerMaps);
}

//
//  void CategoryImpl::sortItemList(ItemList& list)
//
//  Description:
//      Sort a list of Items.
//
void CategoryImpl::sortItemList(ItemList& list)
{
    // Copy our collection into a vector that qsort can deal with.
    int size = list.getSize();
    Item** vector = new Item*[size];
    IteratorOver<Item> iter(&list);
    Item* item;
    int i = 0;
    while ((item = iter()) != NULL) {
	vector[i++] = item;
    }

    // Make sure nothing funny happened with the number of items in
    // the list.
    assert(i == size);

    // Sort the vector.
    qsort(vector, size, sizeof(Item*), compareItems);

    // Copy from the vector back into our list.
    list.removeAll();
    for (i = 0; i < size; i++) {
	list.append(vector[i]);
    }

    delete [] vector;
}

//
//  int CategoryImpl::compareItems(const void* lhs, const void* rhs)
//
//  Description:
//      Compare two Items.  This is a compare func for qsort.
//
//  Parameters:
//      lhs  left hand side
//      rhs  right hand side
//
//  Returns:
//	-1 if lhs is less, 0 if the same, 1 if lhs is more.
//
int CategoryImpl::compareItems(const void* lhs, const void* rhs)
{
    const Item** leftItem = (const Item**)lhs;
    const Item** rightItem = (const Item**)rhs;

    return (*leftItem)->compare(**rightItem);
}

//
//  void CategoryImpl::startNotification()
//
//  Description:
//      Start notifying.
//
void CategoryImpl::startNotification() {
    // Nested notification will cause problems with our item list.
    assert(!_inNotification);
    _inNotification = true;
}


//
//  void CategoryImpl::notifyCurrentState(ListenerMap* lMap)
//
//  Description:
//      Make "listener" pointed to by lMap up to date on everything
//      that's happened so far by telling it about each existing
//      item. Also register listener as a AttrListener if indicated by
//      the "filter" associated with "listener".
//
//  Parameters:
//      lMap	ListenerMap with information about listener to get up
//      	to date and filter specifying data that the listener
//		is interested in.
//
void CategoryImpl::notifyCurrentState(ListenerMap* lMap) {
    
    CategoryListener* listener = lMap->_listener;
    NotificationFilter* filter = lMap->_filter;

#ifndef NDEBUG
    //  If this is the first listener,
    // _impl->_items and attrs should be empty 
    if (!_monitoring) {
	// Should be inexpensive as there should be zero elements
	CollectionOf<Attribute> attrs = _self.copyAttrList();
	assert(_items.getSize() == 0 && attrs.getSize() == 0);
    }
#endif // NDEBUG

    // Handle case where listener is added in the middle
    // of block changes notification
    if (_inBlockChangesNotification && !lMap->_removed) {
	listener->beginBlockChanges();
    }

    // Tell it about all the items we know about.
    IteratorOver<Item> iter(&_items);
    Item* item;
    while ((item = iter()) != NULL) {
	if (filter->monitorsItem(item->getSelector()) && !lMap->_removed) {
	    listener->itemAdded(*item);
	}
    }

    if (filter->monitorsCategoryAttrs()) {
	// Send category attributes
	CollectionOf<Attribute> attrs = _self.copyAttrList();
	IteratorOver<Attribute> attrIter(&attrs);
	Attribute* attr;
	while ((attr = attrIter()) != NULL && !lMap->_removed) {
	    AttrEvent event(_self, *attr);
	    listener->attrChanged(event);
	}
	RemoveAndDestroyContentsOf(&attrs);

	if (!lMap->_removed) {
	    lMap->_isAttrListener = true;
	    // Inform future changes to category attributes
	    _self.adoptAttrListener(listener);
	}
    }

    if (_existsEnded && !lMap->_removed) {
	listener->endExists();
    }

    // If this is the first listener we've gotten, tell our subclass to
    // start monitoring.
    if (!_monitoring) {
	_monitoring = true;
	_self.startMonitor();
    }
}

//
//  void CategoryImpl::notifyChanged(Item& oldItem, Item& newItem)
//
//  Description:
//      Notify listeners that "oldItem" changed.
//
//  Parameters:
//      oldItem  The old item.
//      newItem  The new item.
//
void CategoryImpl::notifyChanged(Item& oldItem, Item& newItem)
{
    startNotification();
    IteratorOver<ListenerMap> iter(&_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = iter()) != NULL) {
	if (lMap->_removed) {
	    continue;
	}

	if (lMap->_filter->monitorsItem(newItem.getSelector())) {
	    lMap->_listener->itemChanged(oldItem, newItem);
	}
    }
    endNotification();
}

//
//  void CategoryImpl::notifyRemoved(Item& item)
//
//  Description:
//      Notify listeners that item was removed.
//
//  Parameters:
//      item  The item that's being removed.
//
void CategoryImpl::notifyRemoved(Item& item)
{
    startNotification();
    IteratorOver<ListenerMap> iter(&_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = iter()) != NULL) {
	if (lMap->_removed) {
	    continue;
	}

	if (lMap->_filter->monitorsItem(item.getSelector())) {
	    lMap->_listener->itemRemoved(item);
	}	
    }
    endNotification();
}

//
//  void CategoryImpl::notifyAdded(Item& item)
//
//  Description:
//      Notify listeners that item was added.
//
//  Parameters:
//      item  The item that was added.
//
void CategoryImpl::notifyAdded(Item& item)
{
    startNotification();
    IteratorOver<ListenerMap> iter(&_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = iter()) != NULL) {
	if (lMap->_removed) {
	    continue;
	}

	if (lMap->_filter->monitorsItem(item.getSelector())) {   
	    lMap->_listener->itemAdded(item);
	}
    }
    endNotification();
}

//
//  void CategoryImpl::endNotification()
//
//  Description:
//      End notifying.
//
void CategoryImpl::endNotification()
{
    assert(_inNotification);
    _inNotification = false;

    // Actually remove any listeners for which orphanCategoryListener was
    // called during notification.
    IteratorOver<ListenerMap> iter(&_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = iter()) != NULL) {
	if (lMap->_removed) {
	    // Check for lMap->_filter->monitorsCategoryAttrs()
	    // is insufficient as this could be true,
	    // but it was never added as an AttrListener
	    // because of a removeCategoryListener call
	    // during notifyCurrentState()
	    if (lMap->_isAttrListener) {
		_self.orphanAttrListener(lMap->_listener);
	    }
	    delete iter.remove();
	}
    }

    // Really add any listeners for which adoptCategoryListener was called
    // during notification.
    if (_addedListenerMaps.getSize() > 0) {
	// Make a copy of addedListenerMaps, because we're going to have
	// to do more notification to let the new listeners know what's
	// going on, and one of the new listeners might decide to add
	// another listener.
	OrderedCollectionOf<ListenerMap> addedListenerMaps(_addedListenerMaps);
	_addedListenerMaps.removeAll();

	IteratorOver<ListenerMap> addedIter(&addedListenerMaps);
	while ((lMap = addedIter()) != NULL) {
	    lMap->_state = MASTER;
	    _listenerMaps.append(lMap);
	}

	startNotification();
	addedIter.reset();
	while ((lMap = addedIter()) != NULL) {
	    if (lMap->_removed) {
		continue;
	    }
	    notifyCurrentState(lMap);
	}
	endNotification();
    }
}

//
//  CategoryListenerHandle*
//  Category::adoptCategoryListener(CategoryListener* listener)
//
//  Description:
//	Add a listener to the list of objects which will get notified
//	of changes to the Category being monitored.
//
//  Params:
//	listener	object to call when something happens
//	filter		The notifications of interest to listener object
//  
//  Returns:
//	The handle to be used for this listener in removeCategoryListener
//	and orphanCategoryListener calls.
CategoryListenerHandle*
Category::adoptCategoryListener(CategoryListener* listener, 
				const NotificationFilter& filter) {

    assert(listener != NULL);

    // Check if listener already monitors another category
    assert(!listener->isAdopted());

#ifndef NDEBUG
    // Check if listener already exists in _listenerMaps
    if (_impl->_listenerMaps.getSize() > 0) {

	IteratorOver<ListenerMap> iter(&_impl->_listenerMaps);
	ListenerMap* lMap;
	while ((lMap = iter()) != NULL) {
	    assert(lMap->_listener != listener);
	}	
    }
    
    // listener does not exist in _listenerMaps
    if (_impl->_inNotification && _impl->_addedListenerMaps.getSize() > 0) {

	// listener could be in _addedListenerMaps
	 IteratorOver<ListenerMap>
	     addedIter(&_impl->_addedListenerMaps);
	 ListenerMap* lMap;
	 while ((lMap = addedIter()) != NULL) {
	      assert(lMap->_listener != listener);
	 }
    }
#endif // NDEBUG

    // Increment listener's category reference count for debugging.
    listener->adopt();

    // listener does not exist, create new ListenerMap element
    NotificationFilter* filterClone = filter.clone();
    ListenerMap* lMap = new ListenerMap(listener, filterClone);
    
    // Add it to our list.
    if (_impl->_inNotification) {
        // If we're in the middle of notification, adding to our
	// master list will blow the iterator that's notifying, and
	// this new listener might not get an accurate picture of the
	// category.  So we'll hold on the the new listener in
	// _addedListenerMaps and add it when we're done notifying.
	lMap->_state = ADDED;
	_impl->_addedListenerMaps.append(lMap);
    } else {
	lMap->_state = MASTER;
	_impl->_listenerMaps.append(lMap);
	_impl->notifyCurrentState(lMap);
    }

    return lMap;
}

//
//  CategoryListener* 
//    Category::orphanCategoryListener(CategoryListenerHandle* 
//				       lMap)
//
//  Description:
//      Remove a listener.
//
//  Parameters:
//       listenerHandle	The handle to listener to remove.
//
CategoryListener* Category::orphanCategoryListener(CategoryListenerHandle* 
						   lMap) {
    assert (lMap != NULL);

    CategoryListener* listener = NULL;

    switch (lMap->_state) {
    case MASTER:
	if (_impl->_inNotification) {
	    lMap->_removed = true;
	    listener = lMap->_listener;
	    lMap->_listener = NULL;
	} else {
	    if (lMap->_filter->monitorsCategoryAttrs()) {
		orphanAttrListener(lMap->_listener);
	    }
	    _impl->_listenerMaps.removePointer(lMap);
	    listener = lMap->_listener;
	    lMap->_listener = NULL;
	    delete lMap;   
	}
	break;

    case ADDED:
	    // Can be in _addedListenerMaps (inNotification == true)
	    //	=> blow it off
	    _impl->_addedListenerMaps.removePointer(lMap);
	    listener = lMap->_listener;
	    lMap->_listener = NULL;
	    delete lMap;   
	    break;
    }

    listener->orphan();
    return listener;
}

//
//  void Category::removeCategoryListener(CategoryListenerHandle* 
//					  listenerHandle)
//
//  Description:
//      Remove a listener from list and memory.
//
//  Parameters:
//       listenerHandle	The handle to listener to remove.
//
void Category::removeCategoryListener(CategoryListenerHandle* 
				      listenerHandle) {
    delete orphanCategoryListener(listenerHandle);
}

//
// Constructor.
//
//  Parameters:
//	selector	The type of the monitored Item that this
//			Category monitors

Category::Category(const String& selector) : 
    AttrBundle(NAME, selector), 
    _impl(new CategoryImpl(*this)) {
}

//
// Destructor.
//
Category::~Category() {
    // Decrement the number of categories that the listeners monitor
    // Also, if the listener monitors Category Attributes, remove it
    // from the list of listeners in AttrBundle.  Else, it will be
    // deleted twice - once by Category and again by AttrBundle.
    IteratorOver<ListenerMap> lIter(&_impl->_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = lIter()) != NULL) {
	lMap->_listener->orphan();
	if (lMap->_filter->monitorsCategoryAttrs()) {
	    orphanAttrListener(lMap->_listener);
	}
    }

    IteratorOver<ListenerMap> aLIter(&_impl->_addedListenerMaps);
    while ((lMap = aLIter()) != NULL) {
	lMap->_listener->orphan();
	if (lMap->_filter->monitorsCategoryAttrs()) {
	    orphanAttrListener(lMap->_listener);
	}
    }
    delete _impl;
}

//
//  void Category::addItem(const Item& item)
//
//  Description:
//      Called by subclasses when a new item is added to the system.
//      Add the item to our list and inform listeners.
//
//  Parameters:
//      item  The item that was added.
//
void Category::addItem(const Item& item) {
    Log::debug(NAME, "addItem: %s", (const char*) item.toString());
    assert(_impl->_monitoring == true);

#ifndef NDEBUG
    IteratorOver<Item> itemIter(&_impl->_items);
    Item* object = NULL;
    while ((object = itemIter()) != NULL) {
	// Make sure that the item does not already exist.
	assert(object->compare(item) != 0);
    }
#endif

    Item* itemClone = dynamic_cast<Item*>(item.clone());
    assert(itemClone != NULL);
    _impl->_items.append(itemClone);
    _impl->notifyAdded(*itemClone);
}

//
//  void Category::changeItem(const Item& item)
//
//  Description:
//      Called by subclasses when an item in the system changed.
//      Change the corresponding item in our list and inform listeners.
//
//  Parameters:
//      item  The item that changed.
//
void Category::changeItem(const Item& item) {
    Log::debug(NAME, "selector: %s", (const char*) getSelector());
    Log::debug(NAME, "changeItem: %s",
	       (const char*) item.toString());
    assert(_impl->_monitoring == true);

    Item* itemClone = NULL;

    IteratorOver<Item> itemIter(&_impl->_items);
    Item* object = NULL;
    while ((object = itemIter()) != NULL) {
	if (object->compare(item) == 0) {
	    // if it's the same object.
	    if (!object->equals(item)) {
		itemClone = dynamic_cast<Item*>(item.clone());
		assert(itemClone != NULL);
		itemIter.insertBefore(itemClone);
		itemIter.remove();
		
	    }
	    
	    break;
	}
    }

    // make sure that this is a change to existing object
    assert(object != NULL);

    // item not found or item not changed
    if (itemClone == NULL)
	return;

    _impl->notifyChanged(*object, *itemClone);
    delete object;
}

//
//  Item* Category::orphanItem(const String& selector)
//
//  Description:
//      Called by subclasses when an item is removed from the
//      system.  Remove the item from our list and inform listeners.
//
//  Parameters:
//      selector  selector of the item that was removed.
//
//  Returns:
//	The object that was removed from the list of items (but the returned
//	object isn't actually removed from memory).

Item* Category::orphanItem(const String& selector) {
    assert(_impl->_monitoring == true);
    Item* removedItem = NULL;

    IteratorOver<Item> itemIter(&_impl->_items);
    Item* item;
    while ((item = itemIter()) != NULL) {
	if (strcmp(item->getSelector(), selector) == 0) {
	    removedItem = itemIter.remove();
	    _impl->notifyRemoved(*removedItem);
	    break;
	}
    }

    return removedItem;
}

//
//  void Category::removeItem(const String& selector)
//
//  Description:
//      Called by subclasses when an item is removed from the
//      system. Remove the item from our list and memory and inform listeners.
//
//  Parameters:
//      selector  selector of the item that was removed.
//
void Category::removeItem(const String& selector) {
    assert(_impl->_monitoring == true);
    delete orphanItem(selector);
}

//
//  void Category::replaceItemList(const ItemList& list)
//
//  Description:
//      Make "list" the new item list for this Category. 
//	If no items have been added yet, this is really easy; we
//      just call addItem() for each object in "list".
//
//	Otherwise, we move the contents of our list (_impl->_items)
//	to "oldList", and step through both the old list and the new
//	list at the same time, adding, changing, or deleting items
//	as appropriate.	
//
//  Parameters:
//      list  New list of monitored items.
//
void Category::replaceItemList(const ItemList& list) {
    assert(_impl->_monitoring == true);
    beginBlockChanges();

    if (_impl->_items.getSize() == 0) {

	ConstIteratorOver<Item> iter(&list);
	const Item* item;
	while ((item = iter()) != NULL) {
	    addItem(*item);
	}

    } else {

	// Make a copy since we want to sort the list
	ItemList listClone(list);
	_impl->sortItemList(listClone);

	// We do our work on a copy of the old list.
	ItemList oldList(_impl->_items);
	ItemList newList;
	ItemList addedList;
	OrderedCollectionOf<ChangedPair> changedList;
	ItemList cruftList;
	_impl->sortItemList(oldList);

	IteratorOver<Item> oldIter(&oldList);
	IteratorOver<Item> newIter(&listClone);
	Item* newItem = newIter();
	Item* oldItem = oldIter();

	while (newItem != NULL) {

	    if (oldItem == NULL) {
		// There are no more items in the old list; all of
		// the remaining new items are to be added.
		addedList.append(newItem);
		newItem = newIter();
		continue;
	    }
		
	    int comparison = newItem->compare(*oldItem);

	    if (comparison < 0) {
		// newItem is lexically less than oldItem, so it's a new
		// one.
		addedList.append(newItem);
		newItem = newIter();
	    } else if (comparison == 0) {
		// newItem and oldItem refer to the same item.
		// Determine whether the item has actually changed.
		if (!oldItem->equals(*newItem)) {
		    Item* newItemClone = dynamic_cast<Item*>(newItem->clone());
		    assert(newItemClone != NULL);
		    newList.append(newItemClone);
		    changedList.append(new ChangedPair(oldItem, newItemClone));
		} else {
		    // Put the old item back in the list since it
		    // hasn't changed.
		    newList.append(oldItem);
		}
		newItem = newIter();
		oldItem = oldIter();
	    } else if (comparison > 0) {
		// old item has been removed.  Don't actually remove
		// the item now, because it's still in the master
		// list.
		_impl->notifyRemoved(*oldItem);
		cruftList.append(oldItem);
		oldItem = oldIter();
	    }
	}

	// Check for remaining old items.
	while (oldItem != NULL) {
	    // Don't actually remove the item now, because it's
	    // still in the master list.
	    _impl->notifyRemoved(*oldItem);
	    cruftList.append(oldItem);
	    oldItem = oldIter();
	}

	_impl->_items = newList;

	// Delayed notification of listeners for changes because some
	// listeners may rely on the state of Category.
	IteratorOver<ChangedPair> changedIter(&changedList);
	ChangedPair* pair;
	while ((pair = changedIter()) != NULL) {
	    _impl->notifyChanged(*pair->_oldItem, *pair->_newItem);
	    delete pair->_oldItem;
	    delete changedIter.remove();
	}

	// Now that we've notified listeners of "changed()" items,
	// it's safe to delete the old items.
	IteratorOver<Item> cruftIter(&cruftList);
	Item* item;
	while ((item = cruftIter()) != NULL) {
	    delete cruftIter.remove();
	}

	IteratorOver<Item> addedIter(&addedList);
	while ((item = addedIter()) != NULL) {
	    addItem(*item);
	}

	listClone.removeAll();
    }

    endBlockChanges();
}

//
//  void Category::beginBlockChanges()
//
//  Description:
//      Begin a change block.  CategoryListeners can defer doing expensive
//      stuff until the change block ends.
//
void Category::beginBlockChanges() {
    Log::debug(NAME, "beginBlockChanges: %s", (const char*) getSelector());
    // Ensure that this is not a nested beginBlockChanges call
    assert(!_impl->_inBlockChangesNotification);
    _impl->_inBlockChangesNotification = true;

    _impl->startNotification();
    IteratorOver<ListenerMap> iter(&_impl->_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = iter()) != NULL) {
	if (lMap->_removed) {
	    continue;
	}

	lMap->_listener->beginBlockChanges();
    }
    _impl->endNotification();
}

//
//  void Category::endBlockChanges()
//
//  Description:
//      End a change block.
//
void Category::endBlockChanges() {
    Log::debug(NAME, "endBlockChanges: %s", (const char*) getSelector());
    assert(_impl->_inBlockChangesNotification);
    _impl->_inBlockChangesNotification = false;

    _impl->startNotification();
    IteratorOver<ListenerMap> iter(&_impl->_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = iter()) != NULL) {
	if (lMap->_removed) {
	    continue;
	}

	lMap->_listener->endBlockChanges();
    }		
    _impl->endNotification();
}

//
//  void Category::endExists()
//
//  Description:
//      End of notifications on existing objects.
//
void Category::endExists() {

    assert(!_impl->_existsEnded);
    _impl->_existsEnded = true;

    _impl->startNotification();
    IteratorOver<ListenerMap> iter(&_impl->_listenerMaps);
    ListenerMap* lMap;
    while ((lMap = iter()) != NULL) {
	if (lMap->_removed) {
	    continue;
	}

	lMap->_listener->endExists();
    }		

    _impl->endNotification();
}

//
//  const ItemList* Category::getItems()
//
//  Description:
//      Get the current list of items.
//
//  Returns:
//	The current list of items.
//
const ItemList* Category::getItems() {
    return &_impl->_items;
}

//
//  bool Category::isMonitoring()
//
//  Description:
//      Called by subclasses to check if the Category 
//	has started monitoring the system
//
//  Returns:
//	true if the Category is monitoring the system, false otherwise
//
bool Category::isMonitoring() {
    return _impl->_monitoring;
}

//
//  bool Category::isInBlockChangesNotification()
//
//  Description:
//	Called by subclasses to check if the Category 
//	is in the middle of a block changes notification.
//
//  Returns:
//	true if the Category is in the middle of a block 
//	changes notification, false otherwise,
//
bool Category::isInBlockChangesNotification() {
    return _impl->_inBlockChangesNotification;
}

//
//  bool Category::hasExistsEnded()
//
//  Description:
//	Called by subclasses to check if the Category 
//	has received an endExists notification.
//
//  Returns:
//	true if the Category has received an endExists,
//	notification, false otherwise.
//
bool Category::hasExistsEnded() {
    return _impl->_existsEnded;
}

//
//  CategoryErrorListenerHandle* 
//  Category::adoptErrorListener(CategoryErrorListener* listener)
//
//  Description:
//      Add an error listener to the list of objects which will get notified
//	of errors to the Category being monitored.
//
//  Parameters:
//      listener	object to call when error occurs 
//
//  Returns:
//	The handle to be used for this listener in 
//	orphanCategoryErrorListener calls.
//
CategoryErrorListenerHandle* 
Category::adoptErrorListener(CategoryErrorListener* listener) {
    assert(listener != NULL);

#ifndef NDEBUG
    // Check if listener already exists in _errorListenerMaps
    if (_impl->_errorListenerMaps.getSize() > 0) {
	IteratorOver<ErrorListenerMap> iter(&_impl->_errorListenerMaps);
	ErrorListenerMap* elMap;
		while ((elMap = iter()) != NULL) {
	    assert(elMap->_listener != listener);
	}	
    }
#endif // NDEBUG

    // listener does not exist, create new ListenerMap element
    ErrorListenerMap* elMap = new ErrorListenerMap(listener);
    _impl->_errorListenerMaps.append(elMap);

    return elMap;
}

//
//   CategoryErrorListener*
//   Category::orphanErrorListener(CategoryErrorListenerHandle* 
//  				   elMap)
//
//  Description:
//      Remove an error listener.
//
//  Parameters:
//      listenerHandle	The handle to error listener to remove.
//
CategoryErrorListener*
Category::orphanErrorListener(CategoryErrorListenerHandle* elMap) {
     assert (elMap != NULL);

     _impl->_errorListenerMaps.removePointer(elMap);
     CategoryErrorListener* listener = elMap->_listener;
     elMap->_listener = NULL;
     delete elMap;
     return listener;
}

//
//  void Category::removeErrorListener(CategoryErrorListenerHandle* 
//  				       listenerHandle)
//
//  Description:
//      Remove an error listener from list and memory.
//
//  Parameters:
//      listenerHandle	The handle to error listener to remove
//
void Category::removeErrorListener(CategoryErrorListenerHandle* 
				   listenerHandle) {
    delete orphanErrorListener(listenerHandle);
}

//
//  void Category::notifyError(const String& errorString)
//
//  Description:
//      Notify listeners that an error has occurred.
//
//  Parameters:
//      errorString Localized string describing the error.
//
void Category::notifyError(const String& errorString) {
    Log::debug(NAME, "Category Error: %s", (const char*) errorString);
    CategoryErrorEvent* event = new CategoryErrorEvent(*this, errorString);

    IteratorOver<ErrorListenerMap> iter(&_impl->_errorListenerMaps);
    ErrorListenerMap* elMap;

    while ((elMap = iter()) != NULL) {
	elMap->_listener->categoryError(*event);
    }

    delete event;
}

END_NAMESPACE(sysadm);


