//
// CategoryFactory.c++
//
//	Factory class for Category objects.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.10 $"

#include <stdio.h>
#include <limits.h>
#include <dlfcn.h>
#include <assert.h>

#include <sysadm/Log.h>
#include <sysadm/CategoryFactory.h>
#include <sysadm/format.h>
#include <sysadm/i18n.h>

BEGIN_NAMESPACE(sysadm);

const char* CategoryFactory::NAME = "CategoryFactory";

DictionaryOf<struct CategoryFactory::DictElem> 
    CategoryFactory::_categories;


//
//  static String computeFuncName(const String& catName)
//
//  Description:
//      Computes function name that will return a new instance 
//	of Category.
//
//  Parameters:
//      catName The name of the Category requested.
//
//  Returns:
//	Function name for creating Category.
//
static String computeFuncName(const String& catName) {
    int len = 3 + catName.getLength() + 1;
    char funcName[len];
    SaStringFormat(funcName, len, "get%s", (const char*)catName);
    return funcName;
}


//
//  static String computeDSOName(const String& catName) 
//
//  Description:
//      Computes DSO name that contains symbol for creation of
//	a new instance of Category.
//
//  Parameters:
//      catName The name of the Category requested.
//
//  Returns:
//	DSO name with symbol for creating Category.
//
static String computeDSOName(const String& catName) {
    char dsoName[PATH_MAX];
    SaStringFormat(dsoName, sizeof dsoName,
		   "%s/%s.so",
		   CONFIG_SYSADM_CATEGORY_DIR, (const char*) catName);
    return dsoName;
}

//
//  Category* CategoryFactory::getCategory(const String& catName,
//	String& errorString)
//
//  Description:
//      Static get method. 
//
//  Parameters:
//      catName  The name of the Category requested.
//	errorString  String to initialize on failure.
//
//  Returns:
//	A reference counted Category instance on success
//	NULL, on failure, errorString is intialized with the
//	localized error string
Category* CategoryFactory::getCategory(const String& catName, 
				       String& errorString) {
    DictElem* dictElem = _categories.lookupByKey(catName);

    if (dictElem == NULL) {	
	Log::debug(NAME, i18n("Creating new instance of category"));
	String dsoName = computeDSOName(catName);
	void* dso = dlopen(dsoName, RTLD_NOW);
	if (dso == NULL) {
	    char str[Log::MAXLEN];
	    SaStringFormat(str, Log::MAXLEN,
			   i18n("Can't load category: %s."), dlerror());
	    errorString = str;
	    Log::error(CategoryFactory::NAME, (const char*) errorString);
	    return NULL;
	}
	
	String funcName = computeFuncName(catName);
	Category* (*getCategory)() = (Category* (*)())dlsym(dso, funcName);
	if (getCategory == NULL) {
	    char str[Log::MAXLEN];
	    SaStringFormat(str, 
			   Log::MAXLEN,
			   i18n("Can't find function %s in %s: %s."),
                           (const char*)funcName, 
                           (const char*)dsoName, dlerror());
	    errorString = str;
	    Log::error(NAME, (const char*) errorString);
	    return NULL;
	}	

	dictElem = new DictElem(getCategory(), dso, 1);
	_categories.add(dictElem, new String(catName));
    } else {
	Log::debug(NAME, i18n("Instance of category already exists"));
	dictElem->_refCount++;
    }
 
    assert(dictElem->_category != NULL);
    return dictElem->_category;
}

//
//  bool CategoryFactory::releaseCategory(Category* category,
//	String& errorString)
//
//  Description:
//      Static release method.
//
//  Parameters:
//      category The Category to be released.
//	errorString  String to initialize on failure.
//
//  Returns:
//	true, on success
//	false, on failure, errorString is intialized with the
//	localized error string
bool CategoryFactory::releaseCategory(Category* category,
				      String& errorString) {
    String catName = category->getSelector();
    DictElem* dictElem = _categories.lookupByKey(catName);
    assert (dictElem != NULL);

    Log::debug(CategoryFactory::NAME, "Unloading %s", (const char*) catName);

    dictElem->_refCount--;
    assert (dictElem->_refCount >= 0);
    int status = 0;
    if (dictElem->_refCount == 0) {
	_categories.remove(catName);
	delete dictElem->_category;
	status = dlclose(dictElem->_dso);
	delete dictElem;
    }

    if (status == 0) {
	return true;
    } else {
	char str[Log::MAXLEN];
	SaStringFormat(str, 
		       Log::MAXLEN,
		       i18n("Can't unload category: %s."), dlerror());
	errorString = str;
	Log::error(CategoryFactory::NAME, (const char*) errorString);
	return false;
    }
}

END_NAMESPACE(sysadm);
