//
// CategoryFactory.h
//
//	Factory class for Category objects.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.6 $"
#ifndef _SYSADM_CATEGORY_FACTORY_H
#define _SYSADM_CATEGORY_FACTORY_H

#include <sysadm/Category.h>

BEGIN_NAMESPACE(sysadm);

//
// CategoryFactory is the factory class for Category objects.
// CategoryFactory methods are mostly used by the Category Service,
// described in sysadmd(1M), to fulfill requests from remote clients.
// They can also be used by any server-side components that require
// information from a Category.  Category subclasses will use the
// macros defined here.
//
// To make information about a Category of type "catName" available to
// the rest of the system the following steps are required:
//
// 1.  Implement a subclass of Category called "catName".
//	- The subclass must have a void constructor.  Typically, this calls the
//	Category base class constructor with the argument "catName".  
//	- Use the convenience macro SaCATEGORY_REF_DECL in the header
//	file to provide declaration for the routines used by the
//	Category Service for obtaining Category instances.
//	- In most cases, only one instance of a particular subclass of
//	Category should exist in an address space. To enforce this, 
//	subclasses should protect their constructors and use the 
//      convenience macro SaCATEGORY_FRIEND_DEF in the class
//      declaration in the header file.
//      - Use the convenience macro SaCATEGORY_REF_DEF in the c++
//      file to provide the definition for the routines used by the
//	Category Service for obtaining Category instances.  This in
//	turn will use the void constructor.
// 2.  Create a library called "catName".so.
// 3.  Install it in /usr/sysadm/category/
//
// The above steps will allow clients to obtain the Category instance
// for "catName".  Category.h describes the different methods
// supported by Category.
// 
class CategoryFactory {
  
  public:

    // Static get method.  Creates a Category instance for catName, if
    // it does not exist.  Returns a reference counted Category instance.
    //
    // The method fails if the Category instance cannot be created.
    // On failure, returns NULL and errorString is initialized with
    // the localized error string.  This can happen if the library 
    // corresponding to catName cannot be loaded or if the library
    // exists but does not define the symbols required. 
    
    static Category* getCategory(const String& catName, String& errorString);


    // Static release method.  Decrements reference count for 
    // Category instance.  Deletes the instance, if there are no
    // references to it.
    //
    // If no error is encountered in uninitializing the internal data 
    // structures used for creating the Category, returns true.
    // Otherwise, returns false and errorString is initialized with
    // the localized error string.  This can happen if the library 
    // loaded for creation of the Category instance cannot be unloaded.
    
    static bool releaseCategory(Category* category, String& errorString);
  
    // Used internally for logging.
    static const char* NAME;

  private:

    // Pointer to the object and a reference count. 
    struct DictElem {
	Category* _category;
	void* _dso;
	int _refCount;
	DictElem(Category* category, void* dso, int refCount) 
	    : _category(category), _dso(dso), _refCount(refCount) {
	}
    };
    static DictionaryOf<DictElem> _categories;

};

END_NAMESPACE(sysadm);

//
// SaCATEGORY_REF_DECL should be used in header files declaring classes
// which are accessed via CategoryFactory.  For example:
//
//    SaCATEGORY_REF_DECL(ExampleClass);
//    class ExampleCategory : public Category {
//    ...
//    };

#define SaCATEGORY_REF_DECL(className) \
    extern "C" NAMESPACE_SCOPE(sysadm)Category* get##className();

//
// SaCATEGORY_FRIEND_DEF should be used in header files 
// within the declaration of classes which
// are accessed via CategoryFactory.  For example:
//
//    class ExampleCategory : public Category {
//    ...
//    protected:
//      SaCATEGORY_FRIEND_DEF(ExampleClass);
//    ...
//    };

#define SaCATEGORY_FRIEND_DEF(className) \
    friend NAMESPACE_SCOPE(sysadm)Category* get##className(); \
    friend class NAMESPACE_SCOPE(sysadm)CategoryFactory;

//
// SaCATEGORY_REF_DEF should be used in c++ files defining classes which
// are accessed via CategoryFactory.  For example:
//
//    #include "ExampleCategory.h"
//    ...
//    SaCATEGORY_REF_DEF(ExampleClass);
//    ...	
//    ExampleCategory::ExampleCategory() {
//    ...
//    }
//    ...

#define SaCATEGORY_REF_DEF(className) \
    extern "C" NAMESPACE_SCOPE(sysadm)Category* get##className() \
    { \
	return new className(); \
    }



#endif  //  _SYSADM_CATEGORY_FACTORY_H
