//
// CategoryListener.h
//
//	Class for monitoring changes in a Category.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.20 $"
#ifndef _SYSADM_CATEGORY_LISTENER_H
#define _SYSADM_CATEGORY_LISTENER_H

#include <sysadm/Item.h>
#include <sysadm/AttrListener.h>

BEGIN_NAMESPACE(sysadm);

//
// An instance of a subclass of CategoryListener receives
// notifications of either the entire list or a subset of Item(s) in
// an instance of a subclass of Category. Furthermore, it can receive 
// notifications of changes to the attributes of the Category instance.
//
// A CategoryListener can expect to receive zero or more
// itemAdded() and attrAdded() 
// calls, followed by an endExists() call followed by zero or
// more addItem(), changeItem(),
// removeItem(), attrAdded(), attrChanged and
// attrRemoved calls.   The endExists() call signals
// that the Category has communicated the entire set of
// Item(s) discovered in the system to the CategoryListener. 
//
// See Category.h for details on how CategoryListener instances can be
// used with a Category instance.
//
// The base class provides do-nothing implementations of
// virtual methods which subclasses override.
//
// CategoryListener instances are mostly used by the Category Service,
// described in sysadmd(1M), to fulfill requests from remote clients.
// They can also be used by any server-side components that require
// information from a Category.
//
class CategoryListener : public AttrListener {

  public:

    // Constructor.
    CategoryListener();

    // Destructor.
    virtual ~CategoryListener();
    
    // Called by Category to notify current list of Item(s) at the time
    // CategoryListener registers for notifications and when Item(s) are
    // later discovered by Category or added to the system.
    virtual void itemAdded(const Item& item);
    
    // Called by Category when an Item changes.  
    virtual void itemChanged(const Item& oldItem, const Item& newItem);

    // Called by Category when an Item is removed.
    virtual void itemRemoved(const Item& item);

    // Called by Category prior to a block of
    // changes.  For example, Category::replaceItemList() calls
    // beginBlockChanges() before changing the list.
    virtual void beginBlockChanges();

    // Called by Category after a block of changes.  For example,
    // Category::replaceItemList() calls endBlockChanges() after
    // changing the list. 
    virtual void endBlockChanges();

    //   
    // Called by Category when it has finished notifying
    // CategoryListener of all Item(s) of a specific type in a system.
    // If the Category has received the endExists() notification from
    // its subclasses before the time the CategoryListener registers for
    // notification, (indicating that the Category subclass has
    // completed discovery of the existing Item(s) of the specific
    // type in the system) it will notify CategoryListener of all the
    // Item(s) in its list and send the endExists() notification.  If not, it
    // will send the endExists() notifications when it receives the
    // corresponding notification from its subclasses.
    //
    virtual void endExists();

    // Called by Category to notify current list of Attribute(s) at the time
    // CategoryListener registers for notifications and when Attribute(s) are
    // later discovered by Category or added.
    virtual void attrChanged(const AttrEvent& event);

  private:	

    // Intentionally undefined.
    CategoryListener(const CategoryListener&);
    CategoryListener& operator=(const CategoryListener&);

    // Private interface used by Category to ensure that listeners
    // are not adopted by more than one Category.
    friend class Category;
    void adopt();
    void orphan();
    bool isAdopted();

    // Opaque implementation.
    friend class CategoryListenerImpl;
    CategoryListenerImpl* _impl;
    
};


END_NAMESPACE(sysadm);
#endif  //  _SYSADM_CATEGORY_LISTENER_H
