//
// ChildAttrAssoc.c++
//
//	Base class for deriving classes to represent relationships governed
//	by attributes of the monitored child items.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
#ident "$Revision: 1.6 $"

#include <sysadm/ChildAttrAssoc.h>
#include <sysadm/Log.h>
#include <string.h>

BEGIN_NAMESPACE(sysadm);

const char* ChildAttrAssoc::NAME = "ChildAttrAssoc";

// ChildAttrAssoc Implementation.
class ChildAttrAssocImpl {
  public:
    ChildAttrAssocImpl(const String& childAttrKey);
    ChildAttrAssocImpl(const String& childAttrNumKeys,
		       const String& childAttrKey);
    ~ChildAttrAssocImpl();
    String _childAttrNumKeys;
    String _childAttrKey;
    bool _oneToOneAssoc;
};

//
//  ChildAttrAssocImpl::ChildAttrAssocImpl(const String& childAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//	childAttrNumKeys  The attribute of the child which holds the
//	                  number of parent items.
//      childAttrKey      The attribute of the child which holds the
//                        selector(s) of the parent items.
//
ChildAttrAssocImpl::ChildAttrAssocImpl(const String& childAttrNumKeys,
				       const String& childAttrKey)
 : _childAttrNumKeys(childAttrNumKeys), _childAttrKey(childAttrKey), 
   _oneToOneAssoc(false) {
    Log::debug(ChildAttrAssoc::NAME, "oneToMany");
}

//
//  ChildAttrAssocImpl::ChildAttrAssocImpl(const String& childAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      childAttrKey The attribute of the child which holds the
//      selector(s) of the parent items.
//
ChildAttrAssocImpl::ChildAttrAssocImpl(const String& childAttrKey)
 : _childAttrKey(childAttrKey), _oneToOneAssoc(true) {
     Log::debug(ChildAttrAssoc::NAME, "oneToOne");
}

//
//  ChildAttrAssocImpl::~ChildAttrAssocImpl()
//
//  Description:
//      Destructor
//
ChildAttrAssocImpl::~ChildAttrAssocImpl() {
}


//
//  ChildAttrAssoc::ChildAttrAssoc(Category& parentCategory, 
//  			           const String& parentSelector,
//  			           Category& childCategory,
//				   const String& childAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentCategory Category of the parent item.
//      parentSelector Selector of the parent item.
//      childCategory Category of the child item(s).
// 	childAttrKey The attribute key of the child pointing to
//      	     the parent item(s)
//
ChildAttrAssoc::ChildAttrAssoc(Category& parentCategory, 
			       const String& parentSelector, 
			       Category& childCategory,
			       const String& childAttrKey)
 : ComputedAssoc(parentCategory, parentSelector, childCategory),
   _impl(new ChildAttrAssocImpl(childAttrKey)){
}

//
//  ChildAttrAssoc::ChildAttrAssoc(Category& parentCategory, 
//	                           const String& parentSelector, 
//  				   Category& childCategory, 
//  				   const String& childAttrNumKeys,
//  				   const String& childAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentCategory     The Category of the parent Item.
//      parentSelector     The selector of the parent Item.
//      childCategory      The Category of the related Item(s).
//      childAttrNumKeys   The attribute key containing the number of
//                         parent items.
//      childAttrKey       The attribute key of the child pointing to
//      	           the parent item(s)
//
ChildAttrAssoc::ChildAttrAssoc(Category& parentCategory, 
			       const String& parentSelector, 
			       Category& childCategory,
			       const String& childAttrNumKeys,
			       const String& childAttrKey)
 : ComputedAssoc(parentCategory, parentSelector, childCategory),
   _impl(new ChildAttrAssocImpl(childAttrNumKeys, childAttrKey)) {
    Log::debug(NAME, "ChildAttrAssoc constructor for selector %s", 
	       (const char*) getSelector());
}

//
//  ChildAttrAssoc::~ChildAttrAssoc() 
//
//  Description:
//      Destructor
//
ChildAttrAssoc::~ChildAttrAssoc() {
}


//
//  void ChildAttrAssoc::parentChanged(const Item& oldItem, 
//  			      const Item& newItem)
//
//  Description:
//      Called by ParentCategoryListener when the parent item
//	changes. Turn off behavior of ComputedAssoc.
//
//  Parameters:
//      oldItem The old item.
//      newItem The new item.
//
void ChildAttrAssoc::parentChanged(const Item& oldItem, 
			      const Item& newItem) {
    Association::parentChanged(oldItem, newItem);
}

//
//  bool ChildAttrAssoc::isChild(const Item& potentialChildItem)
//
//  Description:
//      Check if an item is a child of the parent item.
//
//  Parameters:
//      potentialChildItem An item of the child category.
//
//  Returns:
//	true if the item is a child of the parent item. false otherwise.
//
bool ChildAttrAssoc::isChild(const Item& potentialChildItem) {
    Log::debug(NAME, "isChild");
    const Item* parentItem = getParent();
    bool result = false;
    DictionaryOf<String>* selectors;

    if (_impl->_oneToOneAssoc) {  
	Log::debug(NAME, "oneToOne relationship");
	Attribute attr = potentialChildItem.getAttr(_impl->_childAttrKey);
	if (attr != Attribute::NUL) {
	    result = (strcmp(parentItem->getSelector(), 
			     attr.stringValue()) == 0);
	    return result;
	} else {
	    Log::debug(NAME, "No attribute with key: %s", 
		       (const char*) _impl->_childAttrKey);
	    return false;
	}
    } 

    Log::debug(NAME, "oneToMany relationship");
    selectors = potentialChildItem.createStringSequence(
	_impl->_childAttrNumKeys,_impl->_childAttrKey);

    result = (selectors->lookupByKey(parentItem->getSelector()) !=
	      NULL);
    Item::destroyStringSequence(selectors);
    return result;
}

END_NAMESPACE(sysadm);
