//
// ChildAttrAssoc.h
//
//	Base class for deriving classes to represent relationships in
//	which the child Item(s) store the selectors of one or more
//	parent Item(s) as part of its Attributes.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.7 $"
#ifndef _SYSADM_CHILD_ATTR_ASSOC_H
#define _SYSADM_CHILD_ATTR_ASSOC_H

#include <sysadm/ComputedAssoc.h>

BEGIN_NAMESPACE(sysadm);

//
// Base class for deriving classes to represent relationships in
// which the child Item(s) store the selectors of one or more
// parent Item(s) as part of its Attributes.
//
// Subclasses provide the Attribute's key in the constructors.  The
// ParentAttrAssoc class monitors the parent and child Item(s), keeps
// the list of child Item(s) current and notifies listeners of changes
// to the list of child Item(s).
//
// ParentAttrAssoc can determine the Item(s) of child category that
// belong to this Association in two ways depending on the constructor
// that is used.  One constructor is used when there is a 1-to-1 relationship
// from a child Item to Item(s) of the parent Category. Subclasses
// specify the attribute key of the child Item that holds the parent
// Item selector.
// Another constructor is used when there is a 1-to-n relationship
// from a child Item to Item(s) of the parent Category.  This is based
// on the format for arrays of values specified in Item.h.
// Subclasses specify the value of the key of the child Item that
// holds the number of parent Item selectors and the base name of the
// attribute keys which hold the selectors themselves.
//
// For example, if ClustersAssocWithMachine is a relationship where
// an Item of type ClusterCategory has an array of selectors of
// Item(s) of MachineCategory, NUM_MACHINES could be the key of
// ClusterCategory Item(s) that holds number of values in the array.
// The base name of the attribute keys could be MACHINE.  
// ParentAttrAssoc will monitor Item(s) corresponding to the list of
// selectors in Attributes of the child ClusterCategory Item(s) with
// keys CLUSTER0, CLUSTER1, ... CLUSTER<NUM_CLUSTERS - 1>.
//
// ChildAttrAssoc keeps the list of child Item(s) up-to-date based on
// parent/child changes.
//
// Typically, subclasses only need to call the ChildAttrAssoc
// constructor with the application specific values of the Attribute
// keys and do not need to override any methods.
//
class ChildAttrAssoc : public ComputedAssoc {
  public:
    // For logging.
    static const char* NAME;

    // Constructor.  This version is used for 1-to-n relationships and
    // takes two attribute names.
    // "childAttrNumKeys" is the attribute key which holds the number of
    // parent Item selectors, and "childAttrKey" is the base name of
    // the attribute keys which hold the selectors themselves.
    ChildAttrAssoc(Category& parentCategory,
		   const String& parentSelector,
		   Category& childCategory,
		   const String& childAttrNumKeys,
		   const String& childAttrKey);

    // Constructor.  This version is used for 1-to-1 relationships and
    // takes one attribute name which is the attribute key holding the
    // parent Item selector.
    ChildAttrAssoc(Category& parentCategory, 
		   const String& parentSelector,
		   Category& childCategory, 
		   const String& childAttrKey);

    // Destructor
    virtual ~ChildAttrAssoc();

    // Called when parentItem is changed
    // Overrides ComputedAssoc behavior to do nothing on parentChanged()
    virtual void parentChanged(const Item& oldItem, const Item& newItem);

    // Returns true if the selector of the parent Item exists in
    // Attributes with key(s) holding the selectors of the parent
    // Item(s).
    virtual bool isChild(const Item& potentialChildItem);

  private:

    // Intentionally undefined.	
    ChildAttrAssoc(const ChildAttrAssoc&);
    ChildAttrAssoc& operator=(const ChildAttrAssoc&);
    
    // Opaque implementation.
    friend class ChildAttrAssocImpl;
    ChildAttrAssocImpl* _impl;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_CHILD_ATTR_ASSOC_H
