//
// ChildCategoryListener.c++
//
//	Class for monitoring changes in child item(s) for a ComputedAssoc.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#include "ChildCategoryListener.h"
#include <sysadm/Log.h>
#include <sysadm/format.h>

#include <stdio.h>
#include <limits.h>
#include <dlfcn.h>

BEGIN_NAMESPACE(sysadm);

const char* ChildCategoryListener::NAME = "ChildCategoryListener";

//
//  ChildCategoryListener::ChildCategoryListener(Category& childCategory,
//  					         NotificationFilter* filter,
//  					         ComputedAssoc& assoc)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      childCategory The Category of the related Item(s).
//      filter Filter to be associated with this listener.
//      assoc Association to forward notifications to.
//
ChildCategoryListener::ChildCategoryListener(Category& childCategory,
					     NotificationFilter* filter,
					     ComputedAssoc& assoc)
 : _assoc(assoc), _cat(childCategory), _receivingNewList(false),
   _skipEndBlockChanges(0) {
    _handle = _cat.adoptCategoryListener(this, *filter);
    delete filter;
}

//
//  ChildCategoryListener::~ChildCategoryListener()
//
//  Description:
//      Destructor.
//
ChildCategoryListener::~ChildCategoryListener() {
    _cat.orphanCategoryListener(_handle);
    if (_receivingNewList) {
	RemoveAndDestroyContentsOf(&_newList);
    }
}

//
//  void ChildCategoryListener::itemAdded(const Item& item)
//
//  Description:
//      Called when an item of child category has been added or 
//	detected at startup.
//	Forwards notification to the association immediately
//	or puts them into a list to be sent as a batch if these
//	notifications are a result of changing the notification filter.
//
//  Parameters:
//      item The item that was added.
//
void ChildCategoryListener::itemAdded(const Item& item) {
    if (_receivingNewList && _assoc.isChild(item)) {
	Item* itemClone = dynamic_cast<Item*>(item.clone());
	assert(itemClone != NULL);
	_newList.append(itemClone);
    } else {
	_assoc.childCategoryItemAdded(item);
    }
}


//
//  void ChildCategoryListener::itemChanged(const Item& oldItem, 
//  					const Item& newItem
//
//  Description:
//      Called when a child category item changes.
//	Forwards notification to association.
//
//  Parameters:
//      oldItem The old item.
//      newItem The new item.
//
void ChildCategoryListener::itemChanged(const Item& oldItem, 
					const Item& newItem) {
    assert(!_receivingNewList);
    _assoc.childCategoryItemChanged(oldItem, newItem);
}

//
//  void ChildCategoryListener::itemRemoved(const Item& item)
//
//  Description:
//	Called when a child category item is removed.
//      Forwards notification to association.
//
//  Parameters:
//      item The item that was removed.
//
void ChildCategoryListener::itemRemoved(const Item& item) {
    assert(!_receivingNewList);
    _assoc.childCategoryItemRemoved(item.getSelector());
}

//
//  void ChildCategoryListener::beginBlockChanges() 
//
//  Description:
//	Called by Category prior to a block of changes. 
//	Forward to association unless we are receiving notifications
//	because of changing the notification filter in which case the 
//	batch send that results will send a beginBlockChanges. Also,
//	skip the endBlockChanges call corresponding to this
//	beginBlockChanges. 
//
void ChildCategoryListener::beginBlockChanges() {
    if (!_receivingNewList) {
	_assoc.beginBlockChanges();
    } else {
	_skipEndBlockChanges++;
    }
}

//
//  void ChildCategoryListener::endBlockChanges()
//	
//  Description:
//      Called by Category at the end of a block of changes. 
//	Forward to association unless the corresponding
//	beginBlockChanges was not forwarded.
//
void ChildCategoryListener::endBlockChanges() {
    if (_skipEndBlockChanges == 0) {
	_assoc.endBlockChanges();
    } else {
	_skipEndBlockChanges--;
	assert (_skipEndBlockChanges >= 0);
    }
}

//
//  void ChildCategoryListener::endExists()
//
//  Description:
//      Called by Category after notification of current state.
//	Forward to association unless this marked the end of
//	notifications caused by changing the notification filter.
//	In that case, send the batch notifications via a
//	replaceItemList call.
//
void ChildCategoryListener::endExists() {
    if (_receivingNewList) {
	_assoc.replaceItemList(_newList);
	RemoveAndDestroyContentsOf(&_newList);
	_receivingNewList = false;
    } else {
	// I could have determined whether an endExists was
	// previously received by calling _assoc.hasExistsEnded()
	// But because of a number of bugs reported that relate
	// to two endExists, I wanted to make sure that the 
	// second endExists was due to causes that I know about
	// - in this case because of a parent Item that
	// previously existed.
	if (!_assoc.hasParentExistedPreviously()) {
	    _assoc.endExists();
	}
    }
}

// Do nothing on receiving attrChanged
void ChildCategoryListener::attrChanged(const AttrEvent& /*event*/) {
}

//
//  void ChildCategoryListener::adoptAndReplaceNotificationFilter(
//	NotificationFilter* filter)
//
//  Description:
//      Recomputes the list of children based on the child items that
// 	match filter. Changes filter associated with the listener on 
// 	child category items.
//
//  Parameters:
//      filter The new filter.
//
void ChildCategoryListener::adoptAndReplaceNotificationFilter(
    NotificationFilter* filter) {

    while (_assoc.isInBlockChangesNotification()) {
	_assoc.endBlockChanges();
    }

    if (!_assoc.hasExistsEnded()) {
	_assoc.endExists();
    }

    // Needs to come first before the adoptNotificationFilter call
    RemoveAndDestroyContentsOf(&_newList);
    _receivingNewList = true;
    _skipEndBlockChanges = 0;

    _cat.orphanCategoryListener(_handle);
    _handle = _cat.adoptCategoryListener(this, *filter);
    delete filter;
}

END_NAMESPACE(sysadm);
