//
// ComputedAssoc.c++
//
//	Base class for deriving classes to represent relationships governed
//	by attributes of the monitored parent and child items.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.5 $"

#include <sysadm/ComputedAssoc.h>
#include "ChildCategoryListener.h"
#include <sysadm/Log.h>
#include <string.h>

BEGIN_NAMESPACE(sysadm);

const char* ComputedAssoc::NAME = "ComputedAssoc";

//
// ComputedAssoc implementation.
//
class ComputedAssocImpl {
  public:
    ComputedAssocImpl(Category& childCategory);
    ~ComputedAssocImpl();
    bool exists(const ItemList* items, const String& selector);

    Category& _childCategory;
    ChildCategoryListener* _childListener;
    bool _parentRemoved;
};

//
//  ComputedAssocImpl::ComputedAssocImpl(Category& childCategory)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      childCategory The Category of the related Item(s).
//
ComputedAssocImpl::ComputedAssocImpl(Category& childCategory)
 : _childCategory(childCategory), _childListener(NULL), _parentRemoved(false) {
}


//
//  ComputedAssocImpl::~ComputedAssocImpl()
//
//  Description:
//	Destructor.      
//
ComputedAssocImpl::~ComputedAssocImpl() {
    delete _childListener;
}

//
//  bool ComputedAssocImpl::exists(const ItemList* items, 
//  			           const String& selector)
//
//  Description:
//      Helper function to determine if an item named by selector
//	exists in a sequence of selectors.
//
//  Parameters:
//      items The list of Items.
//      selector The selector of the Item whose existence in the list
//               of Items is to be determined.
//
//  Returns:
//	true, if "selector" exists in "items", false otherwise
//
bool ComputedAssocImpl::exists(const ItemList* items, 
			       const String& selector) {
    ConstIteratorOver<Item> iter(items);
    Item* item;
    while ((item = iter()) != NULL) {
	if (strcmp(item->getSelector(), selector) == 0) {
	    return true;
	}
    }
    return false;
}

//
//  ComputedAssoc::ComputedAssoc(Category& parentCategory, 
//  			         const String& parentSelector,
//  			         const String& childCategory)
//
//  Description:
//      Constructor
//
//  Parameters:
//      parentCategory The Category of the parent Item.
//      parentSelector The selector of the parent Item.
//      childCategory The Category of the related Item(s).
//
ComputedAssoc::ComputedAssoc(Category& parentCategory, 
			     const String& parentSelector,
			     Category& childCategory)
 : Association(parentCategory, parentSelector, childCategory), 
   _impl(new ComputedAssocImpl(childCategory)) {
    Log::debug(NAME, "ComputedAssoc constructor for selector %s", 
	       (const char*) getSelector());
}

//
//  ComputedAssoc::~ComputedAssoc()
//
//  Description:
//      Destructor
//
ComputedAssoc::~ComputedAssoc() {
    delete _impl;
}

//
//  void ComputedAssoc::parentAdded(const Item& item)
//
//  Description:
//      Called by ParentCategoryListener when the parent item
//	is added or detected at startup.
//	Starts the monitoring of item(s) of the child category.
//
//  Parameters:
//      item The item that was added.
//
void ComputedAssoc::parentAdded(const Item& item) {
    Association::parentAdded(item);
    NotificationFilter* filter = createAddedChildNotificationFilter(item);
    _impl->_childListener = 
       new ChildCategoryListener(_impl->_childCategory, filter, *this);
}


//
//  NotificationFilter* 
//	ComputedAssoc::createAddedChildNotificationFilter(const Item& 
//	                                                  parentItem)
//
//  Description:
//      Returns a filter for monitoring all items 
// 	to be associated with the listener on 
// 	child category items when monitoring is started.
//
//  Parameters:
//      parentItem The parent Item that was added.
//
//  Returns:
//	filter to be associated with the listener on child category item.
//
NotificationFilter* 
    ComputedAssoc::createAddedChildNotificationFilter(const Item& 
                                                      /*parentItem*/) {
    NotificationFilter* filter = new NotificationFilter();
    filter->monitorAllItems();
    return filter;
}

//
//  void ComputedAssoc::parentChanged(const Item& oldItem, 
//  				      const Item& newItem)
//
//  Description:
//      Called by ParentCategoryListener when the parent item
//	changes. Get the current list of child items and check if 
// 	any child items need to be added/removed from the list 
//	based on the computation performed by isChild().
//
//  Parameters:
//      oldItem The old parent Item.
//      newItem The new parent Item.
//
void ComputedAssoc::parentChanged(const Item& oldItem, 
				  const Item& newItem) {
    Association::parentChanged(oldItem, newItem);

    // re-registers with the child category for the current list of
    // children
    NotificationFilter* filter =
        createChangedChildNotificationFilter(oldItem, newItem);
    if (filter != NULL)
	adoptAndReplaceChildNotificationFilter(filter);
}

//
//  NotificationFilter* 
//  ComputedAssoc::createChangedChildNotificationFilter(const Item& oldItem,
//                                                      const Item& newItem)
//
//  Description:
//      Returns a filter for monitoring all items 
// 	to be associated with the listener on 
// 	child category items when it is detected that the parent Item changed.
//
//  Parameters:
//      oldItem The old parent Item.
//      newItem The new parent Item.
//
//  Returns:
//	filter to be associated with the listener on child category item.
//
NotificationFilter* 
ComputedAssoc::createChangedChildNotificationFilter(const Item& /*oldItem*/,
						    const Item& newItem) {
    return createAddedChildNotificationFilter(newItem);
}

//
//  void ComputedAssoc::parentRemoved()
//
//  Description:
//      Called when parentItem is removed. 
// 	Sends itemRemoved messages for each child item. 
// 	Stops listening for items of child category.
//
void ComputedAssoc::parentRemoved() {
    Association::parentRemoved();
    delete _impl->_childListener;
    _impl->_childListener = NULL;
    _impl->_parentRemoved = true;
}

bool ComputedAssoc::hasParentExistedPreviously() {
    return _impl->_parentRemoved;
}

//
//  void ComputedAssoc::childCategoryItemAdded(const Item& item)
//
//  Description:
//      Called when an item of child category is added or detected at startup.
//	Calls isChild() and if it returns true, adds the item to our list.
//
//  Parameters:
//      item Item of child category that is added or detected at startup.
//
void ComputedAssoc::childCategoryItemAdded(const Item& item) {
    if (isChild(item)) {
	addItem(item);
    }	
}

//
//  void ComputedAssoc::childCategoryItemChanged(const Item& oldItem, 
//  					         const Item& newItem)
//
//  Description:
//      Called when an item of child category is changed.
//    	Calls isChild() and if it returns true, 
// 	we check if oldItem was previously in the list of
// 	children. If it was not, we add "newItem" to our list. 
// 	If it was, we change "oldItem" in our list.
// 	If isChild() returns false, we check if "oldItem" was
// 	previously in the list of children and if it exists, 
// 	we remove "oldItem" from our list.
//
//  Parameters:
//      oldItem The old item.
//      newItem The new item.
//
void ComputedAssoc::childCategoryItemChanged(const Item& oldItem, 
					     const Item& newItem) {    
    const ItemList* items = getItems();
    bool wasChild = _impl->exists(items, oldItem.getSelector());
    if (isChild(newItem)) {
	if (wasChild) {
	    changeItem(newItem);
	} else {
	    addItem(newItem);
	}
    } else {
	if (wasChild) {
	    removeItem(oldItem.getSelector());
	}
    }
}

//
//  void ComputedAssoc::childCategoryItemRemoved(const String& selector)
//
//  Description:
//      Called when an item of child category is removed.
// 	Removes the item corresponding to "selector" from our list, 
// 	if it is present.
//
//  Parameters:
//      selector The name of the item that was removed.
//
void ComputedAssoc::childCategoryItemRemoved(const String& selector) {
    removeItem(selector);
}

//
//  void ComputedAssoc::adoptAndReplaceChildNotificationFilter(
//      NotificationFilter* filter)
//
//  Description:
//      Recomputes the list of children based on the child items that
// 	match filter. Changes filter associated with the listener on 
// 	child category items.
//
//  Parameters:
//      filter The new filter.
//
void ComputedAssoc::adoptAndReplaceChildNotificationFilter(NotificationFilter*
							   filter) {
    assert(getParent() != NULL);
     _impl->_childListener->adoptAndReplaceNotificationFilter(filter);
}

END_NAMESPACE(sysadm);
