//
// ComputedAssoc.h
//
//	Base class for deriving classes to represent relationships
//	that can be computed from values of attributes of the monitored
//	parent and child Item(s).
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.6 $"
#ifndef _SYSADM_COMPUTED_ASSOC_H
#define _SYSADM_COMPUTED_ASSOC_H

#include <sysadm/Association.h>

BEGIN_NAMESPACE(sysadm);

//
// Base class for deriving classes to represent relationships 
// that can be computed from values of attributes of the monitored
// parent and child Item(s).  This class monitors the parent Item and
// Item(s) in the child Category that are potential children.  The
// Item(s) that are potential children are indicated by a
// NotificationFilter that comes in effect when the parent Item is
// obtained.  The NotificationFilter can be changed when the parent Item
// changes or at any arbitrary time.
//
// Subclasses must override the isChild() method to provide logic that
// determines if the Item of the child Category passed to isChild() is
// a child of the parent Item.
//
// This class calls isChild() anytime there is a change to the parent
// or the monitored child Item(s) and forwards notifications about
// changes in its list of child Item(s) to the listeners on this
// Association.  When the parent Item changes, it gets the
// current list of child items and checks if any child Item(s) need to
// be added/removed from its list based on the computation performed
// by isChild(). When an Item of the child Category is added, removed or
// changed, it checks if the Item should be added/removed/updated in
// its list of child Item(s) based on isChild().
//
// Sub-classes can fine-tune this behaviour by overriding the 
// methods that are called upon notifications related to the parent
// and child Item(s).  For example, subclass ChildAttrAssoc overrides
// parentChanged() to turn off computation of the list of child Item(s) when
// the parent Item changes and only does so on changes in the Item(s) of
// the child category.
//

class ComputedAssoc : public Association {
  public:
    // Used internally for logging.
    static const char* NAME;

  protected:
    // Constructor. 
    ComputedAssoc(Category& parentCategory, 
		  const String& parentSelector,
		  Category& childCategory);

    // Destructor.
    virtual ~ComputedAssoc();

    // Called when an Item in the child category is
    // added/changed.  Also called when the parent Item changes.
    // Subclasses should override this and return true if
    // "potentialChildItem" is a child of the parent Item of this
    // Association.  False otherwise.
    virtual bool isChild(const Item& potentialChildItem) = 0;

    // In addition to the semantics provided by
    // Association::parentAdded it starts monitoring of Item(s) of the
    // child category.  Calls createAddedChildNotificationFilter() to
    // determine the Items of child category that must be monitored.
    virtual void parentAdded(const Item& item);

    // Called by parentAdded() to determine the Item(s) of child
    // category that must be monitored.
    // Defined to return a filter that specifies monitoring of all
    // Item(s) of the child category.
    // Subclasses can override to provide different behavior. 
    // The return value cannot be NULL. The ComputedAssoc class owns the
    // memory returned.
    virtual NotificationFilter* 
        createAddedChildNotificationFilter(const Item& parentItem);

    // Called when parent Item changes.  Gets the current list of
    // child Item(s) and checks if any child Item(s) need to be
    // added/removed from its list by calling isChild() for each Item.
    // Forwards notifications of changes to the registered listeners.
    //
    // Calls createChangedChildNotificationFilter() to determine the
    // Items of child category that must be monitored based on the
    // changes to the parent Item.  If that returns NULL, no change is
    // made to the list of Items of child category that are being
    // monitored (since the most recent call to
    // createAddedChildNotificationFilter() or
    // createChangedChildNotificationFilter() or
    // adoptAndReplaceChildNotificationFilter())
    // If that does not return NULL, recomputes the list of items
    // from the universe of child Item(s) that match the filter
    // specified in the return value.
    virtual void parentChanged(const Item& oldItem, const Item& newItem);

    // Called by parentChanged() to determine the Item(s) of child
    // category that must be monitored.  A return value of NULL
    // indicates no change is required to the list of Items of child
    // category that are being monitored (since the most recent call to
    // createAddedChildNotificationFilter() or
    // createChangedChildNotificationFilter() or
    // adoptAndReplaceChildNotificationFilter()).
    // Defined to return a filter that specifies the monitoring of all
    // items of the child category.
    // Subclasses can override to provide different behavior.
    // The ComputedAssoc class owns the memory returned.
    virtual NotificationFilter* 
        createChangedChildNotificationFilter(const Item& oldItem, 
					     const Item& newItem);
    
    // Provides the semantics of Association::parentRemoved().
    virtual void parentRemoved();

    // Returns true if the parent Item existed and was removed during
    // the lifecycle of this Association instance.  Returns false, if
    // the parent Item never existed during the lifecycle of this
    // Association instance.
    virtual bool hasParentExistedPreviously();

    // Called when a monitored Item of child category is determined to exist in
    // the system at the time monitoring is started and if it is added
    // later to the system.
    // Calls isChild() and if it returns true, adds "item" to the list
    // of child Item(s).  Listeners are notified of any changes.
    virtual void childCategoryItemAdded(const Item& item);

    // Called when a monitored Item of child category changes to update the
    // list of child Item(s).
    // Calls isChild() and if it returns true, 
    // checks if "oldItem" was previously in the list.  If it was not,
    // "newItem" is added.  If it was, "oldItem" is replaced by "newItem".
    // If isChild() returns false, checks if "oldItem" was
    // previously in the list and if so, removes "oldItem".
    // Listeners are notified of any changes.
    virtual void childCategoryItemChanged(const Item& oldItem, 
					  const Item& newItem);

    // Called when a monitored Item of child category is removed.
    // Removes the Item corresponding to "selector" from our list of
    // child Item(s), if it is present.
    virtual void childCategoryItemRemoved(const String& selector);

    // This is a hook that can be called by subclasses at any time to
    // change the list of child Item(s) that are currently being
    // monitored.  Recomputes the list of children from the universe
    // of child Item(s) that match filter.
    virtual void adoptAndReplaceChildNotificationFilter(NotificationFilter* 
							filter);

    // Allow ChildCategoryListener to invoke our protected methods.
    friend class ChildCategoryListener;

  private:

    // Intentionally undefined.	
    ComputedAssoc(const ComputedAssoc&);
    ComputedAssoc& operator=(const ComputedAssoc&);
    
    // Opaque implementation.
    friend class ComputedAssocImpl;
    ComputedAssocImpl* _impl;

};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_COMPUTED_ASSOC_H
