//
// Item.h
//
//	A monitored object within a Category.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.32 $"
#ifndef _SYSADM_ITEM_H
#define _SYSADM_ITEM_H

#include <sysadm/AttrBundle.h>

BEGIN_NAMESPACE(sysadm);

//
// Represents a physical or logical entity that is manipulated by
// system administration operations. An Item is a subclass of
// AttrBundle and has an aggregation of typed key value pairs.  Each
// Item has an associated type and a unique name within that type. For
// example, a user account can have the unique name "foo" within the
// type "user account" and the following Attributes:
//	string, userName, foo
//	long, userId, 3944
//
// To take advantage of the Association mechanism which can monitor
// the relationship between Item(s) of the same or different Category,
// arrays of parameters, specifically those referring to the
// "selectors" of Item(s) should use the following format:
// 1.  One attribute specifies the number of values in the array.
// 2.  Each value is accessed by appending a number to a prefix for
//     the series of values.
//
// For example, if an Item of type group account has an array of
// Strings representing names of users that use its group id, this
// would by represented by an attribute, for example, NUM_USERS that
// specifies the number of values in the array and an attribute, for
// example, USER as the prefix to use.  USER0, USER1
// ... USER<NUM_USERS - 1> would be the keys of the actual String values.
//
// Item(s) of a specific type are aggregated in an instance of a
// subclass of Category. The type of the Item corresponds to the
// selector of the Category.  For example, the collection of user
// account Item(s) can be represented by a Category instance. 
// Subclasses of Category use Item to inform Category base class of
// the current state of the system and of any changes.  The Category
// base class in turn notifies registered CategoryListener instances.
//
// See Category.h for more details on the use of Item(s).
//
class Item : public AttrBundle {
    
  public:

    // The "type" of an Item instance is the name (selector) of the
    // Category instance that it belongs to. The "selector" of an Item
    // instance is the unique name of the Item instance within the Category
    // instance.  Category will assert if Item(s) with non-unique
    // named are added via Category::addItem().
    Item(const String& type, const String& selector);

    // Construct an Item that was serialized by a previous call
    // to Item::serialize().
    Item(const String& stream);

    // Destructor.
    virtual ~Item();
    
    // Create a new Item which is a copy of this one.  The caller owns
    // the memory returned.
    virtual AttrBundle* clone() const;

    // Compare "this" with "other".  Return < 0 if "this" < "other", 0
    // if "this" == "other", and > 0 if "this > "other".  The base class
    // method returns strcmp(this->getSelector(), other.getSelector()).
    virtual int compare(const Item& other) const;

    // Obtain a dictionary of strings corresponding attributes
    // specifying the number of values in the String sequence and the base
    // prefix used for each element of the String sequence.  The return value
    // should be destroyed by calling the static method destroyStringSequence.
    virtual DictionaryOf<String>* createStringSequence(
	const String& numKeysAttr, const String& keyAttrBase) const;

    // Free the memory associatied with the dictionary of strings previously
    // obtained via createStringSequence.
    static void destroyStringSequence(DictionaryOf<String>* seq);

};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_ITEM_H
