//
// NotificationFilter.c++
//
//	Class representing the changes that a CategoryListener is monitoring
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.11 $"

#include <sysadm/NotificationFilter.h>
#include <sysadm/DictionaryOf.h>
#include <assert.h>

BEGIN_NAMESPACE(sysadm);

// 
// NotificationFilter implementation.
//
class NotificationFilterImpl {
  
  public:
    NotificationFilterImpl() : _allItems(false),
        _allCategoryAttrs(false) {
    }

    ~NotificationFilterImpl();

    void removeList();

    bool _allItems;
    bool _allCategoryAttrs;

    DictionaryOf<String> _selectorList;
};

//
// Destructor
//
NotificationFilterImpl::~NotificationFilterImpl() {
}

//
// Remove the list of items from NotificationFilter.
//
void NotificationFilterImpl::removeList() {
    DictionaryOfIterator<String> iter(&_selectorList);
    while (iter() != NULL) {
	iter.remove();
    }
}

//
// Constructor.
//
NotificationFilter::NotificationFilter() : 
    _impl(new NotificationFilterImpl) {
}

//
// Copy Constructor.
//
NotificationFilter::NotificationFilter(const NotificationFilter& filter) 
: _impl(new NotificationFilterImpl) {
    _impl->_allItems = filter._impl->_allItems;
    _impl->_allCategoryAttrs = filter._impl->_allCategoryAttrs;
    DictionaryOfIterator<String> iter(&filter._impl->_selectorList);
    String* selector;
    while ((selector = iter()) != NULL) {
	String* selectorClone = new String(*selector);
	_impl->_selectorList.add(selectorClone, selectorClone);
    }
}

//
// Destructor.
//
NotificationFilter::~NotificationFilter() {
    delete _impl;
}

//
//  void NotificationFilter::monitorItem(const String& selector)
//
//  Description:
//	Register interest in Item 
//
//  Parameters:
//	selector Name of Item
void NotificationFilter::monitorItem(const String& selector) {
    _impl->_allItems = false;

    String* selectorClone = new String(selector);
    _impl->_selectorList.add(selectorClone, selectorClone);
}

//
//  void NotificationFilter::unmonitorItem(const String& selector)
//
//  Description:
//	Unregister interest in Item 
//
//  Parameters:
//	selector Name of Item
void NotificationFilter::unmonitorItem(const String& selector) {
    assert(_impl->_allItems == false);
    _impl->_selectorList.remove(selector);
}

//
//  void NotificationFilter::monitorAllItems()
//
//  Description:
//	Register interest in all Items in Category.
void NotificationFilter::monitorAllItems() {
    _impl->_allItems = true;

    // Clear _selectorList
    _impl->removeList();
}

//
//  Description:
//	Unregister interest in all Items in Category.
void NotificationFilter::unmonitorAllItems() {
    _impl->_allItems = false;

    // Clear _selectorList
    _impl->removeList();
}


//
//  void NotificationFilter::monitorCategoryAttrs()
//  Description:
//	Register interest in Category attributes
//
void NotificationFilter::monitorCategoryAttrs() {
    _impl->_allCategoryAttrs = true;	
}

//
//  void NotificationFilter::unmonitorCategoryAttrs()
//  Description:
//	Unregister interest in Category attributes
//
void NotificationFilter::unmonitorCategoryAttrs() {
    _impl->_allCategoryAttrs = false;	
}

//
//  bool NotificationFilter::monitorsItem
//
//  Description:
//	Determine if the filter monitors specified item
//
//  Params:
//	selector	Name of item
//
//  Returns:
//	true if filter monitors selector, false otherwise
bool NotificationFilter::monitorsItem(const String& selector) {
    if (_impl->_allItems)
	return true;

    const String* str = _impl->_selectorList.lookupByKey(selector);
    return (str != NULL);
}

//
//  bool NotificationFilter::monitorsCategoryAttrs()
//
//  Description:
//	Determine if the filter monitors category attributes
//
//  Returns:
//	true if filter monitors category, false otherwise
bool NotificationFilter::monitorsCategoryAttrs() {
    return _impl->_allCategoryAttrs;
}

//
//  NotificationFilter* NotificationFilter::clone() const
//
//  Description:
//      Create a copy of this NotificationFilter.
//
//  Returns:
//	A copy of this NotificationFilter.
//
NotificationFilter* NotificationFilter::clone() const
{
    return new NotificationFilter(*this);
}

END_NAMESPACE(sysadm);

