//
// ParentAttrAssoc.c++
//
//	Base class for deriving classes to represent relationships governed
//	by an attribute of the monitored parent item.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.7 $"

#include <sysadm/ParentAttrAssoc.h>
#include <sysadm/Log.h>
#include <string.h>

BEGIN_NAMESPACE(sysadm);

const char* ParentAttrAssoc::NAME = "ParentAttrAssoc";

//
// ParentAttrAssoc implementation.
//
class ParentAttrAssocImpl {
  public:
    ParentAttrAssocImpl(const String& parentAttrKey);
    ParentAttrAssocImpl(const String& parentAttrNumKeys,
			const String& parentAttrKey);
    ~ParentAttrAssocImpl();
    String _parentAttrNumKeys;
    String _parentAttr;
    DictionaryOf<String>* _selectors;
    bool _oneToOneAssoc;

    bool selectorsChanged(const DictionaryOf<String>& newSelectors);
};

//
//  ParentAttrAssocImpl::ParentAttrAssocImpl(const String& parentAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//	parentAttrNumKeys  The attribute of the parent which holds the
//	                   number of child items.
//      parentAttrKey      The attribute of the parent which holds the
//                         selector(s) of the child items.
//
ParentAttrAssocImpl::ParentAttrAssocImpl(const String& parentAttrNumKeys,
					 const String& parentAttrKey)
 : _parentAttrNumKeys(parentAttrNumKeys), _parentAttr(parentAttrKey),
   _oneToOneAssoc(false), _selectors(NULL) {
}

//
//  ParentAttrAssocImpl::ParentAttrAssocImpl(const String& parentAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentAttrKey The attribute of the parent which holds the
//                    selector(s) of the child items.
//
ParentAttrAssocImpl::ParentAttrAssocImpl(const String& parentAttrKey)
 : _parentAttr(parentAttrKey),
   _oneToOneAssoc(true), _selectors(NULL) {
}

//
//  ParentAttrAssocImpl::~ParentAttrAssocImpl()
//
//  Description:
//      Destructor.
//
ParentAttrAssocImpl::~ParentAttrAssocImpl() {
    if (_selectors != NULL) {
	Item::destroyStringSequence(_selectors);
    }
}

//
//  bool ParentAttrAssocImpl::selectorsChanged(const DictionaryOf<String>& 
//  					       newSelectors)
//
//  Description:
//      Helper function to calculate if the selectors have changed.
//	Change in order of selectors is ignored.
//
//  Parameters:
//      newSelectors New list of selectors.
//
//  Returns:
//	true if the selectors have changed, false otherwise.
//
bool ParentAttrAssocImpl::selectorsChanged(const DictionaryOf<String>& 
					   newSelectors) {
    if (_selectors->getSize() != newSelectors.getSize()) {
	return true;
    } else {
	DictionaryOfIterator<String> iter(_selectors);
	String* selector;
	while ((selector = iter()) != NULL) {
	    if (newSelectors.lookupByKey(*selector) == NULL) {
		return true;
	    }
	}
    }
    return false;
}

//
//  ParentAttrAssoc::ParentAttrAssoc(Categpry& parentCategory, 
//	                             const String& parentSelector, 
//  				     Category& childCategory, 
//  				     const String& parentAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentCategory The Category of the parent Item.
//      parentSelector The selector of the parent Item.
//      childCategory The Category of the related Item(s).
//      parentAttrKey The attribute key of the parent pointing to
//      	      the child item(s)
//
ParentAttrAssoc::ParentAttrAssoc(Category& parentCategory, 
				 const String& parentSelector, 
				 Category& childCategory,
				 const String& parentAttrKey)
 : ComputedAssoc(parentCategory, parentSelector, childCategory),
   _impl(new ParentAttrAssocImpl(parentAttrKey)) {
    Log::debug(NAME, "ParentAttrAssoc constructor for selector %s", 
	       (const char*) getSelector());
}


//
//  ParentAttrAssoc::ParentAttrAssoc(Categpry& parentCategory, 
//	                             const String& parentSelector, 
//  				     Category& childCategory, 
//  				     const String& parentAttrNumKeys,
//  				     const String& parentAttrKey)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentCategory     The Category of the parent Item.
//      parentSelector     The selector of the parent Item.
//      childCategory      The Category of the related Item(s).
//      parentAttrNumKeys  The attribute key containing the number of
//                         child items.
//      parentAttrKey      The attribute key of the parent pointing to
//      	           the child item(s)
//
ParentAttrAssoc::ParentAttrAssoc(Category& parentCategory, 
				 const String& parentSelector, 
				 Category& childCategory,
				 const String& parentAttrNumKeys,
				 const String& parentAttrKey)
 : ComputedAssoc(parentCategory, parentSelector, childCategory),
   _impl(new ParentAttrAssocImpl(parentAttrNumKeys, parentAttrKey)) {
    Log::debug(NAME, "ParentAttrAssoc constructor for selector %s", 
	       (const char*) getSelector());
}

//
//  ParentAttrAssoc::~ParentAttrAssoc()
//
//  Description:
//     	Destructor. 
//
ParentAttrAssoc::~ParentAttrAssoc() {
    delete _impl;
}

//
//  bool ParentAttrAssoc::isChild(const Item& potentialChildItem)
//
//  Description:
//      Check if an item is a child of the parent item.
//
//  Parameters:
//      potentialChildItem An item of the child category.
//
//  Returns:
//	true if the item is a child of the parent item. false otherwise.
//
bool ParentAttrAssoc::isChild(const Item& potentialChildItem) {
    if (_impl->_oneToOneAssoc) {
	return (strcmp(getParent()->getAttr(_impl->_parentAttr).stringValue(),
		       potentialChildItem.getSelector()) == 0);
    } else {
	return
	    (_impl->_selectors->lookupByKey(potentialChildItem.getSelector())
	     != NULL);
    }
}

//
//  NotificationFilter* 
//	ParentAttrAssoc::createAddedChildNotificationFilter(const
//	                                                    Item& parentItem)
//
//  Description:
//      Specifies the filter to be used when monitoring is started
//	on items of the child category.
//
//  Parameters:
//      parentItem The parent Item that was added.
//
//  Returns:
//	Filter with items corresponding to
// 	attribute key  specified by parameter "parentAttrKey"
//	in the constructor call.
//
NotificationFilter* 
    ParentAttrAssoc::createAddedChildNotificationFilter(const Item& 
							parentItem) {
    NotificationFilter* filter = new NotificationFilter();

    // Check if it is an attribute pointing to a single instance 
    // of child
    if (_impl->_oneToOneAssoc) {
	Attribute attr = parentItem.getAttr(_impl->_parentAttr);
	assert(attr != Attribute::NUL);
	filter->monitorItem(attr.stringValue());
    } else {
	 _impl->_selectors =
	     parentItem.createStringSequence(_impl->_parentAttrNumKeys,
					     _impl->_parentAttr);
	DictionaryOfIterator<String> iter(_impl->_selectors);
	String* selector;
	while ((selector = iter()) != NULL) {
	    Log::debug(NAME, "Monitoring: %s", (const char*) *selector);
	    filter->monitorItem(*selector);
	}
    }
    return filter;
}

//
//  NotificationFilter*
//      ParentAttrAssoc::createChangedChildNotificationFilter(
//	    const Item& parentItem)
//
//  Parameters:
//      oldItem The old parent Item.
//      newItem The new parent Item.
//
//  Description:
//      Specifies the filter to be used when monitoring is started
//	on items of the child category.
//
//  Returns:
//	Filter with items corresponding to
// 	attribute key  specified by parameter "parentAttrKey"
//	in the constructor call.
NotificationFilter*
ParentAttrAssoc::createChangedChildNotificationFilter(const Item& oldItem,
						      const Item& newItem) {

    NotificationFilter* filter = NULL;

    if (_impl->_oneToOneAssoc) {

	Attribute newAttr = newItem.getAttr(_impl->_parentAttr);
	if (oldItem.getAttr(*_impl->_parentAttr) != newAttr) {
	    filter = new NotificationFilter();
	    filter->monitorItem(newAttr.stringValue());
	}

    } else {

	DictionaryOf<String>* newSelectors = 
	    newItem.createStringSequence(_impl->_parentAttrNumKeys,
					 _impl->_parentAttr);
	bool hasChanged = _impl->selectorsChanged(*newSelectors);
	Log::debug(NAME, "hasChanged: %s", (hasChanged) ? "true" :
		   "false");
	Item::destroyStringSequence(_impl->_selectors);
	_impl->_selectors = newSelectors;

	if (hasChanged) {
	    filter = new NotificationFilter();
	    DictionaryOfIterator<String> iter(newSelectors);
	    String* selector;
	    while ((selector = iter()) != NULL) {
		Log::debug(NAME, "Monitoring: %s", (const char*) *selector);
		filter->monitorItem(*selector);
	    }
	}

    }
    return filter;
}

END_NAMESPACE(sysadm);


