//
// ParentCategoryListener.c++
//
//	Class for monitoring changes in a parent item for an association.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#include "ParentCategoryListener.h"
#include <sysadm/Log.h>
#include <sysadm/format.h>

BEGIN_NAMESPACE(sysadm);

const char* ParentCategoryListener::NAME = "ParentCategoryListener";

//
//  ParentCategoryListener::
//	ParentCategoryListener(Category& parentCategory,
//                             const String& parentSelector,
//	                       Association& assoc)
//
//  Description:
//      Constructor.
//
//  Parameters:
//      parentCategory Category of the parent item.
//      parentSelector Selector of the parent item.
//      assoc Association to forward notifications to.
//
ParentCategoryListener::ParentCategoryListener(Category& parentCategory,
                                               const String& parentSelector,
					       Association& assoc)
 : _assoc(assoc), _cat(parentCategory) {
    NotificationFilter filter;
    filter.monitorItem(parentSelector);
    _handle = _cat.adoptCategoryListener(this, filter);
}

//
//  ParentCategoryListener::~ParentCategoryListener()
//
//  Description:
//      Destructor.
//
ParentCategoryListener::~ParentCategoryListener() {
    _cat.orphanCategoryListener(_handle);
}

//
//  void ParentCategoryListener::itemAdded(const Item& item)
//
//  Description:
//      Called when parent item has been added or detected at startup.
//	Forwards notification to association.
//
//  Parameters:
//      item The item that was added.
//
void ParentCategoryListener::itemAdded(const Item& item) {
    _assoc.parentAdded(item);
}

//
//  void ParentCategoryListener::itemChanged(const Item& oldItem,
//  					     const Item& newItem)
//
//  Description:
//      Called when the parent item changes.
//	Forwards notification to association.
//
//  Parameters:
//      oldItem The old item.
//      newItem The new item.
//
void ParentCategoryListener::itemChanged(const Item& oldItem,
					 const Item& newItem) {
    _assoc.parentChanged(oldItem, newItem);
}

//
//  void ParentCategoryListener::itemRemoved(const Item& /*item*/)
//
//  Description:
//      Called when the parent item changes.
//	Forwards notification to association.
//
//  Parameters:
//      item The item that was removed.
//
void ParentCategoryListener::itemRemoved(const Item& /*item*/) {
    _assoc.parentRemoved();
}

// Do nothing on receiving beginBlockChanges
void ParentCategoryListener::beginBlockChanges() {
}

// Do nothing on receiving endBlockChanges
void ParentCategoryListener::endBlockChanges() {
}

// Do nothing on receiving endExists
void ParentCategoryListener::endExists() {
    Log::debug(NAME, "endExists");
}

// Do nothing on receiving attrChanged
void ParentCategoryListener::attrChanged(const AttrEvent& /*event*/) {
}

END_NAMESPACE(sysadm);
