//
// listCategory.c++
//
//	Print out all the items in a category.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.8 $"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sysadm/AssocFactory.h>
#include <sysadm/AppContext.h>
#include <sysadm/Log.h>
#include <sysadm/LogListener.h>
#include <sysadm/LogFilter.h>
#include <sysadm/i18n.h>

USING_NAMESPACE(sysadm);

static bool gExit = true;
static bool gVerbose = false;

class StderrLogListener : public LogListener {
  public:
    StderrLogListener() {
	LogFilter* filter = new LogFilter;
	filter->levelOn(Log::WARNING | Log::ERROR | Log::FATAL);
	if (gVerbose) {
	    filter->levelOn(Log::TRACE | Log::DEBUG | Log::INFO);
	}
	adoptFilter(filter);
    }
    virtual void handleMessage(const char* module, Log::Level level,
			       const char* format, va_list args) {
	fprintf(stderr, "%s %s: ", Log::levelToString(level), module);
	vfprintf(stderr, format, args);
	putc('\n', stderr);
	fflush(stderr);
    }
};

class CategoryPrinter : public CategoryListener {
    void printItem(const Item& item) {
	printf("%s:\n", (const char*)item.getSelector());
	CollectionOf<Attribute> attrs(item.copyAttrList());
	IteratorOver<Attribute> iter(&attrs);
	Attribute* attr;
	int maxKeyWidth = 0;
	while ((attr = iter()) != NULL) {
	    int keyWidth = attr->getKey().getLength();
	    if (keyWidth > maxKeyWidth) {
		maxKeyWidth = keyWidth;
	    }
	}
	iter.reset();
	char formatString[100];
	sprintf(formatString, "    %%-%ds : %%s\n", maxKeyWidth);
	while ((attr = iter()) != NULL) {
	    printf(formatString, (const char*)attr->getKey(),
		   (const char*)attr->getValueString());
	}
    }
    virtual void itemAdded(const Item& item) {
	printf("itemAdded: ");
	printItem(item);
    }
    virtual void itemChanged(const Item&, const Item& item) {
	printf("itemChanged: ");
	printItem(item);
    }
    virtual void itemRemoved(const Item& item) {
	printf("itemRemoved: ");
	printItem(item);
    }
    virtual void endExists() {
	if (gExit) {
	    exit(0);
	}
    }
};

void usage()
{
    fprintf(stderr,
	    "usage: listAssoc [ -f -v ] <parent cat> <parent "
	    "select> <child cat>\n");
    exit(1);
}

void main(int argc, char* argv[])
{
    int c;
    while ((c = getopt(argc, argv, "fv")) != EOF) {
	switch (c) {
	case 'f':
	    gExit = false;
	    break;
	case 'v':
	    gVerbose = true;
	    break;
	default:
	    usage();
	    break;
	}
    }
    
    if (optind != argc - 3) {
	usage();
    }
    
    Log::getLog().adoptLogListener(new StderrLogListener);

    String errorString;
    Category* cat = AssocFactory::getAssociation(argv[argc - 3],
						 argv[argc - 2],
						 argv[argc - 1],
						 errorString);
    if (cat == NULL) {
	fprintf(stderr, "Can't get Category: %s\n", argv[1]);
	exit(1);
    }

    NotificationFilter filt;
    filt.monitorAllItems();
    cat->adoptCategoryListener(new CategoryPrinter, filt);
    AppContext::getAppContext().run();
}
