//
// listCategory.c++
//
//	Print out all the items in a category.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.10 $"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sysadm/CategoryFactory.h>
#include <sysadm/AppContext.h>
#include <sysadm/Log.h>
#include <sysadm/LogListener.h>
#include <sysadm/LogFilter.h>
#include <sysadm/i18n.h>

USING_NAMESPACE(sysadm);

static bool gVerbose = false;
static bool gExit = true;

class StderrLogListener : public LogListener {
  public:
    StderrLogListener() {
	LogFilter* filter = new LogFilter;
	filter->levelOn(Log::WARNING | Log::ERROR | Log::FATAL);
	if (gVerbose) {
	    filter->levelOn(Log::INFO | Log::DEBUG | Log::TRACE);
	}
	adoptFilter(filter);
    }
    virtual void handleMessage(const char* module, Log::Level level,
			       const char* format, va_list args) {
	fprintf(stderr, "%s %s: ", Log::levelToString(level), module);
	vfprintf(stderr, format, args);
	putc('\n', stderr);
	fflush(stderr);
    }
};

class CategoryPrinter : public CategoryListener {
    virtual void itemAdded(const Item& item) {
	printItem("ADDED", item);
    }
    virtual void itemChanged(const Item&, const Item& newItem) {
	printItem("CHANGED", newItem);
    }
    virtual void itemRemoved(const Item& item) {
	printItem("DELETED", item);
    }
    virtual void endExists() {
	if (gExit) {
	    exit(0);
	}
    }
    virtual void attrChanged(const AttrEvent& event) {
	Attribute attr(event.getChangedAttr());
	printf("attr: %s=%s\n", (const char*)attr.getKey(),
	       (const char*)attr.getValueString());
    }
    void printItem(const char* prefix,  const Item& item) {
	printf("%s: %s:\n", prefix, (const char*)item.getSelector());
	CollectionOf<Attribute> attrs(item.copyAttrList());
	IteratorOver<Attribute> iter(&attrs);
	Attribute* attr;
	int maxKeyWidth = 0;
	while ((attr = iter()) != NULL) {
	    int keyWidth = attr->getKey().getLength();
	    if (keyWidth > maxKeyWidth) {
		maxKeyWidth = keyWidth;
	    }
	}
	iter.reset();
	char formatString[100];
	sprintf(formatString, "    %%-%ds : %%s\n", maxKeyWidth);
	while ((attr = iter()) != NULL) {
	    printf(formatString, (const char*)attr->getKey(),
		   (const char*)attr->getValueString());
	}
    }
};

void usage()
{
    fprintf(stderr, "usage: listCategory [ -v -f ] <category name>\n");
    exit(1);
}

void main(int argc, char* argv[])
{
    int c;
    while ((c = getopt(argc, argv, "fv")) != EOF) {
	switch (c) {
	case 'f':
	    gExit = false;
	    break;
	case 'v':
	    gVerbose = true;
	    break;
	default:
	    usage();
	    break;
	}
    }
    
    if (optind != argc - 1) {
	usage();
    }

    char* catName = argv[argc - 1];

    Log::getLog().adoptLogListener(new StderrLogListener);

    String errorString;
    Category* cat = CategoryFactory::getCategory(catName, errorString);
    if (cat == NULL) {
	fprintf(stderr, "Can't get Category: %s\n", catName);
	fprintf(stderr, "Error: %s\n", (const char*) errorString);
	exit(1);
    }

    NotificationFilter filt;
    filt.monitorAllItems();
    filt.monitorCategoryAttrs();
    cat->adoptCategoryListener(new CategoryPrinter, filt);
  
    AppContext::getAppContext().run();
}
