//
// ByteStreamConnection.c++
//
//	Connection that converts between Packets and byte streams.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.9 $"

#include <string.h>

#include <sysadm/ByteStreamConnection.h>
#include <sysadm/Log.h>
#include <sysadm/i18n.h>
#include <sysadm/AppContext.h>

BEGIN_NAMESPACE(sysadm);

#define PACKET "packet"
#define DATA_LENGTH "dataLength"
#define PACKET_LOG_PAD 2

//
// ByteStream constructor.
//
ByteStreamConnection::ByteStreamConnection()
: _buf(NULL), _bufLength(0)
{
}

//
// ByteStream destructor.
//
ByteStreamConnection::~ByteStreamConnection()
{
    delete [] _buf;
}

//
//  void ByteStreamConnection::sendPacket(const Packet& packet)
//
//  Description:
//      Send a packet to our peer.
//
//  Parameters:
//      packet  The packet to send.
//
void ByteStreamConnection::sendPacket(const Packet& packet)
{
    AttrBundle attrs("", "");
    attrs.setAttr(Attribute(PACKET, packet));
    size_t dataLength = packet.getDataLength();
    attrs.setAttr(Attribute(DATA_LENGTH, (long long)dataLength));

    // It's very important not to put logging between the calls to
    // sendData() below, because the Log service itself might end up
    // calling us re-entrantly.  It's up to the Log service to prevent
    // infinite recursion, and it's up to us to ensure the integrity
    // of the data stream.
    Log::trace(CONNECTION_SERVICE, "Sending packet:\n%s",
	       (const char*)packet.toString(PACKET_LOG_PAD));
    if (dataLength > 0) {
	Log::trace(CONNECTION_SERVICE, "Sending %d bytes of packet data.",
	    dataLength);
    }

    String stream(attrs.serialize());
    stream += "\n";
    sendData((const char*)stream, stream.getLength());
    if (dataLength > 0) {
	sendData(packet.getData(), dataLength);
    }
}

//
//  void ByteStreamConnection::handleInput()
//
//  Description:
//      Called when input is available from our socket.  Read from the
//      socket and break what we read up into Packets which we
//      distribute to listeners by calling receivePacket().
//
void ByteStreamConnection::handleInput()
{
#ifdef DEBUG_CONNECTION
    // Very low buffer size to test that buffering actually works.
    const int READ_LENGTH = 1;
#else
    const int READ_LENGTH = 512;
#endif
    char buf[READ_LENGTH];
    int n = receiveData(buf, READ_LENGTH);
    if (n < 0) {
	return;
    }
    if (n == 0) {
	Log::info(CONNECTION_SERVICE, i18n("Lost connection."));
	AppContext::getAppContext().exit(0);
	return;
    }
    processInput(buf, n);
}

//
//  void ByteStreamConnection::processInput(char* inputBuffer, int inputLength)
//
//  Description:
//      Process input received from the connection.  We process input
//      a packet at a time; if we haven't read the entire packet, we
//      just buffer what we've got so far to be used the next time
//      we're called.
//
//  Parameters:
//      buf  Input that was just read.
//      len  length of input what was just read.
//
void ByteStreamConnection::processInput(char* inputBuffer, int inputLength)
{
    char* pc;			// Points to where we're parsing.
    int parseLength;		// Length of what we're parsing.

    if (_bufLength == 0) {
	// Make sure buffer is really empty.
	assert(_buf == NULL);

	// We don't have anything buffered from a previous call, so we
	// can parse "inputBuffer" without copying it.
	pc = inputBuffer;
	parseLength = inputLength;
    } else {
	// We have unparsed input buffered from a previous call.  Make
	// our buffer big enough to store its current contents plus
	// this new input, and then combine the buffer with the new
	// input, which we will then parse.
	char* newBuf = new char[_bufLength + inputLength];
	memcpy(newBuf, _buf, _bufLength);
	memcpy(newBuf + _bufLength, inputBuffer, inputLength);
	delete [] _buf;
	_buf = newBuf;
	_bufLength += inputLength;
	pc = _buf;
	parseLength = _bufLength;
    } 

    // Parse "pc" into packets.  Packets are delimited by newline
    // characters.  If a packet has additional data associated with
    // it, the data will immediately follow the newline character
    // which terminates the packet.
    //
    // If "pc" is a partial packet, we just leave it.  It will stay in
    // our buffer and we'll try again after having read more input.
    char* newLine;
    while ((newLine = (char*)memchr(pc, '\n', parseLength)) != NULL) {
	int lineLength = newLine - pc;
	AttrBundle attrs(String(pc, lineLength));
	Attribute dataLengthAttr(attrs.getAttr(DATA_LENGTH));
	if (dataLengthAttr == Attribute::NUL) {
	    Log::fatal(CONNECTION_SERVICE, i18n("Malformed packet."));
	    AppContext::getAppContext().exit(1);
	    return;
	}
	long dataLength = (long)dataLengthAttr.longValue();
	// -1 for '\n'
	if (parseLength - lineLength - 1 < dataLength) {
	    break;
	}

	Packet packet(attrs.getAttr(PACKET).bundleValue());
	Log::trace(CONNECTION_SERVICE, "Got packet:\n%s",
		   (const char*)packet.toString(PACKET_LOG_PAD));
	if (dataLength > 0) {
	    Log::trace(CONNECTION_SERVICE, "Got %d bytes of packet data.",
	               dataLength);
	    packet.setData(newLine + 1, dataLength);
	}

	distributePacket(packet);

	// +1 for '\n'
	pc += dataLength + lineLength + 1;
	parseLength -= dataLength + lineLength + 1;
    }

    // If pc points to the beginning of _buf, we didn't parse
    // anything, so there's no need to adjust our buffers.
    if (pc == _buf) {
	return;
    }

    // parseLength > 0 means that there's a partial packet that we
    // need to buffer for later.
    if (parseLength > 0) {
	if (_buf != NULL) {

	    // pc should be pointing somewhere into the middle of
	    // _buf.  We want to move that portion of _buf that comes
	    // after pc to the beginning of _buf.
	    assert(_buf + _bufLength == pc + parseLength);
	    assert(pc > _buf);

	    memmove(_buf, pc, parseLength);
	    _bufLength = parseLength;
	} else {
	    // We were parsing "inputBuffer" directly.  We need to
	    // create a buffer to store that portion of it which we
	    // did not parse.
	    char* newBuf = new char[parseLength];
	    memcpy(newBuf, pc, parseLength);
	    _bufLength = parseLength;
	    // Make sure we're not leaking.
	    assert(_buf == NULL);
	    _buf = newBuf;
	}
    } else {
	// We parsed all input that we've read so far, so we free any
	// buffer that we had.
	delete [] _buf;
	_buf = NULL;
	_bufLength = 0;
    }
}

END_NAMESPACE(sysadm);
