//
// ByteStreamConnection.h
//
//	Connection that converts between Packets and byte streams.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ifndef _SYSADM_BYTE_STREAM_CONNECTION_H
#define _SYSADM_BYTE_STREAM_CONNECTION_H
#ident "$Revision: 1.4 $"

#include <sysadm/Connection.h>

BEGIN_NAMESPACE(sysadm);

//
// ByteStreamConnection is a subclass of Connection that sends a
// packet by encapsulating it in an AttrBundle, serializing it, and
// sending the resulting byte stream via sendData(), which a subclass
// must provide.
//
// The handleInput() method should be called by a subclass whenever
// some input is available to be returned by the subclass-supplied
// receiveData() method.  handleInput() calls receiveData() repeatedly
// until it returns -1, and parses the data stream into packets which
// it then distributes by calling Connection::distributePacket().
//
// ByteStreamConnection handles arbitrary data (see Packet::setData())
// by inserting it directly into the packet stream.
//
class ByteStreamConnection : public Connection {
  public:

    ByteStreamConnection();
    virtual ~ByteStreamConnection();

    // Send a packet.
    virtual void sendPacket(const Packet& packet);

  protected:

    // Subclasses must implement this method for sending data to our
    // peer.
    virtual void sendData(const void* data, size_t nbytes) = 0;

    // Subclasses must implement this method for receiving data from
    // our peer.  This method should not block.  handleInput(), which
    // calls this method, handles the cases of partial packet reads.
    // This method should return the actual number of bytes received.
    // If no input is available, -1 should be returned.  If the
    // connection is lost, 0 should be returned.
    virtual ssize_t receiveData(void* data, size_t nbytes) = 0;
    
    // Called by monitorCallback() when input is available.
    // Connection calls receiveData() repeatedly as long as
    // receiveData() returns information, and breaks the data up into
    // packets.  receivePacket() is called for each Packet.
    virtual void handleInput();

  private:
    void processInput(char* buf, int len);

    char* _buf;
    int _bufLength;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_BYTE_STREAM_CONNECTION_H
