//
// Connection.c++
//
//	Connection to our peer.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.10 $"

#include <unistd.h>
#include <sysadm/Connection.h>
#include <sysadm/AppContext.h>
#include <sysadm/Log.h>

BEGIN_NAMESPACE(sysadm);

//
// For logging.
//
const char Connection::CONNECTION_SERVICE[] = "conn";

//
// Connection constructor.
//
Connection::Connection()
: _remoteHost(String::EMPTY)
{
}

//
// Connection destructor.
//
Connection::~Connection()
{
    RemoveAndDestroyContentsOf(&_listeners);
}

//
//  void Connection::adoptPacketListener(const String& type,
//  				         PacketListener* listener)
//
//  Description:
//      Adopt a listener that will have its receivePacket() method
//      called when we receive a Packet with a type of "type".
//
//  Parameters:
//      type      Type of packets this listener is to receive.
//      listener  Listener to get notified when packets of "type" arrive.
//
void Connection::adoptPacketListener(const String& type,
				     PacketListener* listener)
{
    // We only support one listener per packet type.
    assert(_listeners.lookupByKey(type) == NULL);
    _listeners.add(listener, new String(type));
}

//
//  PacketListener* Connection::orphanPacketListener(const String& type)
//
//  Description:
//      Remove the listener for Packets of type "type".
//
//  Parameters:
//      type  type to remove listener for.
//
//  Returns:
//	The listener for type "type".
//
PacketListener* Connection::orphanPacketListener(const String& type)
{
    return _listeners.remove(type);
}

//
//  void Connection::distributePacket(const Packet& packet)
//
//  Description:
//      Receive a packet.  Find the right listener and call its
//      receivePacket() method.
//
//  Parameters:
//      packet  Packet that we received.
//
void Connection::distributePacket(const Packet& packet)
{
    PacketListener* listener = _listeners.lookupByKey(packet.getType());
    if (listener != NULL) {
	listener->receivePacket(packet);
    }
}

String Connection::remoteHostName()
{
    return _remoteHost;
}

void Connection::setRemoteHostName(const String& host)
{
    _remoteHost = host;
}

END_NAMESPACE(sysadm);
