//
// SocketConnection.c++
//
//	ByteStreamConnection reads and writes byte streams on a
//	socket.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.9 $"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <unistd.h>

#include <sysadm/io.h>
#include <sysadm/SocketConnection.h>
#include <sysadm/AppContext.h>

BEGIN_NAMESPACE(sysadm);

//
// SocketConnection constructor.
//
SocketConnection::SocketConnection(int sock)
: _sock(sock)
{
    AppContext& app(AppContext::getAppContext());
    _monitor = app.registerMonitor(sock, monitorCallback, this);
}

//
// SocketConnection destructor.
//
SocketConnection::~SocketConnection()
{
    AppContext& app(AppContext::getAppContext());
    app.unregisterMonitor(_monitor);
    ::close(_sock);
}

//
//  String SocketConnection::remoteHostName()
//
//  Description:
//      Get a name for the remote host.
//
//  Returns:
//	Name of remote host.
//
String SocketConnection::remoteHostName()
{
    String hostName = Connection::remoteHostName();
    if (hostName.getLength()) {
        return hostName;
    } 

    sockaddr_in addr;
    socklen_t nameLen = sizeof addr;
    if (getpeername(_sock, (struct sockaddr*)&addr, &nameLen) == -1
	|| addr.sin_family != AF_INET) {
	return String::EMPTY;
    }

    //  The first argument is a pointer to a 32-bit int, but gethostbyaddr
    //  on linux wants it to be a char.
    hostent* ent = gethostbyaddr((const char *)(&addr.sin_addr.s_addr),
				 sizeof addr.sin_addr.s_addr,
				 AF_INET);
    if (ent != NULL) {
	return String(ent->h_name);
    }
    
    return String(inet_ntoa(addr.sin_addr));
}

//
//  void SocketConnection::sendData(const void* data, size_t nbytes)
//
//  Description:
//      Send data to our peer.
//
//  Parameters:
//      data    The data to send.
//      nbytes  The number of bytes to send.
//
void SocketConnection::sendData(const void* data, size_t nbytes)
{
    if (SaWriteFully(_sock, data, nbytes) == -1) {
	perror("SaWriteFully");
	exit(1);
    }
}

//
//  ssize_t SocketConnection::receiveData(void* data, size_t nbytes)
//
//  Description:
//      Receive data from our peer.
//
//  Parameters:
//      data    Buffer to read data into.
//      nbytes  Maximum number of bytes to read.
//
//  Returns:
//	The actual number of bytes read.
//
ssize_t SocketConnection::receiveData(void* data, size_t nbytes)
{
    return SaRead(_sock, data, nbytes);
}


//
//  void Socketonnection::monitorCallback(void* clientData, int, int)
//
//  Description:
//      Called by AppContext when some input is available on our
//      socket.
//
//  Parameters:
//      clientData   Connection* (this is a static method)
//
void SocketConnection::monitorCallback(void* clientData, int, int)
{
    ((SocketConnection*)clientData)->handleInput();
}

END_NAMESPACE(sysadm);
