//
// FileBasedCategory.c++
//
//	Base class for Categories that generate Items based on file
//	contents.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.6 $"

#include <sysadm/AppContext.h>
#include <sysadm/Log.h>
#include <sysadm/FileBasedCategory.h>
#include <string.h>
#include <strings.h>

BEGIN_NAMESPACE(sysadm);

//
// FileBasedCategory constructor.
//
FileBasedCategory::FileBasedCategory(const String& selector, 
				     const String& filename)
        : Category(selector), _filename(filename),
	  _inputId(-1), _itemType(selector), _needToCloseFAM(false),
	  _directory(filename)
{
    char* pos = strrchr(_directory.writableCharPtr(),'/');
    if (pos != NULL) {
	//strip off everything after the last slash
	_shortFilename = pos+1;
	pos++;	
	*pos = '\0';
    } else {
	_shortFilename = filename;
    }
}


//
// FileBasedCategory destructor.
//
FileBasedCategory::~FileBasedCategory()
{
    if (_needToCloseFAM) {
	
	if (FAMClose(&_famConnection) == -1) {
	    Log::fatal(getSelector(), "Error closing FAM connection.");
	} 
        if (_inputId != -1) {
            AppContext::getAppContext().unregisterMonitor(_inputId);
	}
    }
}

//
//  void FileBasedCategory::startMonitor()
//
//  Description:
//      Called when it's time to start famming the test file
//
void FileBasedCategory::startMonitor()
{
    Log::trace(getSelector(), "startMonitor()");
    
    FAMRequest _famRequest;
    // open the connection to FAM.  This initializes _famConnection
    
    if (FAMOpen(&_famConnection) == -1) {
	Log::fatal(getSelector(), "Couldn't open the FAM Connection");
    }
    else {
	_needToCloseFAM = true;
	// start monitoring the file
        if (FAMMonitorDirectory(&_famConnection,
				_directory,
				&_famRequest,
				NULL) == -1){
	    
	    Log::fatal(getSelector(),
		      "Couldn't start monitoring the directory");
        }
        else {
             // get the file descriptor
            int fd = FAMCONNECTION_GETFD(&_famConnection);

            
            _inputId = AppContext::getAppContext().
                registerMonitor(fd, handleInput, this, AppContext::INPUT);
	}
    }
}


//
//  void FileBasedCategory::handleInput(void* clientData, 
//                                      int /*id*/, int /*fd*/)
//
//  Description:
//      Called when we receive some input from FAM
//
//  Parameters:
//      clientData  FileBasedCategory* (this is a static method)
//
void FileBasedCategory::handleInput(void* clientData, 
				    int /*id*/, 
				    int /*fd*/) {
    FileBasedCategory* self = (FileBasedCategory*)clientData;
    
    FAMEvent event;  // will return information about the FAM event
    
    while(FAMPending(&self->_famConnection) == 1) {
        // While there are events pending . . .
	if (FAMNextEvent(&self->_famConnection,&event) == -1) {
	    Log::fatal(self->getSelector(),"FAMNextEvent failed");
	    break;
	}
        // put the event info in event

	if (strcmp(event.filename,self->_shortFilename)) {
	    continue;
	}
        // skip everything unless the filename matches.

	switch  (event.code) { // see fam man page for codes
	case FAMDeleted:
	    self->replaceItemList(ItemList());
	    break;                
	case FAMExists:
	case FAMCreated:
	case FAMChanged:
	    ItemList* itemList =
		self->getItemsFromFile((const char*) self->_filename,
				       (const char*) self->_itemType);
	    self->replaceItemList(*itemList);
	    RemoveAndDestroyContentsOf(itemList);
	    delete itemList;
	    break;
        }
        if (event.code == FAMExists) {
            self->endExists();
        }
    }
}


#define LINE_LENGTH 1000

typedef struct AttrStruct {
    String key;
    int type;
} attrDef;

typedef OrderedCollectionOf<attrDef> AttrDefList; 
typedef IteratorOver<Item> ItemIter;
typedef IteratorOver<attrDef> AttrDefIter; 

//  ItemList* FileBasedCategory::getItemsFromFile(const char* file,
//                                                 const char* itemType) {
//
//  Description:
//      Extracts information from "file" to create item of type
//      "itemType" and returns a list of Items
//
//  Parameters:
//      file The filename to read the info from
//      itemType The string to use as they type of the Item
//
ItemList* FileBasedCategory::getItemsFromFile(const char* file,
                                                const char* itemType) {
    ItemList* items = new ItemList;
    FILE* fp = fopen(file,"r");
    if (!fp) {
        Log::fatal(getSelector(),
		  "Couldn't open file %s for reading",file);
        return NULL;
    }
    
    char line[LINE_LENGTH];
    
    AttrDefList attrDefs; 
    char * result = fgets(line,LINE_LENGTH,fp);
    while (result != NULL && result[0] == '#') {
	result = fgets(line,LINE_LENGTH,fp);
    }
    if (result == NULL) {
	Log::fatal(getSelector(),"No lines in file!");
    }
    if (line[strlen(line)-1] == '\n') {
	    line[strlen(line)-1] = '\0';
    }
    Log::debug(getSelector(),"definition line: %s",line);
    char* lasts = NULL;
    bool firstTime = true;
    while (1) {
        char* key;
        char* type;	
	key = strtok_r(firstTime ? line : NULL ,"!",&lasts);
	firstTime = false;
	if (key == NULL) {
	    break;
	}
	if (strchr(key,':')) {
	    Log::fatal(getSelector(),
		       "Found a Attribute name without a type following");
	}
	type = strtok_r(NULL,":",&lasts);
	if (type == NULL) {
	    Log::fatal(getSelector(),
		       "Got attribute name: \"%s\", but no type followed", 
		       key);
	}
	if (strchr(type,'!')) {
	     Log::fatal(getSelector(),
			"Found an Attribute with two types specified");
	}  
	if (strlen(type) != 1) {
	    Log::fatal(getSelector(),
			 "Found an Attribute with an illegal type");
	}

	Log::debug(getSelector(),"key is %s, type is %c", key, type[0]);
        attrDef* newAttrDef = new attrDef;
        newAttrDef->key = key;
        switch (type[0]) {
	case 'S' :
	case 's' : newAttrDef->type=Attribute::STRING;
	    break;
	case 'L' :
	case 'l': newAttrDef->type=Attribute::LONG;
	    break;
	case 'B' :
	case 'b' : newAttrDef->type=Attribute::BOOLEAN;
	    break;
	case 'D' : 
	case 'd' : newAttrDef->type=Attribute::DOUBLE;
	    break;
        }
        attrDefs.add(newAttrDef);
    }
    lasts = NULL;
    AttrDefIter attrDefIter(&attrDefs);
    attrDef* currentAttrDef;
    int size = attrDefs.getSize();
    while (fgets(line,LINE_LENGTH,fp)) {
	if (line[0] == '\n' || line[0] == '#') {
	    continue;
	}
	if (line[strlen(line)-1] == '\n') {
	    line[strlen(line)-1] = '\0';
	}
	attrDefIter.reset();
        Log::debug(getSelector(), "item line: %s",line);
        char* value;
        value = strtok_r(line,":",&lasts);
        Log::debug(getSelector(), "Selector: %s",value);
        Item* item = new Item(itemType,value);
        Attribute* attr;
	String currentKey; 
        for (int counter = 0 ; counter < size ; counter++) {
	    value = strtok_r(NULL,":",&lasts);
	    if (value == NULL) {
		Log::fatal(getSelector(),
			   "Not enough attributes for item %d",counter);
	    }	   
	    currentAttrDef = attrDefIter();
	    currentKey = String(currentAttrDef->key);
	    Log::debug(getSelector(), 
		       "Value was %s for Attribute %s",
		       value,
		       (const char *)currentKey);
            switch (currentAttrDef->type) {
	    case Attribute::STRING :
		attr = new Attribute(currentKey,
				     value);
		break;
	    case Attribute::LONG :
		attr = new Attribute(currentKey,
				     (long long)atol(value));
		break;
	    case Attribute::BOOLEAN:
		attr = new Attribute(currentKey,
				     value[0]=='T' || value[0]=='t');
		break;
	    case Attribute::DOUBLE:
		attr = new Attribute(currentKey,
				     atof(value));
		break;
            } 
            item->setAttr(*attr);
            delete attr;
        }  
        items->add(item);
    }
    RemoveAndDestroyContentsOf(&attrDefs);
    return items;
}
END_NAMESPACE(sysadm);
