/*
 * SaParam.c
 *
 *	Interface for passing lists of key-value pairs across command
 *	invocations.
 *
 *  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.13 $"

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>

#include <sysadm/sysadm.h>
#include <sysadm/io.h>
#include <sysadm/SaParam.h>
#include <sysadm/format.h>

#define ALLOC_INCR 10
#define NUM_BUF_SIZE 10
#define QUOTE_CHAR '%'

/*
 * An element in our array of key/value pairs.
 */
typedef struct _SaKeyValue {
    char *key;
    char *value;
    int position;		/* For maintaining order */
    bool visible;		/* For controlling visibility */
} SaKeyValue;  

/*
 * Parameter information.  This is the data structure returned by
 * SaParamCreate().
 */
struct _SaParam {
    SaKeyValue **params;	/* Array of key/value pairs */
    SaKeyValue **paramsByName;	/* Array of key/value pairs sorted by name */
    int nAlloc;			/* Number allocated in params */ 
    int nParams;		/* Number used in params */
    int dirty;			/* if non-zero, Compress() is needed */
    int nextPosition;		/* position of next element added */
};

/*
 * Iterator for SaParam.
 */
struct _SaParamIter {
    SaParam *param;		/* corresponding SaParam */
    int index;			/* index of current element */
};

/*
 * Arguments for the exec(2) system call.
 */
struct _SaParamExecArgs {
    char **argList;
    int numArgs;
    bool overFlow;
    SaParam *pipeArgs;
};

/*
 * Element in our table of escape information, which we use to escape
 * characters which have special meaning to us.
 */
typedef struct _EscapeInfo
{
    char toEscape;		/* Character to escape */
    const char *escapeStr;	/* String to replace it with */
    int escapeLen;		/* Length of escape string */
} EscapeInfo;

/*
 * Table of escape information.  If a key or a value contains any of
 * the characters in the column on the left, they are replaced with the
 * escape sequence in the column in the middle when transferred across a
 * command line or via a file descriptor.  We do this so that keys and
 * values can contain characters which are significant in the format
 * we use to transfer keys and values from one process to another.
 *
 * The column on the right is for the length of each sequence and is
 * initialized at runtime.
 */
static EscapeInfo gEscapes[] = {
    { '=',	"%3d",		0 }, /* '=' separates keys from values */
    { '%',	"%25", 		0 }, /* '%' introduces escape strings */
    { '\n',	"%0a",		0 }  /* '\n' terminates a key/value pair */
};
#define NUM_ESCAPES (sizeof gEscapes / sizeof *gEscapes)

/*
 * Caches the value of sysconf(_SC_ARG_MAX).
 */
static long gArgMax = 0;

/*
 *  static char *Quote(const char *str)
 *
 *  Description:
 *      Replace special characters in "str" with their corresponding
 *      escape codes.  Allocates a new string which must be freed by
 *      caller.
 *
 *  Parameters:
 *      str  String to quote
 *
 *  Returns:
 *	String with escape characters replaced.  Caller must free this
 *	string. 
 */
static char *Quote(const char *str)
{
    int size, i;
    const char *pc;
    char *quoted, *dest;

    assert(str != NULL);

    /* Figure out how much space we will need */
    size = 0;
    for (pc = str; *pc != '\0'; pc++) {
	for (i = 0; i < NUM_ESCAPES; i++) {
	    /*
	     * If it's a special character, count the number of
	     * characters in the escape sequence
	     */
	    if (*pc == gEscapes[i].toEscape) {
		size += gEscapes[i].escapeLen;
		break;
	    }
	}

	/* Otherwise, it's just one character */
	if (i == NUM_ESCAPES) {
	    size++;
	}
    }

    /* Allocate the new string */
    quoted = malloc(size + 1);
    if (quoted == NULL) {
	return NULL;
    }

    /* Fill the new string with characters and escape sequences */
    dest = quoted;
    for (pc = str; *pc != '\0'; pc++) {
	for (i = 0; i < NUM_ESCAPES; i++) {
	    if (*pc == gEscapes[i].toEscape) {
		strcpy(dest, gEscapes[i].escapeStr);
		dest += gEscapes[i].escapeLen;
		break;
	    }
	}

	if (i == NUM_ESCAPES) {
	    *dest++ = *pc;
	}
    }

    *dest = '\0';

    /* Make sure we did our math right */
    assert(strlen(quoted) == size);
    return quoted;
}

/*
 *  static char *Unquote(const char *quoted)
 *
 *  Description:
 *      Remove escape sequences from "quoted".  A new string without
 *      escape sequences is allocated and returned.
 *
 *  Parameters:
 *      quoted  string with (possibly) escape sequences.
 *
 *  Returns:
 *	A string without escape sequences; the returned string may
 *	contain special characters.  Caller must free this string.
 */
static char *Unquote(const char *quoted)
{
    const char *pc;
    char *str, *dest;
    int i;

    assert(quoted != NULL);

    /*
     * The escape sequences are all bigger than one character, so we
     * know the unquoted string will be no longer than the quoted
     * string
     */
    str = malloc(strlen(quoted) + 1);
    if (str == NULL) {
	return NULL;
    }

    /* Replace escape sequences with the appropriate characters. */
    pc = quoted;
    dest = str;
    while (*pc != '\0') {
	/* Fast path for regular characters */
	if (*pc != QUOTE_CHAR) {
	    *dest++ = *pc++;
	    continue;
	}

	for (i = 0; i < NUM_ESCAPES; i++) {
	    if (strncmp(pc, gEscapes[i].escapeStr,
			gEscapes[i].escapeLen) == 0) {
		*dest++ = gEscapes[i].toEscape;
		pc += gEscapes[i].escapeLen;
		break;
	    }
	}

	if (i == NUM_ESCAPES) {
	    *dest++ = *pc++;
	}
    }
    *dest = '\0';

    /* Make sure we did our math right */
    assert(strlen(str) <= strlen(quoted));
    return str;
}

/*
 *  static int ParseKeyValue(SaParam *param, char *keyValue)
 *
 *  Description:
 *      Parse a string of the form "key=value" into its key and value
 *      components, and add the key/value pair to param.  Note that this
 *      function will scribble on "keyValue" as a side-effect of the
 *      parsing.
 *      
 *      Also note that if value is the empty string, no value will be
 *      set.
 *
 *  Parameters:
 *      param     Destination for key/value pair.
 *      keyValue  String of the form "key=value" to be parsed.  The '='
 *                character will be overwritten with a '\0' as a side-effect.
 *
 *  Returns:
 *	0 if successful, -1 if error.
 */
static int ParseKeyValue(SaParam *param, char *keyValue)
{
    char *equal;
    char *key, *value;
    int status;

    equal = strchr(keyValue, '=');
    *equal = '\0';

    key = Unquote(keyValue);
    value = Unquote(equal + 1);

    status = -1;
    if (key != NULL && value != NULL > 0) {
	if (strlen(value) == 0) {
	    status = 0;
	} else {
	    status = SaParamSet(param, key, value);
	}
    }

    free(key);
    free(value);
    
    return status;
}

/*
 *  static int ComparePositions(const void *left, const void *right)
 *
 *  Description:
 *      qsort callback function for comparing two SaKeyValue
 *      structures when sorting by position.
 *
 *  Parameters:
 *      left   left side of comparison
 *      right  right side of comparison
 *
 *  Returns:
 *	< 0 if left < right, 0 if left == right, > 0 if left > right.
 */
static int ComparePositions(const void *left, const void *right)
{
    const SaKeyValue *leftKV = *(const SaKeyValue **)left;
    const SaKeyValue *rightKV = *(const SaKeyValue **)right;
    return leftKV->position - rightKV->position;
}

/*
 *  static int CompareKeys(const void *left, const void *right)
 *
 *  Description:
 *      qsort callback function for comparing two SaKeyValue structures
 *      when sorting by key.  Position is used to distinguish between
 *      structures with the same key.
 *
 *  Parameters:
 *      left   left side of comparison.
 *      right  right side of comparion.
 *
 *  Returns:
 *	< 0 if left < right, 0 if left == right, > 0 if left > right.
 */
static int CompareKeys(const void *left, const void *right)
{
    const SaKeyValue *leftKV = *(const SaKeyValue **)left;
    const SaKeyValue *rightKV = *(const SaKeyValue **)right;
    int compare = strcmp(leftKV->key, rightKV->key);
    return compare ? compare : ComparePositions(left, right);
}

/*
 *  static int Compress(SaParam *param)
 *
 *  Description:
 *      Remove SaKeyValue structures with duplicate keys from
 *      "param".  We take care to preserve the order of the key/value
 *      pairs.
 *
 *  Parameters:
 *      param  SaParam to compress.
 *
 *  Returns:
 *	0 if successful, -1 if error.
 */
static int Compress(SaParam *param)
{
    SaKeyValue **src, **dest;
    int nParams, i;

    if (!param->dirty) {
	return 0;
    }

    /*
     * Sort in alphabetical order by key.
     */
    qsort(param->params, param->nParams, sizeof *param->params,
	  CompareKeys);

    /*
     * Eliminate elements with duplicate keys.  The last element with
     * a duplicate key wins, but retains the position of the first
     * element with that key.
     */
    src = param->params + 1;
    dest = param->params;
    while (src - param->params < param->nParams) {
	if (strcmp((*src)->key, (*dest)->key) == 0) {
	    /*
	     * Eliminate duplicate key.  Use the value of the later
	     * pair and the position of the earlier pair.
	     */
	    free((*dest)->value);
	    (*dest)->value = (*src)->value;
	    (*src)->value = NULL;
	} else {
	    dest++;
	    if (src != dest) {
		free((*dest)->key);
		free((*dest)->value);
		**dest = **src;
		(*src)->key = NULL;
		(*src)->value = NULL;
	    }
	}
	src++;
    }

    nParams = dest - param->params + 1;

    /*
     * Make a copy of the sorted-by-name list so we can use binary
     * search in SaParamGet().
     */
    free(param->paramsByName);
    param->paramsByName = malloc(sizeof *param->paramsByName * nParams);
    if (param->paramsByName == NULL) {
	return -1;
    }

    memcpy(param->paramsByName, param->params,
	   sizeof *param->paramsByName * nParams);

    /*
     * Cleanup any elements we're no longer using.
     */
    for (i = nParams; i < param->nParams; i++) {
	free(param->params[i]->key);
	free(param->params[i]->value);
	free(param->params[i]);
	param->params[i] = NULL;
    }

    /*
     * Sort by position.
     */
    param->nParams = nParams;
    qsort(param->params, param->nParams, sizeof *param->params,
	  ComparePositions);

    param->dirty = 0;

    return 0;
}

/*
 *  SaParam *SaParamCreate(void)
 *
 *  Description:
 *      Create a new SaParam object.  Initially it has no parameters.
 *
 *  Returns:
 *	A new SaParam object; NULL if no memory can be allocated.
 *	Use SaParamDestroy() when done with the return value to free
 *	it.
 */
SaParam *SaParamCreate(void)
{
    SaParam *param = malloc(sizeof *param);
    if (param == NULL) {
	return NULL;
    }

    /* We allocate space for parameters in ALLOC_INCR increments. */
    param->nParams = 0;
    param->nAlloc = ALLOC_INCR;
    param->params = malloc(sizeof *param->params * ALLOC_INCR);
    param->dirty = 0;
    param->nextPosition = 0;
    param->paramsByName = NULL;
    if (param->params == NULL) {
	free(param);
	return NULL;
    }

    /*
     * If this is the first time we've successfully created an SaParam
     * object, intialize the escape table.
     */
    if (gEscapes[0].escapeLen == 0) {
	int i;
	for (i = 0; i < NUM_ESCAPES; i++) {
	    /*
	     * Take this opportunity to enforce our assumption that
	     * each escape string starts with QUOTE_CHAR.
	     */
	    assert(gEscapes[i].escapeStr[0] == QUOTE_CHAR);
	    gEscapes[i].escapeLen = strlen(gEscapes[i].escapeStr);
	}
    }

    return param;
}

/*
 *  void SaParamDestroy(SaParam *param)
 *
 *  Description:
 *      Free memory associated with an SaParam object.
 *
 *  Parameters:
 *      param  SaParam object to free.
 */
void SaParamDestroy(SaParam *param)
{
    int i;
    if (param == NULL) {
	return;
    }

    free(param->paramsByName);
    /*
     * Since we check for malloc failures in SaParamCreate and we
     * don't replace params if realloc fails, this should never be
     * NULL.
     */
    assert(param->params != NULL);

    for (i = 0; i < param->nParams; i++) {
	free(param->params[i]->key);
	free(param->params[i]->value);
	free(param->params[i]);
    }

    free(param->params);
    free(param);
}

/*
 *  static int InternalParamSet(SaParam *param, const char *key,
 *                              const char *value, bool visible)
 *
 *  Description:
 *      Set parameter "key" to "value".  InternalParamSet just appends
 *      a new SaKeyValue to the end of "param"'s array.  Compress(),
 *      which is called whenever the client tries to access key/value
 *      pairs in "param", will take care of eliminating pairs with
 *      duplicate keys.
 *
 *  Parameters:
 *      param    Place to hold parameters.
 *      key      key for this parameter.
 *      value    value for this parameter.
 *      visible  if false, we need to hide this parameter from log
 *               files and "ps" output.  It might be a password, for
 *               example.
 *
 *  Returns:
 *	0 if successful, -1 if no memory is available.
 */
static int InternalParamSet(SaParam *param, const char *key,
			    const char *value, bool visible)
{
    SaKeyValue *keyVal, **params;

    assert(param != NULL);

    /* Make our array of parameters bigger if necessary */
    if (param->nParams == param->nAlloc) {
	params = realloc(param->params,
			 (param->nAlloc + ALLOC_INCR) *
			 sizeof *param->params);
	if (params == NULL) {
	    return -1;
	}

	param->nAlloc += ALLOC_INCR;
	param->params = params;
    }

    /* Add the new parameter */
    keyVal = malloc(sizeof *keyVal);
    if (keyVal == NULL) {
	return -1;
    }
    keyVal->key = strdup(key);
    keyVal->value = strdup(value);
    if (keyVal->key == NULL || keyVal->value == NULL) {
	free(keyVal->key);
	free(keyVal->value);
	free(keyVal);
	return -1;
    }
    keyVal->position = param->nextPosition++;
    keyVal->visible = visible;

    param->params[param->nParams++] = keyVal;
    param->dirty = 1;
    return 0;
}

/*
 *  int SaParamSet(SaParam *param, const char *key, const char *value)
 *
 *  Description:
 *      Set parameter for "key" to "value".
 *
 *  Parameters:
 *      param  Parameter container.
 *      key    key of parameter to set.
 *      value  value that parameter is to have.
 *
 *  Returns:
 *	0 if successful, -1 otherwise.
 */
int SaParamSet(SaParam *param, const char *key, const char *value)
{
    return InternalParamSet(param, key, value, true);
}

/*
 *  int SaParamSet(SaParam *param, const char *key, const char *value)
 *
 *  Description:
 *      Set parameter for "key" to "value".  The parameter is marked
 *      invisible, which will hide it from log files and ps output
 *      when the SaExecArgs functions are used.
 *
 *  Parameters:
 *      param  Parameter container.
 *      key    key of parameter to set.
 *      value  value that parameter is to have.
 *
 *  Returns:
 *	0 if successful, -1 otherwise.
 */
int SaParamSetHidden(SaParam *param, const char *key, const char *value)
{
    return InternalParamSet(param, key, value, false);
}

/*
 *  const char *SaParamGet(SaParam *param, const char *key)
 *
 *  Description:
 *      Retrieve a parameter by key.
 *
 *  Parameters:
 *      param  Parameter container.
 *      key    key of parameter to lookup.
 *
 *  Returns:
 *	value corresponding to key.  The pointer returned is owned by
 *	"param", and will remain valid as long as SaParamDestroy() or
 *	SaParamSet() for this key are not called.
 */
const char *SaParamGet(SaParam *param, const char *key)
{
    int start, end, middle, compare;

    assert(param != NULL);
    if (Compress(param) == -1) {
	return NULL;
    }

    start = 0;
    end = param->nParams;

    /*
     * Binary search.
     */
    while (start < end) {
	middle = start + (end - start) / 2;
	compare = strcmp(param->paramsByName[middle]->key, key);
	if (compare == 0) {
	    return param->paramsByName[middle]->value;
	} else if (compare < 0) {
	    start = middle + 1;
	} else {
	    end = middle;
	}
    }
    return NULL;
}

/*
 *  int SaParamSend(SaParam *param, int fd)
 *
 *  Description:
 *      Send parameters on file descriptor "fd".
 *
 *  Parameters:
 *      param  Parameters to send.
 *      fd     File descriptor to send parameters on.
 *
 *  Returns:
 *	0 if successful, -1 if error.  This can fail if we run out of
 *	memory or if one of the calls to write() fails.
 */
int SaParamSend(SaParam *param, int fd)
{
    int i, status;
    char *key, *value;
    char buf[NUM_BUF_SIZE];

    assert(param != NULL);
    if (Compress(param) == -1) {
	return -1;
    }
    
    for (i = 0; i < param->nParams; i++) {
	key = Quote(param->params[i]->key);
	value = Quote(param->params[i]->value);
	if (key == NULL || value == NULL) {
	    free(key);
	    free(value);
	    return -1;
	}

	/*
	 * For each parameter, we write:
	 * 	- An 8 character hexadecimal ascii representation of
	 *        the number of bytes taken up by the key, the value,
	 *     	  an equal sign, and a newline.
	 * 	- A space.
	 * 	- The quoted key.
	 * 	- An equal sign.
	 * 	- The quoted value.
	 * 	- A newline character.
	 */
	SaStringFormat(buf, sizeof buf, "%08x ",
		       strlen(key) + strlen(value) + 2);
	assert(strlen(buf) == NUM_BUF_SIZE - 1);
	status = SaWriteFully(fd, buf, strlen(buf)) == strlen(buf) &&
	    SaWriteFully(fd, key, strlen(key)) == strlen(key) &&
	    SaWriteFully(fd, "=", 1) == 1 &&
	    SaWriteFully(fd, value, strlen(value)) == strlen(value) &&
	    SaWriteFully(fd, "\n", 1) == 1;

	free(key);
	free(value);

	if (!status) {
	    return -1;
	}
    }

    /* Parameter list is terminated by a line with just a 0 on it */
    #define END_STR "00000000\n"
    if (SaWriteFully(fd, END_STR, strlen(END_STR)) != strlen(END_STR)) {
	return -1;
    }

    return 0;
}

/*
 *  int SaParamReceive(SaParam *param, int fd)
 *
 *  Description:
 *      Receive parameters over the file descriptor "fd".
 *
 *  Parameters:
 *      param  Place to receive parameters into.
 *      fd     File descriptor from which to receive parameters.
 *
 *  Returns:
 *	0 if successful, -1 if error.  This function can fail if we
 *	run out of memory or if a call to read() fails.
 */
int SaParamReceive(SaParam *param, int fd)
{
    char numBuf[NUM_BUF_SIZE];
    int nread, status;
    long size;
    char *buf;

    assert(param != NULL);

    for (;;) {
	/* Read the size in hex of the rest of the line */
	nread = SaReadFully(fd, numBuf, sizeof numBuf - 1);
	if (nread != sizeof numBuf - 1) {
	    return -1;
	}

	numBuf[sizeof numBuf - 1] = '\0';

	/* "16" means use hexadecimal */
	size = strtol(numBuf, NULL, 16);

	if (size < 0) {
	    return -1;
	}

	if (size == 0) {
	    return 0;
	}

	/* Allocate a buffer in which to read the rest of the line */
	buf = malloc(size);
	if (buf == NULL) {
	    return -1;
	}

	/* Read the rest of the line */
	nread = SaReadFully(fd, buf, size);
	if (nread != size || buf[size - 1] != '\n') {
	    free (buf);
	    return -1;
	}

	/* NULL-terminate by replacing the newline */
	buf[size - 1] = '\0';

	status = ParseKeyValue(param, buf);
	free(buf);
	if (status == -1) {
	    return -1;
	}
    }
}

/*
 *  int SaParamParseArgs(SaParam *param, int argc, char *argv[])
 *
 *  Description:
 *      Parse key/value pairs from the command line.  If
 *      SaPARAM_INPUT_ARG is specified on the command line, also read
 *      key/value pairs from standard input.
 *
 *  Parameters:
 *      param  Gets filled in with key value pairs from the command line.
 *      argc   Number of arguments.
 *      argv   Argument vector.
 *
 *  Returns:
 *	0 if successful, -1 if memory is exhausted, SaParamReceive()
 *	fails, or a misformatted argument is encountered.
 */
int SaParamParseArgs(SaParam *param, int argc, char *argv[])
{
    char *equal, *arg;
    int status;

    assert(param != NULL);

    while (--argc) {
	++argv;
	equal = strchr(*argv, '=');
	if (equal != NULL) {
	    arg = strdup(*argv);
	    if (arg == NULL) {
		return -1;
	    }

	    status = ParseKeyValue(param, arg);
	    free(arg);
	    if (status == -1) {
		return -1;
	    }
	} else if (strcmp(*argv, SaPARAM_INPUT_ARG) == 0) {
	    if (SaParamReceive(param, 0) == -1) {
		return -1;
	    }
	} else {
	    return -1;
	}
    }

    return 0;
}

/*
 *  static int CreatePipeArgs(SaParamExecArgs *args)
 *
 *  Description:
 *      Create the pipeArgs filed of SaParamExecArgs if it does not
 *      already exist.
 *
 *  Parameters:
 *      args  SaExecArgs to create pipeArgs for.
 *
 *  Returns:
 *	0 if successful, -1 if error.
 */
static int CreatePipeArgs(SaParamExecArgs *args)
{
    if (args->pipeArgs != NULL) {
	return 0;
    }
    args->pipeArgs = SaParamCreate();
    if (args->pipeArgs == NULL) {
	return -1;
    } else {
	char *argBuf = strdup(SaPARAM_INPUT_ARG);
	if (argBuf == NULL) {
	    return -1;
	} else {
	    args->argList[args->numArgs++] = argBuf;
	}
    }
    return 0;
}

/*
 *  static int AddArg(SaParamExecArgs *args, size_t *argListSize,
 *  		      const char *key, const char* value, bool visible)
 *
 *  Description:
 *      Helper function for SaParamExecCreateV().  Add one argument
 *      either to args->argList or args->params, depending on whether
 *      there's room left in args->argList (as indicated by
 *      "*argListSize").
 *
 *      AddArg() is used with a "value" of NULL for handling the initial
 *      arguments passed to SaParamExecCreateV().  If these initial
 *      arguments overflow the command line AddArg() will fail because
 *      we have no way of storing these arguments in an SaParam
 *      structure and passing them to another program in a meaningful
 *      way.
 *
 *  Parameters:
 *      args          SaParamExecArgs to add this arg to.
 *      argListSize   value/result; size of args added so far.
 *      key           key to add.
 *      value         value to add.  If this is NULL, then "key" is
 *                    the entire argument.
 *      visible       If false, this arg needs to go into pipeArgs
 *                    so it won't show up on "ps" command line or in
 *                    log files.
 *
 *  Returns:
 *	0 if successful, -1 if error.
 */
static int AddArg(SaParamExecArgs *args, size_t *argListSize,
		  const char *key, const char *value, bool visible)
{
    char *quoteKey, *quoteValue, *argBuf;
    int status;
    size_t argSize;

    /*
     * if args->overFlow is true, it means that we've reached the
     * maximum size of what we can pass on the command line.
     * Everything after that goes into a new SaParam struct for
     * sending via stdin.
     */
    if (args->overFlow) {
	/*
	 * We can't properly deal with the case that the args passed
	 * to SaParamExecArgsCreate + environment won't fit in gArgMax
	 * because we have no meaningful way of transmitting
	 * value-less arguments.
	 */
	if (value == NULL) {
	    return -1;
	}
	assert(args->pipeArgs != NULL);
	return SaParamSet(args->pipeArgs, key, value);
    } else if (!visible) {
	status = CreatePipeArgs(args);
	if (status == 0) {
	    status = SaParamSet(args->pipeArgs, key, value);
	}
	return status;
    }

    quoteKey = Quote(key);
    quoteValue = value ? Quote(value) : NULL;
    if (quoteKey == NULL || value != NULL && quoteValue == NULL) {
	status = -1;
    } else {
	argSize = strlen(quoteKey) + 1;	       /* +1 for '\0' */ 
	if (quoteValue != NULL) {
	    argSize += strlen(quoteValue) + 1; /* +1 for '=' */
	}
	if (*argListSize + argSize + strlen(SaPARAM_INPUT_ARG) + 1 > gArgMax) {
	    if (value == NULL) {
		/*
		 * We can't properly deal with the case that the
		 * args passed to SaParamExecArgsCreate +
		 * environment won't fit in gArgMax because we have no
		 * meaningful way of transmitting value-less arguments.
		 */
		status = -1;
	    } else {
		/*
		 * We've filled up the command line.  Subsequent
		 * parameters need to go into args->pipeArgs.
		 */
		assert(!args->overFlow);
		args->overFlow = true;
		status = CreatePipeArgs(args);
		if (status != -1) {
		    status = SaParamSet(args->pipeArgs, key, value);
		}
	    }
	} else {
	    argBuf = malloc(argSize);
	    if (argBuf == NULL) {
		status = -1;
	    } else {
		if (quoteValue != NULL) {
		    SaStringFormat(argBuf, argSize, "%s=%s",
				   quoteKey, quoteValue);
		} else {
		    strcpy(argBuf, quoteKey);
		}
		args->argList[args->numArgs++] = argBuf;
		*argListSize += argSize;
		status = 0;
	    }
	}
    }
    free(quoteKey);
    free(quoteValue);
    return status;
}

/*
 *  size_t SaParamGetEnvSize(const char *const *env)
 *
 *  Description:
 *      Calculate how much space "env" takes up.
 *
 *  Parameters:
 *      env  Environment pointer.
 *
 *  Returns:
 *	Number of bytes to pass "env" across a call to exec().
 */
size_t SaParamGetEnvSize(const char *const *env)
{
    int size = 0;
    const char *const *p;

    for (p = env; p != NULL && *p != NULL; p++) {
	size += strlen(*p) + 1;		/* +1 for '\0' terminator */ 
    }

    return size;
}    

/*
 *  SaParamExecArgs *SaParamExecArgsCreate(SaParam *param,
 *  				           const char *const *env, ...)
 *
 *  Description:
 *      Create an SaParamExecArgs structure for use with the exec
 *      system call.  See SaParam.h for more details.
 *
 *  Parameters:
 *      param  SaParam we want to pass to another command.
 *      env    Environment pointer that will be passed to command.
 *      ...    Initial arguments that will be passed to command.
 *
 *  Returns:
 *	SaParamExecArgs pointer if successful, NULL if error.  Fails
 *      if we run out of memory or if the initial arguments + the
 *      environment won't fit within sysconf(_SC_ARG_MAX) bytes.
 */
SaParamExecArgs *SaParamExecArgsCreate(SaParam *param,
				       const char *const *env, ...)
{
    va_list ap;
    int argc;
    SaParamExecArgs *args;
    const char *arg;
    const char **argv;
    
    argc = 0;
    va_start(ap, env);
    while ((arg = va_arg(ap, const char *)) != NULL) {
	argc++;
    }

    argv = malloc((argc + 1) * sizeof *argv);

    argc = 0;
    va_start(ap, env);
    while ((arg = va_arg(ap, const char *)) != NULL) {
	argv[argc++] = arg;
    }
    va_end(ap);

    argv[argc] = NULL;
    args = SaParamExecArgsCreateV(param, argc, argv, env);
    free(argv);

    return args;
}

/*
 *  SaParamExecArgs *SaParamExecArgsCreateV(SaParam *param,
 *  					    int argc,
 *  					    const char *const *argv,
 *  					    const char *const *env)
 *
 *  Description:
 *      varargs version of SaParamExecArgs().  See SaParam.h for more
 *      details. 
 *
 *  Parameters:
 *      param  SaParam we want to pass to another command.
 *      argc   Number of initial arguments that will be passed.
 *      argv   Vector of initial arguments that will be passed.
 *      env    Environment pointer that will be passed to command.
 *
 *  Returns:
 *	SaParamExecArgs pointer if successful, NULL if error.  Fails
 *      if we run out of memory or if the initial arguments + the
 *      environment won't fit within sysconf(_SC_ARG_MAX) bytes.
 */
SaParamExecArgs *SaParamExecArgsCreateV(SaParam *param,
					int argc,
					const char *const *argv,
					const char *const *env)
{
    int numAlloc;
    char **argList;
    size_t argListSize = SaParamGetEnvSize(env);
    SaParamExecArgs *args;
    SaParamIter *iter;
    const char *key, *value;

    if (gArgMax == 0) {
	gArgMax = sysconf(_SC_ARG_MAX);
    }

    if (Compress(param) == -1) {
	return NULL;
    }

    numAlloc = param->nParams + argc + 1;	/* NULL terminator */

    /*
     * Allocate a new SaParamExecArgs struct.
     */
    argList = malloc(sizeof *argList * numAlloc);
    if (argList == NULL) {
	return NULL;
    }
    args = malloc(sizeof *args);
    if (args == NULL) {
	free(argList);
	return NULL;
    }
    args->argList = argList;
    args->numArgs = 0;
    args->overFlow = false;
    args->pipeArgs = NULL;

    /*
     * Set arguments based on what was passed to us.
     */
    while (argc--) {
	if (AddArg(args, &argListSize, *argv++, NULL, true) == -1) {
	    SaParamExecArgsDestroy(args);
	    return NULL;
	}
    }

    /*
     * Continue setting arguments based on "params".
     */
    iter = SaParamIterCreate(param);
    while ((key = SaParamIterGetKey(iter)) != NULL) {
	value = SaParamIterGetValue(iter);
	if (AddArg(args, &argListSize, key, value,
		   iter->param->params[iter->index]->visible) == -1) {
	    SaParamExecArgsDestroy(args);
	    return NULL;
	}
    }
    SaParamIterDestroy(iter);

    assert(args->numArgs < numAlloc);
    args->argList[args->numArgs] = NULL;

    /*
     * Since we compressed before adding anything to pipeArgs, it's
     * still compressed.
     */
    if (args->pipeArgs != NULL) {
	args->pipeArgs->dirty = 0;
    }

    return args;
}

/*
 *  void SaParamExecArgsDestroy(SaParamExecArgs *args)
 *
 *  Description:
 *      Free an SaParamExecArgs struct allocated via
 *      SaParamExecArgsCreate().
 *
 *  Parameters:
 *      args  SaParamExecArgs struct to free.
 */
void SaParamExecArgsDestroy(SaParamExecArgs *args)
{
    int ii;
    if (args != NULL) {
	for (ii = 0; ii < args->numArgs; ii++) {
	    free(args->argList[ii]);
	}
	free(args->argList);
	SaParamDestroy(args->pipeArgs);
    }
    free(args);
}

/*
 *  char *const *SaParamExecArgsGet(SaParamExecArgs *args)
 *
 *  Description:
 *      Get argument list suitable for passing to execv().
 *
 *  Parameters:
 *      args  Contains the argument list.
 *
 *  Returns:
 *	Argument list for passing to execv().
 */
char *const *SaParamExecArgsGet(SaParamExecArgs *args)
{
    assert(args != NULL);
    return args->argList;
}

/*
 *  int SaParamExecArgsSendNeeded(SaParamExecArgs *args)
 *
 *  Parameters:
 *      args  SaParamExecArgs that we're checking.
 *
 *  Returns:
 *	non-zero if SaParamExecArgsSend() should be called, 0 if it
 *      does not need to be called.
 */
int SaParamExecArgsSendNeeded(SaParamExecArgs *args)
{
    assert(args != NULL);
    return args->pipeArgs != NULL;
}

/*
 *  int SaParamExecArgsSend(SaParamExecArgs *args, int fd)
 *
 *  Description:
 *      Send pipeArgs arguments to a command via "fd".  "fd" is
 *      presumably the write end of the command's standard input.
 *
 *  Parameters:
 *      args  SaParamExecArgs to send pipeArgs of.
 *      fd    File descriptor to write pipeArgs onto.
 *
 *  Returns:
 *	0 if successful, -1 if error.
 */
int SaParamExecArgsSend(SaParamExecArgs *args, int fd)
{
    assert(args != NULL);
    if (args->pipeArgs == NULL) {
	return 0;
    }
    return SaParamSend(args->pipeArgs, fd);
}

/*
 *  SaParamIter *SaParamIterCreate(SaParam *param)
 *
 *  Description:
 *      Create an iterator for an SaParam object.
 *
 *  Parameters:
 *      param  SaParam object to iterate over.
 *
 *  Returns:
 *	An iterator for SaParam.  Returns NULL if no memory is
 *	available.
 */
SaParamIter *SaParamIterCreate(SaParam *param)
{
    SaParamIter *iter;

    if (Compress(param) == -1) {
	return NULL;
    }

    iter = malloc(sizeof *iter);

    if (iter != NULL) {
	iter->param = param;
	/*
	 * We start with index == -1 because we want to keep track of
	 * the current element at all times, so that
	 * SaParamIterGetValue() gets the value corresponding to the
	 * last key returned by SaParamIterGetKey().
	 */
	iter->index = -1;
    }
    return iter;
}

/*
 *  void SaParamIterDestroy(SaParamIter *iter)
 *
 *  Description:
 *      Free the memory associated with an SaPeramIter.
 *
 *  Parameters:
 *      iter  Iterator to destroy.
 */
void SaParamIterDestroy(SaParamIter *iter)
{
    free(iter);
}

/*
 *  const char *SaParamIterGetKey(SaParamIter *iter)
 *
 *  Description:
 *      Get the next key in "iter".
 *
 *  Parameters:
 *      iter  Iterator to get next key of.
 *
 *  Returns:
 *	The next key.
 */
const char *SaParamIterGetKey(SaParamIter *iter)
{
    assert(iter != NULL);

    if (iter->index == iter->param->nParams) {
	return NULL;
    }

    /*
     * Since we don't increment "index" beyond the end and we don't
     * have a facility for removing a key/value pair, "index" should
     * never be greater than nParams.
     */
    assert(iter->index < iter->param->nParams);

    if (++(iter->index) < iter->param->nParams) {
	assert(iter->index >= 0);
	return iter->param->params[iter->index]->key;
    }
    return NULL;
}

/*
 *  const char *SaParamIterGetValue(SaParamIter *iter)
 *
 *  Description:
 *      Get the value corresponding to the last key returned by
 *      SaParamIterGetKey().  This is more efficient that calling
 *      SaParamGet() because the SaParamIter knows where we are with
 *      respect to the corresponding SaParam object.
 *
 *  Parameters:
 *      iter  Iterator object to get value of.
 *
 *  Returns:
 *	value corresponding to the last key returned by
 *	SaParamIterGetKey() for this SaParamIter.
 */
const char *SaParamIterGetValue(SaParamIter *iter)
{
    assert(iter != NULL);

    if (iter->index < iter->param->nParams) {
	assert(iter->index >= 0);
	return iter->param->params[iter->index]->value;
    }
    return NULL;
}

/*
 *  void SaParamIterReset(SaParamIter *iter)
 *
 *  Description:
 *      Reset an Iterator back to the beginning.
 *
 *  Parameters:
 *      iter  Iterator to reset.
 */
void SaParamIterReset(SaParamIter *iter)
{
    assert(iter != NULL);
    iter->index = -1;
}
