/*
 *  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */
/*
 * SaParam.h
 *
 *	Interface for passing lists of key/value pairs across command
 *	invocations.  key/value pairs can be passed on the command line,
 *	in which case each argument on the command line must be of the
 *	form "key=value", or they can be passed via file descriptor 0
 *	of the command being run.  When key/value pairs are passed via
 *	file descriptor, the parameter "-input" (SaPARAM_INPUT_ARG)
 *	should be specified on the command line.
 *
 *	This API is used by both the caller of the command and the
 *	command itself.  On the calling side, use is as follows:
 *	
 *	// Create an SaParam object
 *	SaParam *params = SaParamCreate();
 *	
 *	// Set parameters for the command
 *	SaParamSet(params, "name", "rogerc");
 *	SaParamSet(params, "uid", "3099");
 *	
 *	// Create a pipe to be used to pass parameters (error code
 *	// omitted)
 *	int pipes[2];
 *	pipe(pipes);
 *	
 *	// Create a child process to run the command (error code
 *	// omitted)
 *	pid = fork();
 *	if (pid == 0) {
 *	    dup2(pipes[0], 0);
 *	    if (pipes[0] != 0)
 *	    {
 *	        close(pipes[0]);
 *	    }
 *	    close(pipes[1]);
 *	    execl(command, command, SaPARAM_INPUT_ARG, NULL);
 *	    perror(command);
 *	    exit(1);
 *	}
 *	
 *	// In the parent process, pass the parameters to the child
 *	close(pipes[0]);
 *	SaParamSend(params, pipes[1]);
 *	
 *	On the command side, use is like this:
 *	
 *	int main(int argc, char *argv[])
 *	{
 *	    const char *name = NULL;
 *	    const char *uid = NULL;
 *	
 *	    // Create params object
 *	    SaParam *params = SaParamCreate();
 *	
 *	    // Parse the command line.
 *	    SaParamParseArgs(params, argc, argv);
 *	
 *	    name = SaParamGet(params, "name");
 *	    uid = SaParamGet(params, "uid");
 *	    
 *	    ...
 *	}
 */

#ident "$Revision: 1.7 $"

#ifndef _SA_PARAM_H
#define _SA_PARAM_H
#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>

/*
 * If SaParamParseArgs() sees this argument, it will read arguments
 * from file descriptor 0 in addition to the command line.
 */
#define SaPARAM_INPUT_ARG "-input"

typedef struct _SaParam SaParam;
typedef struct _SaParamIter SaParamIter;
typedef struct _SaParamExecArgs SaParamExecArgs;

/*
 * Creating and destroying parameters.  SaParamCreate() returns NULL
 * if malloc() fails.
 */
extern SaParam *SaParamCreate(void);
extern void SaParamDestroy(SaParam *param);

/*
 * Getting and setting parameters.  SaParamSet() returns 0 if
 * successful, -1 if malloc() fails.  SaParamGet() returns NULL if
 * there is no value for "key".
 */
extern int SaParamSet(SaParam *param, const char *key, const char *value);
extern const char *SaParamGet(SaParam *param, const char *key);

/*
 * SaParamSetHidden is used to hide sensitive parameters from log
 * files.  If SaParamSetHidden(param, "key", "value") is called, then
 * the SaExecArgs routines will pass the parameter for "key" via stdin
 * rather than on the command line.
 */
extern int SaParamSetHidden(SaParam *param, const char *key, const
			     char *value);
			     

/*
 * Sending and receiving parameters.  SaParamSend() returns 0 if
 * successful, -1 if it cannot write to fd.  Both methods return -1 if
 * malloc() fails.
 */
extern int SaParamSend(SaParam *param, int fd);
extern int SaParamReceive(SaParam *param, int fd);

/*
 * Parse command line arguments of the form "key=value" into key/value
 * pairs.  In any key/value pair, if "value" is the empty string then
 * subsequent calls to SaParamGet() for that key will return NULL.  If
 * the argument SaPARAM_INPUT_ARG is encountered, key value pairs are
 * read from file descriptor 0 as well by calling SaParamReceive().
 * 
 * Returns 0 if successful, -1 if memory is exhausted, if read fails,
 * or if an unrecognized argument is encountered.
 */
extern int SaParamParseArgs(SaParam *param, int argc, char *argv[]);

/*
 * Pass arguments to another command via exec().
 */
/*
 * SaParamExecArgsCreate() breaks "params" into two chunks.  The first
 * chunk is returned by SaParamExecArgsGet() and should be passed to
 * the execv() system call as its second parameter.  The second chunk
 * should be sent to the command's standard input by calling
 * SaParamExecArgsSend().
 *
 * The dividing line between the two chunks is the point at which the
 * size of the arguments + the size of the environment reaches the
 * maximum size supported by the kernel.  If all of the parameters fit
 * on the command line, the second chunk will be empty.
 *
 * Any parameters that were set with SaParamSetHidden() are passed in
 * the second chunk, regardless of whether the args fit.
 * 
 * The "env" argument to SaParamExecArgsCreate() is the environment
 * which will be used for the new command, and is used to calculate
 * the maximum argument size.  The arguments following "env" are the
 * initial arguments for the command; SaParamExecArgsCreate() will
 * append the arguments corresponding to "param".  The last argument
 * to SaParamExecArgsCreate() should be NULL.
 */
extern SaParamExecArgs *SaParamExecArgsCreate(SaParam *param,
					      const char *const *env,
					      ...);

/*
 * SaPramExecArgsCreateV() takes arguments in argc, argv form instead
 * of as varargs. 
 */
extern SaParamExecArgs *SaParamExecArgsCreateV(SaParam *param,
					       int argc,
					       const char *const *argv,
					       const char *const *env);
extern void SaParamExecArgsDestroy(SaParamExecArgs *args);

/* 
 * SaParamExecArgsGet() returns the first chunk in a form suitable for
 * passing to execv().  The return value is owned by libsysadmParam
 * and should not be modified or freed by the caller.  The return
 * value remains valid until SaParamExecArgsDestroy() is called.
 * 
 * SaParamExecArgsSend() sends the second chunk over "fd".  "fd" is
 * assumed to the write end of a pipe which is the command's standard
 * input.
 *
 * SaParamExecArgsSendNeeded() can be used to determine whether a call
 * to SaParamExecArgsSend() is necessary.  If
 * SaParamExecArgsSendNeeded() returns 0, then there is no need to
 * call SaParamExecArgsSend() (although it doesn't hurt).
 */
extern char *const *SaParamExecArgsGet(SaParamExecArgs *args);
extern int SaParamExecArgsSendNeeded(SaParamExecArgs *args);
extern int SaParamExecArgsSend(SaParamExecArgs *args, int fd);

/*
 * Utility function for measuring the size of an environment.
 */
extern size_t SaParamGetEnvSize(const char *const *env);

/*
 * Iterating over the keys.
 */
extern SaParamIter *SaParamIterCreate(SaParam *param);
extern void SaParamIterDestroy(SaParamIter *iter);
extern const char *SaParamIterGetKey(SaParamIter *iter);
extern const char *SaParamIterGetValue(SaParamIter *iter);
extern void SaParamIterReset(SaParamIter *iter);

#ifdef __cplusplus
}
#endif
#endif /* _SA_PARAM_H */
