//
// AppContext.c++
//
//	application context for non-X programs.
//
//
//  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.9 $"

#include <iostream.h>

#include <sys/types.h>
#include <sys/time.h>
#include <sys/select.h>

#include <errno.h>
#include <unistd.h>
#include <stdio.h>

#include <sysadm/AppContext.h>
#include <sysadm/OrderedCollectionOf.h>
#include <sysadm/DictionaryOf.h>

BEGIN_NAMESPACE(sysadm);

static int gCurrentId = 0;
static AppContext* gAppContext = NULL;

//
// Monitor objects are used to store information about a file
// descriptor and its corresponding callback and client data.
//
class Monitor {
  public:
    Monitor(int fd, void* client,
	    AppContext::MonitorCallback func,
	    AppContext::EAccess access)
    : _fd(fd), _client(client), _func(func), _deleted(false),
      _access(access), _id(++gCurrentId)
    {
    }

    int _fd;
    void* _client;
    AppContext::MonitorCallback _func;
    AppContext::EAccess _access;
    bool _deleted;
    int _id;
};

//
// Timer objects are used to store information about a timeout and
// its corresponding callback and client data.
//
class Timer {
  public:
    Timer(unsigned long sec, void* client,
	  AppContext::TimerCallback func)
    : _sec(sec), _client(client), _func(func), _deleted(false),
      _id(++gCurrentId)
    {
    }

    unsigned long _sec;
    void* _client;
    AppContext::TimerCallback _func;
    bool _deleted;
    int _id;
};

//
// Opaque implementation of AppContext.
//
class AppContextImpl {
  public:

    AppContextImpl()
    : _maxFd(-1), _exitCode(0), _needToExit(false), _propNum(0)
    {
	FD_ZERO(&_readFds);
	FD_ZERO(&_writeFds);
    }

    ~AppContextImpl() {
	RemoveAndDestroyContentsOf(&_properties);
    }

    class Property {
      public:
	String _value;
	int _num;
	Property(String value, int num) : _value(value), _num(num) { }
    };

    void reapMonitors();
    void reapTimers();

    OrderedCollectionOf<Monitor> _monitors;
    OrderedCollectionOf<Timer> _timers;
    DictionaryOf<Property> _properties;
    fd_set _readFds;
    fd_set _writeFds;
    int _maxFd;
    int _exitCode;
    bool _needToExit;
    int _propNum;
};

//
//  void AppContextImpl::reapMonitors()
//
//  Description:
//      Delete all Monitor objects corresponding to removed monitors.
//      Monitors are reaped periodically rather than deleted in
//      unregisterMonitor() to allow unregisterMonitor() to be safely
//      called from within the callback function it's unregistering.
//
void AppContextImpl::reapMonitors()
{
    IteratorOver<Monitor> iter(&_monitors);
    Monitor* monitor;

    while ((monitor = iter()) != NULL) {
	if (monitor->_deleted) {
	    delete iter.remove();
	}
    }
}

//
//  void AppContextImpl::reapTimers()
//
//  Description:
//      Delete all Timer objects corresponding to removed timouts.
//      Timers are reaped periodically rather than deleted in
//      unregiterTimer() to allow unregisterTimer() to be safely
//      called from within the callback function it's unregistering.
//
void AppContextImpl::reapTimers()
{
    IteratorOver<Timer> iter(&_timers);
    Timer* timer;

    while ((timer = iter()) != NULL) {
	if (timer->_deleted) {
	    delete iter.remove();
	}
    }
}

//
// AppContext constructor.
//
AppContext::AppContext()
: _impl(new AppContextImpl)
{
    assert(gAppContext == NULL);
    gAppContext = this;
}

//
// AppContext destructor.
//
AppContext::~AppContext()
{
    delete _impl;
    assert(this == gAppContext);
    gAppContext = NULL;
}

//
//  int AppContext::registerMonitor(int fd, MonitorCallback callback,
//  				void * clientData, EAccess access)
//
//  Description:
//      Register a monitor callback for a file descriptor.
//
//  Parameters:
//      fd           file descriptor to monitor.
//      callback     method to call when something of interest happens.
//      clientData   client data to pass to callback method.
//      access       whether to monitor for input or output.
//
//  Returns:
//	An ID which can be used to remove the monitor at a later
//	time.
//
int AppContext::registerMonitor(int fd, MonitorCallback callback,
				void * clientData, EAccess access)
{
    if (access == INPUT) {
	// It's an error to add same file descriptor twice.
	assert(!FD_ISSET(fd, &_impl->_readFds));
	FD_SET(fd, &_impl->_readFds);
    } else {
	// It's an error to add same file descriptor twice.
	assert(!FD_ISSET(fd, &_impl->_writeFds));
	FD_SET(fd, &_impl->_writeFds);
    }
    
    if (fd + 1 > _impl->_maxFd) {
	_impl->_maxFd = fd + 1;
    }

    Monitor* monitor = new Monitor(fd, clientData, callback, access);
    _impl->_monitors.add(monitor);
    return monitor->_id;
}

//
//  void AppContext::unregisterMonitor(int id)
//
//  Description:
//      Stop monitoring a file descriptor.
//
//  Parameters:
//      id  id of file descriptor monitor to stop.
//
void AppContext::unregisterMonitor(int id)
{
    IteratorOver<Monitor> iter(&_impl->_monitors);
    Monitor* monitor = NULL;
    while ((monitor = iter()) != NULL) {
	if (monitor->_id == id) {
	    if (monitor->_access == INPUT) {
		FD_CLR(monitor->_fd, &_impl->_readFds);
	    } else {
		FD_CLR(monitor->_fd, &_impl->_writeFds);
	    }
	    monitor->_deleted = true;
	}
    }
}

//
//  int AppContext::registerTimer(unsigned long seconds,
//  				  TimerCallback callback,
//  				  void * clientData)
//
//  Description:
//      Arrange to have "callback" called after "seconds" seconds have
//      expired.
//
//  Parameters:
//      seconds      Number of seconds to wait.
//      callback     Function to call after "seconds" have elapsed.
//      clientData   Data to pass to "callback".
//      
//  Returns:
//      An id which can be used to remove the timer.
//
int AppContext::registerTimer(unsigned long seconds,
			      TimerCallback callback,
			      void * clientData)
{
    time_t currentTime = time(NULL);
    unsigned long timeOutVal = currentTime + seconds;
    Timer* newTimer = new Timer(timeOutVal, clientData, callback);    

    // Keep the list sorted by time, with the smaller ones first.
    // This is important in run(), below.
    IteratorOver<Timer> iter(&_impl->_timers);
    Timer* timer = NULL;
    while ((timer = iter()) != NULL) {
	if (timeOutVal < timer->_sec) {
	    iter.insertBefore(newTimer);
	    return newTimer->_id;
	}
    }

    _impl->_timers.append(newTimer);
    return newTimer->_id;
}

//
//  void AppContext::unregisterTimer(int id)
//
//  Description:
//      Arrange to prevent a timer from being called before it
//      expires.
//
//  Parameters:
//      id  identifies the timer to be removed.
//
void AppContext::unregisterTimer(int id)
{
    IteratorOver<Timer> iter(&_impl->_timers);
    Timer* timer = NULL;
    while ((timer = iter()) != NULL) {
	if (timer->_id == id) {
	    timer->_deleted = true;
	    break;
	}
    }
}

//
//  int AppContext::run()
//
//  Description:
//      Main loop for an AppContext based program.  Use our monitors
//      and timers to produce arguments to select().  Call expired
//      timers and monitors as appropriate.
//
//  Returns:
//	Exit code passed to exit() method.
//
int AppContext::run()
{
    while (!_impl->_needToExit) {
	// Call expired timeouts.
	time_t currentTime = time(NULL);
	IteratorOver<Timer> tIter(&_impl->_timers);
	Timer* timer = NULL;
	while (!_impl->_needToExit &&
	       (timer = tIter()) != NULL && currentTime > timer->_sec) {
	    if (!timer->_deleted) {
		timer->_func(timer->_client, timer->_id);
		timer->_deleted = true;
		timer = NULL;
		currentTime = time(NULL);
	    }
	}

	_impl->reapTimers();

	if (_impl->_needToExit) {
	    break;
	}

	IteratorOver<Timer> iter(&_impl->_timers);
	timer = iter();

	timeval timeOut;
	if (timer) {
	    timeOut.tv_usec = 0;
	    timeOut.tv_sec = (time_t)timer->_sec - currentTime;
	}

	fd_set rfds = _impl->_readFds;
	fd_set wfds = _impl->_writeFds;
	
	int nfds = select(_impl->_maxFd, &rfds, &wfds, NULL,
			  timer ? &timeOut : NULL);
	if (nfds == -1) {
	    if (errno == EINTR) {
		continue;
	    }
	    perror("select");
	    exit(1);
	}

	// Call monitors.
	IteratorOver<Monitor> mIter(&_impl->_monitors);
	Monitor* monitor = NULL;
	while (!_impl->_needToExit
	       && nfds > 0 && (monitor = mIter()) != NULL) {
	    if (!monitor->_deleted && monitor->_access == INPUT
		&& FD_ISSET(monitor->_fd, &rfds)) {
		monitor->_func(monitor->_client, monitor->_id, monitor->_fd);
		nfds--;
	    }
	}

	mIter.reset();
	while (!_impl->_needToExit
	       && nfds > 0 && (monitor = mIter()) != NULL) {
	    if (!monitor->_deleted && monitor->_access == OUTPUT
		&& FD_ISSET(monitor->_fd, &wfds)) {
		monitor->_func(monitor->_client, monitor->_id, monitor->_fd);
		nfds--;
	    }
	}

	_impl->reapMonitors();
    }

    return _impl->_exitCode;
}

//
//  void AppContext::exit(int exitCode)
//
//  Description:
//      Inform AppContext that it's time to quit.  This will kick us
//      out of our run() loop.
//
//  Parameters:
//      exitCode  Application exit code.
//
void AppContext::exit(int exitCode)
{
    _impl->_exitCode = exitCode;
    _impl->_needToExit = true;
}

//
//  void AppContext::setProperty(String key, String value)
//
//  Description:
//      Set a property on this application context.
//
//  Parameters:
//      key    key of property to set.
//      value  value of property to set.
//
void AppContext::setProperty(String key, String value)
{
    _impl->_properties.add(
	new AppContextImpl::Property(value, _impl->_propNum++),
	new String(key));
}

//
//  String AppContext::getProperty(String key)
//
//  Description:
//      Get a property from this application context.
//
//  Parameters:
//      key  key of property to get.
//
//  Returns:
//	value of property corresponding to key.
//
String AppContext::getProperty(String key)
{
    AppContextImpl::Property* prop = _impl->_properties.lookupByKey(key);
    if (prop == NULL) {
	return String::NUL;
    }
    return prop->_value;
}

//
//  int AppContext::getPropertyNum(String key)
//
//  Description:
//      Get the number of a property from this application context.
//
//  Parameters:
//      key  key of property number to get.
//
//  Returns:
//	Number of this property, or -1 if it doesn't exist.
//
int AppContext::getPropertyNum(String key)
{
    AppContextImpl::Property* prop = _impl->_properties.lookupByKey(key);
    if (prop == NULL) {
	return -1;
    }
    return prop->_num;
}

//
//  AppContext& AppContext::getAppContext()
//
//  Returns:
//	The one and only AppContext object.
//
AppContext& AppContext::getAppContext()
{
    if (!gAppContext) {
	new AppContext;
    }
    return *gAppContext;
}

END_NAMESPACE(sysadm);
