//
// AppContext.h
//
//	An interface for providing access to an application's
//	select(2) loop.  Can be used as-is (with the run() method) or
//	subclassed if using another environment (such as Xt) which
//	owns the event loop. 
//
//
//  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ifndef _SYSADM_APP_CONTEXT_H
#define _SYSADM_APP_CONTEXT_H
#ident "$Revision: 1.7 $"

#include <sysadm/String.h>

BEGIN_NAMESPACE(sysadm);

//
// AppContext provides a run() method to act as the main loop of an
// event driven program.  Methods are called for getting notification
// of file descriptor and timer-based events, which are used to set
// parameters to be passed to select(2) in run().
//
class AppContext {
  public:
    AppContext();
    virtual ~AppContext();

    // Callback type for getting events about file descriptors.
    typedef void (*MonitorCallback)(void* clientData,
				    int id,
				    int fd);

    enum EAccess { INPUT, OUTPUT };

    // Register for events on a file descriptor.  It is an error to
    // add more than one monitor with the same access for one file
    // descriptor.  It's OK to monitor a single file descriptor for
    // both INPUT and OUTPUT (by calling registerMonitor() twice).
    virtual int registerMonitor(int fd,
				MonitorCallback callback,
				void * clientData,
				EAccess access = INPUT);

    // Unregister for events on a file descriptor.
    virtual void unregisterMonitor(int id);


    // Timers are fired once, then removed. They can be reasserted
    // if an interval is required.
    typedef void (*TimerCallback)(void* clientData, int id);

    // Register a timer.
    virtual int registerTimer(unsigned long seconds,
			      TimerCallback callback,
			      void * client_data);

    // Unregister a timer.
    virtual void unregisterTimer(int id);

    // Main select loop.  Runs until exit() is called.  Returns
    // exitCode passed to exit().
    virtual int run();

    // Terminate the main loop.
    virtual void exit(int exitCode);

    // Set a property for this application.
    virtual void setProperty(String key, String value);

    // Get a property for this application.
    virtual String getProperty(String key);

    // Get the number of a property.  This can be used to determine
    // the order in which two properties were set.
    virtual int getPropertyNum(String key);

    // Get access to an instance of AppContext.
    static AppContext& getAppContext();

  private:

    // Intentionally undefined.
    AppContext(const AppContext&);
    AppContext& operator=(const AppContext&);

    class AppContextImpl* _impl;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_APP_CONTEXT_H
