//
// AttrBundle.c++
//
//	Bundle of attributes.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.15 $"

#include <string.h>
#include <sysadm/AttrBundle.h>

#define BUNDLE_VERSION "A"
#define NEST_PAD 4

BEGIN_NAMESPACE(sysadm);

// Element in our table of escape information, which we use to escape
// characters which have special meaning to us.
struct EscapeInfo {
    char toEscape;		// Character to escape
    const char* escapeStr;	// String to replace it with
    int escapeLen;		// Length of the escape string
};

// Table of escape information.  If a key or a value contains any of
// the characters in the column on the left, they are replaced with the
// escape sequence in the column in the middle when transferred across a
// command line or via a file descriptor.  We do this so that keys and
// values can contain characters which are significant in the format
// we use to transfer keys and values from one process to another.
//
// The column on the right is the length of each sequence.
static EscapeInfo gEscapes[] = {
    { '!',	"@21",		3 }, // '!' separates selector from
                                     // first key and keys from types.
    { ':',	"@3a",		3 }, // ':' separates type from selector
                                     // and terminates a key/value pair.
    { '=',	"@3d",		3 }, // '=' separates types from values.
    { '@',	"@40",		3 }, // '@' introduces escape strings.
    { '\n', 	"@0a",		3 }, // Escape newlines for benefit of
                                     // Packet class.
};

END_NAMESPACE(sysadm);

USING_NAMESPACE(sysadm);

//
// Construct an AttrBundle with "type" and "selector".
//
AttrBundle::AttrBundle(const String& type, const String& selector)
: _type(type), _selector(selector)
{
}

//
// Construct an AttrBundle from serialized format.  Serialized format
// is: "type:selector!key1!type1=value1:key2!type2=value2...".  Any
// embedded characters from the set [!:=@\n] are replaced by their
// escaped equivalents @21, @3a, @3d, @40, and @0a.  If any portion of
// the string is mal-formed an assertion will fail.  Two examples of
// serialized AttrBundles are:
// 
//   url:file@3a//sgi.com/etc/motd
//        type        url
//        selector    file://sgi.com/etc/motd
//        attributes  none
//   url:http@3a//www.sgi.com/canvas.gif!mimeType!s=image/gif
//        type        url
//        selector    http://www.sgi.com/canvas.gif
//        attributes  mimeType (string) =image/gif
//
AttrBundle::AttrBundle(const String& stream)
: _type(String::EMPTY), _selector(String::EMPTY)
{
    int streamLen = stream.getLength();
    CharIndex start = 0;

    // Get the version
    CharIndex stop = stream.find(':', start);
    if (stop != String::NotFound) {
	// For now, we just ignore the version.  If the version is ever
	// revved, we can use the version to be backwards compatible.
	start = stop + 1;
    } else {
	return;
    }

    // Get the type
    stop = stream.find(':', start);
    if (stop != String::NotFound) {
	_type = stream.getSubString(start, stop);
	unescape(_type);
	start = stop + 1;
    } else {
	return;
    }

    // now selector
    stop = stream.find(':', start);
    if (stop != String::NotFound) {
	_selector = stream.getSubString(start, stop);
	unescape(_selector);
	start = stop + 1;
    } else {
	_selector = stream.getSubString(start, streamLen);
	unescape(_selector);
	return;
    }

    // Now invisible attrs.
    CharIndex stopInvisibleAttrs = stream.find('!', start);
    if (stopInvisibleAttrs != String::NotFound) {
	while (start < stopInvisibleAttrs) {
	    stop = stream.find(':', start);
	    if (stop == String::NotFound) {
		break;
	    }
	    String key(stream.getSubString(start, stop));
	    unescape(key);
	    setAttrVisible(key, false);
	    start = stop + 1;
	}
    }
    start = stopInvisibleAttrs + 1;

    // Now get key!type=value:key!type=value...
    while (start < streamLen)
    {
	stop = stream.find('!', start);
	if (stop == String::NotFound) {
	    return;
	}

	String key(stream.getSubString(start, stop));
	unescape(key);
	start = stop + 1;

	stop = stream.find('=', start);
	if (stop == String::NotFound) {
	    return;
	}

	String type(stream.getSubString(start, stop));
	unescape(type);
	start = stop + 1;

	stop = stream.find(':', start);
	if (stop == String::NotFound) {
	    stop = streamLen;
	}

	String value(stream. getSubString(start, stop));
	unescape(value);
	start = stop + 1;

	setAttr(Attribute(key, type, value));
    }
}

//
// Copy constructor.
//
AttrBundle::AttrBundle(const AttrBundle& other)
: _type(other._type), _selector(other._selector)
{
    ConstDictionaryOfIterator<Attribute> iter(&other._attrs);
    const Attribute* attr;
    while ((attr = iter()) != NULL) {
	setAttr(*attr);
	String key(attr->getKey());
	setAttrVisible(key, other.getAttrVisible(key));
    }
}

//
// Destructor.
//
AttrBundle::~AttrBundle()
{
    // IMPORTANT: Do not RemoveAndDestroyContentsOf(&_invisibleAttrs),
    // unless you change the fact that the elements are also the
    // keys.  The keys get destroyed in the DictionaryOf destructor,
    // so that will take care of the elements as well.
    RemoveAndDestroyContentsOf(&_attrs);
    RemoveAndDestroyContentsOf(&_listeners);
}

//
//  AttrBundle* AttrBundle::clone() const
//
//  Description:
//      Create a copy of this AttrBundle.
//
//  Returns:
//	A copy of this AttrBundle.
//
AttrBundle* AttrBundle::clone() const
{
    return new AttrBundle(*this);
}

//
// Type accessor.
//
String AttrBundle::getType() const
{
    return _type;
}

//
// Selector accessor.
//
String AttrBundle::getSelector() const
{
    return _selector;
}

//
//  String AttrBundle::serialize() const
//
//  Description:
//      Turn this AttrBundle into a string representation.  This can
//      be used later to recreate the state of this AttrBundle.
//
//  Returns:
//	String representation of this AttrBundle.
//
String AttrBundle::serialize() const
{
    // First pass, figure out how much space we need.
    //
    int maxLen = escapeLength(BUNDLE_VERSION);

    maxLen += escapeLength(_type);
    maxLen += escapeLength(_selector);
    maxLen += 3;  // :s in version:type:sel:
    
    ConstDictionaryOfIterator<String> invisibleIter(&_invisibleAttrs);
    const String* invisibleKey;
    while ((invisibleKey = invisibleIter()) != NULL) {
	maxLen += escapeLength(*invisibleKey) + 1;
    }
    maxLen++;			// '!' delimiter.

    ConstDictionaryOfIterator<Attribute> si(&_attrs);
    const Attribute* attr;
    while ((attr = si()) != NULL) {
	maxLen += escapeLength(attr->getKey());
	maxLen += escapeLength(attr->getValueString());
	maxLen += escapeLength(attr->getTypeString());
	maxLen += 3;		// Add in the '!', '=', ':' delimiters
    }

    maxLen++;			// null terminator.

    maxLen++;			// One extra byte so we can add a
				// newline for sending over the wire.

    String rval(maxLen, '\0');

    // Now we will copy the first components, escaping in place as we
    // go.
    CharIndex index = 0;	// index is always insertion point
    char* rptr = rval.writableCharPtr();
    addStringWithEscape(rptr, index, BUNDLE_VERSION);
    rptr[index++] = ':';
    addStringWithEscape(rptr, index, _type);
    rptr[index++] = ':';
    addStringWithEscape(rptr, index, _selector);
    rptr[index++] = ':';

    // Add invisible Attribute keys.
    invisibleIter.reset();
    while ((invisibleKey = invisibleIter()) != NULL) {
	addStringWithEscape(rptr, index, *invisibleKey);
	rptr[index++] = ':';
    }
    rptr[index++] = '!';

    // Now add key!type=value:key!type=value...
    //
    si.reset();
    while ((attr = si()) != NULL) {
	addStringWithEscape(rptr, index, attr->getKey());
	rptr[index++] = '!';
	addStringWithEscape(rptr, index, attr->getTypeString());
	rptr[index++] = '=';
	addStringWithEscape(rptr, index, attr->getValueString());
	rptr[index++] = ':';
    }

    rptr[--index] = '\000';
    assert(index < maxLen);
    return(rval);
}

//
//  static int DoCompare(const void* left, const void* right)
//
//  Description:
//      Callback function for qsort, used for sorting an array of
//      pointers to Strings.
//
//  Parameters:
//      left   left side of comparison.
//      right  rigth side of comparison.
//
//  Returns:
//	< 0 if left < right, 0 if left == right, > 0 if left > right.
//
static int DoCompare(const void* left, const void* right)
{
    String** leftString = (String **)left;
    String** rightString = (String **)right;
    return strcmp(**leftString, **rightString);
}

//
//  String AttrBundle::toString(int pad) const
//
//  Description:
//      Convert this AttrBundle to a String suitable for display in
//      log files.
//
//  Parameters:
//      pad  Amount of padding to display to the left, useful for
//      formatting in log files.
//
//  Returns:
//	String version of this AttrBundle.
//
String AttrBundle::toString(int pad) const
{
    // Set up padding.  We pad attributes two + "pad" spaces on the
    // left.
    String padStr(pad + 3);
    for (int ii = 0; ii < pad; ii++) {
	padStr += " ";
    }

    String* keys[_attrs.getSize()];
    // Figure out max key length and length of string we'll create.
    int maxKeyLen = 0;
    int numVisibleAttrs = 0;
    // 2 for colon and space
    int strLen = padStr.getLength() +
	_type.getLength() + _selector.getLength() + 2;

    ConstDictionaryOfIterator<Attribute> iter(&_attrs);
    const Attribute* attr;
    while ((attr = iter()) != NULL) {
	String key(attr->getKey());
	if (getAttrVisible(key)) {
	    keys[numVisibleAttrs++] = new String(key);
	    int len = attr->getKey().getLength();
	    if (len > maxKeyLen) {
		maxKeyLen = len;
	    }
	    if (attr->getType() == Attribute::BUNDLE) {
		// +1 for extra newline.
		strLen += AttrBundle(attr->bundleValue())
		    .toString(pad + NEST_PAD).getLength() + 1;
	    } else {
		strLen += attr->getValueString().getLength();
	    }
	}
    }
    // 5 for space, colon, newline, and two extra spaces (padding).
    strLen += (maxKeyLen + padStr.getLength() + 5) * numVisibleAttrs;

    // Alphabetize by key.
    qsort(keys, numVisibleAttrs, sizeof(const char*), DoCompare);

    // Create string version of this AttrBundle.
    String str(strLen);
    str += padStr;
    padStr += "  ";
    str += _type;
    str += ": ";
    str += _selector;
    for (int key = 0; key < numVisibleAttrs; key++) {
	str += "\n";
	str += padStr;
	str += *keys[key];
	for (int ii = maxKeyLen - keys[key]->getLength(); ii > 0; ii--) {
	    str += " ";
	}
	attr = _attrs.lookupByKey(*keys[key]);
	// No Attributes should have been deleted since we collected
	// our array of keys.
	assert(attr != NULL);
	str += ": ";
	if (attr->getType() == Attribute::BUNDLE) {
	    str += "\n";
	    str += AttrBundle(attr->bundleValue()). toString(pad + NEST_PAD);
	} else {
	    str += attr->getValueString();
	}
	delete keys[key];
    }

    // Check our math.
    assert(strLen == str.getLength());
    return str;
}

//
//  void AttrBundle::setAttrVisible(const String& key, bool visible)
//
//  Description:
//      Set visibility of Attribute named by "key".
//
//  Parameters:
//      key      Key to set visibility of.
//      visible  true if Attribute is to be visible, false otherwise.
//
void AttrBundle::setAttrVisible(const String& key, bool visible)
{
    if (visible) {
	// Don't delete return value, because it's the same as its key
	// and therefore owned by _invisibleAttrs.
	_invisibleAttrs.remove(key);
    } else {
	String* str = new String(key);
	_invisibleAttrs.add(str, str);
    }
}

//
//  bool AttrBundle::getAttrVisible(const String& key) const
//
//  Description:
//      Get the visibility of Attribute named by "key".
//
//  Parameters:
//      key  Key of Attribute to check visibility of.
//
//  Returns:
//	visibility of Attribute named by "key".
//
bool AttrBundle::getAttrVisible(const String& key) const
{
    return _invisibleAttrs.lookupByKey(key) == NULL;
}

//
//  Attribute AttrBundle::getAttr(const String& key)
//
//  Parameters:
//      key  key of attribute we want.
//
//  Returns:
//	Attribute corresponding to "key", or Attribute::NUL if not
//	found.
//
Attribute AttrBundle::getAttr(const String& key) const
{
    const Attribute* attr = _attrs.lookupByKey(key);
    return attr ? *attr : Attribute::NUL;
}

//
//  void AttrBundle::setAttr(const Attribute& attr)
//
//  Description:
//      Set an Attribute.  Replaces any previous attribute that had
//      the same key.
//
//  Parameters:
//      attr  The attribute to set.
//
void AttrBundle::setAttr(const Attribute& attr)
{
    Attribute* old = _attrs.lookupByKey(attr.getKey());

    // It's illegal to try to change the type of an attribute.
    assert(old == NULL || attr.getType() == old->getType());

    if (old == NULL || *old != attr) {
	delete _attrs.remove(attr.getKey());
	_attrs.add(new Attribute(attr), new String(attr.getKey()));
	notifyChanged(attr);
    }
}

//
//  CollectionOf<Attribute> AttrBundle::copyAttrList()
//
//  Description:
//      Make a copy of the attribute list.  Caller is responsible for
//      deleting the Attributes in the returned list.
//
//  Returns:
//	List of Attributes in this AttrBundle.
//
CollectionOf<Attribute> AttrBundle::copyAttrList() const
{
    CollectionOf<Attribute> attrs;
    ConstDictionaryOfIterator<Attribute> iter(&_attrs);
    Attribute* attr;
    while ((attr = iter()) != NULL) {
	attrs.add(new Attribute(*attr));
    }
    return attrs;
}

//
//  void AttrBundle::unescape(String& packed)
//
//  Description:
//	Replace all escaped characters in "packed" with their unescaped
//	equivalents.  The escaped characters are described in the comments
//	for AttrBundle::AttrBundle(const String&)
//
//  Parameters:
//	packed	string to unescape
//
void AttrBundle::unescape(String& packed)
{
    // Scan the string, looking for escape markers.
    //
    unsigned int oldLen = packed.getLength();
    CharIndex rd = 0;
    CharIndex wr = 0;
    char* cptr = packed.writableCharPtr();
    while (cptr[rd]) {
	if (cptr[rd] != '@') {
	    cptr[wr] = cptr[rd];
	} else {
	    bool match = false;
	    for (int i = 0; i < sizeof(gEscapes)/sizeof(*gEscapes); i++) {
		if (strncmp(cptr + rd, gEscapes[i].escapeStr,
			    gEscapes[i].escapeLen) == 0) {
		    cptr[wr] = gEscapes[i].toEscape;
		    match = true;
		    rd += 2;
		    break;
		}
	    }

	    // Don't allow bogus escape sequences.
	    assert(match == true);
	}
	rd++;
	wr++;
    }
    cptr[wr++] = 0;
    // Zero the rest of the string, so that String::getHashValue()
    // works.
    while (wr < oldLen) {
	cptr[wr++] = 0;
    }
}

//
//  void AttrBundle::addStringWithEscape(char* dest,
//  				         CharIndex& destIndex,
//  				         const String& toAdd)
//
//  Description:
//	Escape all special characters in "toAdd" and copy the escaped
//	string to "dest" beginning at position "destIndex".
//
//  Parameters:
//	dest		destination for escaped string
//	destIndex	position in "dest" at which to copy.  This is
//	                a value/result parameter.
//	toAdd		unescaped string to be escaped and copied to "dest"
//
void AttrBundle::addStringWithEscape(char* dest,
				     CharIndex& destIndex,
				     const String& toAdd)
{
    const char* itemStr = toAdd;
    int itemLen = toAdd.getLength();
    for (CharIndex i = 0; i < itemLen; i++) {
	bool escaped = false;
	for (int match = 0;
	     match < sizeof(gEscapes)/sizeof(*gEscapes);
	     match++) {
	    if (itemStr[i] == gEscapes[match].toEscape) {
		strcpy(&dest[destIndex], gEscapes[match].escapeStr);
		destIndex += gEscapes[match].escapeLen;
		escaped = true;
		break;
	    }
	}
	if (!escaped) {
	    dest[destIndex++] = itemStr[i];
	}
    }
}

//
//  int AttrBundle::escapeLength(const String& str)
//
//  Description:
//	Compute the length of the entire string, taking any escape
//	characters into consideration.
//
//  Parameters:
//	str	string whose escaped length should be computed
//
//  Returns:
//	Length needed for escaped string.  Does NOT include a NULL terminator.
//	
//
int AttrBundle::escapeLength(const String& str)
{
    // Compute the length of the entire string, taking any escape
    // characters into consideration.
    //
    int len = 0;
    for (CharIndex i = 0; i < str.getLength(); i++) {
	len++;
	for (int match = 0;
	     match < sizeof(gEscapes)/sizeof(*gEscapes);
	     match++) {
	    if (str[i] == gEscapes[match].toEscape) {
		len += (gEscapes[match].escapeLen - 1);
		break;
	    }
	}
    }
    return(len);
}

//
//  bool AttrBundle::equals(AttrBundle& other)
//
//  Description:
//      Determine whether "other" is the same as this AttrBundle.
//      Compare attributes.
//
//  Parameters:
//      other  AttrBundle to compare
//
//  Returns:
//	"true" if AttrBundles are the same, "false" if they are
//	different
//
bool AttrBundle::equals(const AttrBundle& other) const
{
    if (strcmp(_type, other._type) != 0) {
	return false;
    }

    if (strcmp(_selector, other._selector) != 0) {
	return false;
    }

    if (_attrs.getSize() != other._attrs.getSize()) {
	return false;
    }

    ConstDictionaryOfIterator<Attribute> iter(&other._attrs);
    const Attribute* attr;
    while ((attr = iter()) != NULL) {
	
	const Attribute* myAttr = _attrs.lookupByKey(attr->getKey());
	if (myAttr == NULL || *myAttr != *attr) {
	    return false;
	}
    }

    return true;
}

//
//  void AttrBundle::adoptAttrListener(AttrListener* listener)
//
//  Description:
//      Add "listener" to our notification list.
//
//  Parameters:
//      listener  AttrListener to add.
//
void AttrBundle::adoptAttrListener(AttrListener* listener)
{
    _listeners.add(listener);
}

//
//  AttrListener* AttrBundle::orphanAttrListener(AttrListener* listener)
//
//  Description:
//      Remove "listener" from our notification list.
//
//  Parameters:
//      listener  AttrListener to remove.
//
//  Returns:
//	"listener", unless it's not in our list, in which case we
//	return NULL.
//
AttrListener* AttrBundle::orphanAttrListener(AttrListener* listener)
{
    return _listeners.removePointer(listener);
}

//
//  void AttrBundle::removeAttrListener(AttrListener* listener)
//
//  Description:
//      Remove "listener" from our notification list and delete it.
//
//  Parameters:
//      listener  AttrListener to remove.
//
void AttrBundle::removeAttrListener(AttrListener* listener)
{
    AttrListener* toDelete = orphanAttrListener(listener);
    // This prevents the caller from having to worry about whether it
    // should delete "listener" if it's not actually installed.
    assert(toDelete == listener);
    delete toDelete;
}

//
//  void AttrBundle::notifyChanged(const Attribute& attr)
//
//  Description:
//      Notify listeners that "attr" has changed.
//
//  Parameters:
//      attr  The attribute that changed.
//
void AttrBundle::notifyChanged(const Attribute& attr)
{
    AttrEvent event(*this, attr);

    IteratorOver<AttrListener> iter(&_listeners);
    AttrListener* listener;
    while ((listener = iter()) != NULL) {
	listener->attrChanged(event);
    }
}
