//
// AttrBundle.h
//
//	Bundle of Attributes, used to represent information about an
//	object.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.11 $"
#ifndef _SYSADM_ATTR_BUNDLE_H
#define _SYSADM_ATTR_BUNDLE_H

#include <sysadm/Attribute.h>
#include <sysadm/String.h>
#include <sysadm/DictionaryOf.h>
#include <sysadm/CollectionOf.h>
#include <sysadm/AttrListener.h>

BEGIN_NAMESPACE(sysadm);

//
// AttrBundle aggregates Attribute objects into bundles of key/value
// pairs.
//
// In addition to a number of key/value pairs, an AttrBundle has a
// type and selector which is used to identify Items and route
// Packets.
//
// An AttrBundle stores an independent mapping of key names to
// visibility flags, controlled via the setAttrVisible()
// method.  The visibility mapping is used in the toString()
// method to prevent invisible Attributes from being part of the
// String returned.  This facility is used to prevent sensitive
// information such as passwords from appearing in logging or debug
// output.
//
class AttrBundle {
  public:

    // Construct an AttrBundle with "type" and "selector".
    AttrBundle(const String& type, const String& selector);

    // Construct an AttrBundle that was serialized by a previous call
    // to AttrBundle::serialize().
    AttrBundle(const String& stream);

    // Copy constructor.  The list of listeners is not copied.
    AttrBundle(const AttrBundle& other);

    // Destructor.
    virtual ~AttrBundle();

    // Create a new AttrBundle which is a copy of this one.  The list
    // of listeners is not copied.
    virtual AttrBundle* clone() const;

    // Get the type of this AttrBundle.
    virtual String getType() const;

    // Get the selector of this AttrBundle.
    virtual String getSelector() const;

    // Turn this AttrBundle into an ascii string representation.  This
    // can be turned back into an AttrBundle using the constructor
    // that takes a "stream" argument.
    virtual String serialize() const;

    // Turn this AttrBundle into an ascii string representation
    // suitable for human consumption.  "pad" is the number of padding
    // spaces to insert at the left edge of each line.
    virtual String toString(int pad = 0) const;

    // Control the visibility of an Attribute's value in the String
    // returned by toString().  This is used to prevent passwords from
    // showing up in debugging logs.  By default, Attribute values are
    // visible.
    virtual void setAttrVisible(const String& key, bool visible);

    // Determine whether the value of the Attribute associated with
    // "key" should be visible in log files.
    virtual bool getAttrVisible(const String& key) const;

    // Get a named attribute.  If the attribute does not exist,
    // Attribute::NUL is returned.
    virtual Attribute getAttr(const String& key) const;

    // Set an Attribute.
    virtual void setAttr(const Attribute& attr);

    // Get the entire Attribute list.  The caller is responsible for
    // deleting all of the Attributes in the list.
    virtual CollectionOf<Attribute> copyAttrList() const;
    
    // Return "true" if "other" is the same as this AttrBundle,
    // "false" if they are the different.  Returns "true" if the
    // type, selector, and Attributes of "other" are the same as this
    // AttrBundle.  The visibility flags are ignored.
    virtual bool equals(const AttrBundle& other) const;

    // "listener" will get notified when Attributes change.
    virtual void adoptAttrListener(AttrListener* listener);

    // "listener" will no longer get notified when Attributes change.
    virtual AttrListener* orphanAttrListener(AttrListener* listener);

    // "listener" will no longer get notified when Attributes change.
    // "listener" will be deleted.
    virtual void removeAttrListener(AttrListener* listener);

  private:

    // Intentionally undefined.
    AttrBundle& operator=(const AttrBundle&);

    // Notify listeners that an Attribute has changed.
    void notifyChanged(const Attribute& attr);

    // Remove escape strings from "packed".
    static void unescape(String& packed);

    // Add an escaped version of "toAdd" to "dest".
    static void addStringWithEscape(char* dest,
				    CharIndex& destIndex,
				    const String& toAdd);

    // Figure out how long "str" will be once escapes are added.
    static int escapeLength(const String& str);

    String _type;
    String _selector;
    DictionaryOf<Attribute> _attrs;
    DictionaryOf<String> _invisibleAttrs;
    CollectionOf<AttrListener> _listeners;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_ATTR_BUNDLE_H
