//
// Attribute.c++
//
//	Class for storing key/value pairs.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.8 $"

#include <assert.h>

#include <sysadm/Attribute.h>
#include <sysadm/AttrBundle.h>

BEGIN_NAMESPACE(sysadm);

//
// Indexing this array with an Attribute::EType gives a string
// representation of an attribute's type.
//
const char* Attribute::_typeStrings[] = { "n", "s", "l", "b", "d", "a" };

//
// This is here to support testing of an attribute to see if it's
// NULL.
//
const Attribute Attribute::NUL("");

//
// Construct a String Attribute.
//
Attribute::Attribute(const String& key, const String& value)
: _key(key), _type(STRING)
{
    _value._sval = new String(value);
}

//
// Construct a String Attribute.
//
Attribute::Attribute(const String& key, const char* value)
: _key(key), _type(STRING)
{
    _value._sval = new String(value);
}

//
// Construct a long Attribute.
//
Attribute::Attribute(const String& key, long long value)
: _key(key), _type(LONG)
{
    _value._lval = value;
}

// 
// Construct a bool Attribute.
//
Attribute::Attribute(const String& key, bool value)
: _key(key), _type(BOOLEAN)
{
    _value._bval = value;
}

//
// Construct a double Attribute.
//
Attribute::Attribute(const String& key, double value)
: _key(key), _type(DOUBLE)
{
    _value._dval = value;
}

//
// Construct a bundle Attribute.
//
Attribute::Attribute(const String& key, const AttrBundle& value)
: _key(key), _type(BUNDLE)
{
    _value._sval = new String(value.serialize());
}

//
// Construct an attribute based on string representations of type and
// value.
//
Attribute::Attribute(const String& key, const String& type,
		     const String& value)
: _key(key), _type(NONE)
{
    int i;
    for (i = 0; i < sizeof _typeStrings/sizeof *_typeStrings; i++) {
	if (type == _typeStrings[i]) {
	    _type = (EType)i;
	    break;
	}
    }

    // Make sure we found a real type.
    assert(i < sizeof _typeStrings/sizeof *_typeStrings);

    setValueFromString(value);
}

//
// Construct an Attribute based on "type" and a string
// representation of "value".
//
Attribute::Attribute(const String& key, EType type, const String& value)
: _key(key), _type(type)
{
    setValueFromString(value);
}

//
// Construct an attribute with no value.
//
Attribute::Attribute(const String& key)
: _key(key), _type(NONE)
{
}

//
// Copy constructor.
//
Attribute::Attribute(const Attribute& other)
: _key(other._key), _type(other._type)
{
    copyValue(other);
}

//
// Destructor.
//
Attribute::~Attribute()
{
    destroy();
}

//
// Assignment operator
//
Attribute& Attribute::operator=(const Attribute& other)
{
    if (this != &other) {
	destroy();
	_key = other._key;
	_type = other._type;
	copyValue(other);
    }
    return *this;
}

//
//  String Attribute::getKey() const
//
//  Returns:
//	The key of this Attribute.
//
String Attribute::getKey() const
{
    return _key;
}

//
//  Attribute::EType Attribute::getType() const
//
//  Returns:
//	The type of this Attribute.
//
Attribute::EType Attribute::getType() const
{
    return _type;
}

//
//  String Attribute::stringValue() const
//
//  Returns:
//	String value of this Attribute.
//
String Attribute::stringValue() const
{
    assert(_type == STRING);
    return *_value._sval;
}

//
//  long long Attribute::longValue() const
//
//  Returns:
//	long value of this Attribute.
//
long long Attribute::longValue() const
{
    assert(_type == LONG);
    return _value._lval;
}

//
//  bool Attribute::booleanValue() const
//
//  Returns:
//	bool value of this Attribute.
//
bool Attribute::booleanValue() const
{
    assert(_type == BOOLEAN);
    return _value._bval;
}

//
//  double Attribute::doubleValue() const
//
//  Returns:
//	double value of this Attribute.
//
double Attribute::doubleValue() const
{
    assert(_type == DOUBLE);
    return _value._dval;
}

//
//  String Attribute::bundleValue() const
//
//  Returns:
//	bundle value of this Attribute.  We'd like to return an
//	AttrBundle, but that would introduce a circular dependency
//	between Attribute and AttrBundle.  The caller will have to use
//	the "deserialize" AttrBundle constructor to turn the stream
//	back into an AttrBundle.
//
String Attribute::bundleValue() const
{
    assert(_type == BUNDLE);
    return *_value._sval;
}

//
//  String Attribute::getTypeString() const
//
//  Returns:
//	A String representation of this Attribute's type.  This is
//	used in AttrBundle::serialize().
//
String Attribute::getTypeString() const
{
    assert(_type < sizeof _typeStrings/sizeof *_typeStrings);
    return _typeStrings[_type];
}

//
//  String Attribute::getValueString() const
//
//  Returns:
//      A String representation of this Attribute's value.  This is
//      used in AttrBundle::serialize().
//
String Attribute::getValueString() const
{
    switch (_type) {
    case STRING:
    case BUNDLE:
	return *_value._sval;
    case LONG:
	return StringFromLongLong(_value._lval);
    case BOOLEAN:
	return StringFromBool(_value._bval);
    case DOUBLE:
	return StringFromDouble(_value._dval);
    case NONE:
    default:
	return "";
    }
}

//
//  bool Attribute::operator==(const Attribute& other) const
//
//  Returns:
//	true if this and other are equal, false otherwise.
//
bool Attribute::operator==(const Attribute& other) const
{
    if (_key != other._key || _type != other._type) {
	return false;
    }
    switch (_type) {
    case STRING:
    case BUNDLE:
	return *_value._sval == *other._value._sval;
    case LONG:
	return _value._lval == other._value._lval;
    case BOOLEAN:
	return _value._bval == other._value._bval;
    case DOUBLE:
	return _value._dval == other._value._dval;
    default:
	return true;
    }
}

//
//  bool Attribute::operator!=(const Attribute& other) const
//
//  Returns:
//	true if this and other are not equal, false otherwise.
//
bool Attribute::operator!=(const Attribute& other) const
{
    return !operator==(other);
}

//
//  void Attribute::setValueFromString(const String& value)
//
//  Description:
//      Set the value of this Attribute from the string representation
//      "value".  "value" will be converted based on the value of
//      _type.
//
//  Parameters:
//      value  string representation of value to store in this
//             Attribute.
//
void Attribute::setValueFromString(const String& value)
{
    switch (_type) {
    case STRING:
    case BUNDLE:
	_value._sval = new String(value);
	break;
    case LONG:
	if (!StringToLongLong(value, _value._lval)) {
	    _value._lval = 0LL;
	}
	break;
    case BOOLEAN:
	if (!StringToBool(value, _value._bval)) {
	    _value._bval = false;
	}
	break;
    case DOUBLE:
	if (!StringToDouble(value, _value._dval)) {
	    _value._dval = 0.0;
	}
	break;
    }
}

//
//  void Attribute::destroy()
//
//  Description:
//      Release resources owned by this Attribute.
//
void Attribute::destroy()
{
    if (_type == STRING || _type == BUNDLE) {
	delete _value._sval;
	_type = NONE;
    }
}

//
//  void Attribute::copyValue(const Attribute& other)
//
//  Description:
//      Make this Attribute's vale into a copy of "other"'s.
//
//  Parameters:
//      other  Attribute to copy.
//
void Attribute::copyValue(const Attribute& other)
{
    switch (other._type) {
    case STRING:
	_value._sval = new String(other.stringValue());
	break;
    case BUNDLE:
	_value._sval = new String(other.bundleValue());
	break;
    case LONG:
	_value._lval = other.longValue();
	break;
    case BOOLEAN:
	_value._bval = other.booleanValue();
	break;
    case DOUBLE:
	_value._dval = other.doubleValue();
	break;
    }
}

END_NAMESPACE(sysadm);
