//
// Attribute.h
//
//	Class for storing key/value pairs.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ifndef _SYSADM_ATTRIBUTE_H
#define _SYSADM_ATTRIBUTE_H

#ident "$Revision: 1.8 $"

#include <sysadm/String.h>

BEGIN_NAMESPACE(sysadm);

//
// Attribute is a class for storing typed key/value pairs.  The only
// types supported are String, long, bool, double, and AttrBundle.
//
// Attributes are aggregated into AttrBundles to represent bundles of
// information.
//
class Attribute {
  public:

    // Enumeration of the supported types.  "NONE" is only used for
    // the NUL Attribute.
    enum EType { NONE, STRING, LONG, BOOLEAN, DOUBLE, BUNDLE };
    
    // Construct a String Attribute.
    Attribute(const String& key, const String& value);
    Attribute(const String& key, const char* value);

    // Construct a long Attribute.
    Attribute(const String& key, long long value);

    // Construct a bool Attribute.
    Attribute(const String& key, bool value);

    // Construct a double Attribute.
    Attribute(const String& key, double value);

    // Construct a bundle Attribute.
    Attribute(const String& key, const class AttrBundle& bundle);

    // Construct an Attribute based on string representation of "type"
    // and "value".  This is used when deserializing AttrBundles.
    Attribute(const String& key, const String& type, const String& value);

    // Construct an Attribute based on "type" and a string
    // representation of "value".
    Attribute(const String& key, EType type, const String& value);

    // Copy constructor.
    Attribute(const Attribute& other);

    // Destructor.
    virtual ~Attribute();

    // Assignment operator.
    Attribute& operator=(const Attribute& other);
    
    // Accessors for key and type.
    virtual String getKey() const;
    virtual EType getType() const;

    // Getting the value of an Attribute.  These all assert that the
    // Attribute is of the right type.
    virtual String stringValue() const;
    virtual long long longValue() const;
    virtual bool booleanValue() const;
    virtual double doubleValue() const;
    // bundleValue() returns a serialized AttrBundle rather than an
    // AttrBundle, to avoid circular dependency between Attribute and
    // AttrBundle.
    virtual String bundleValue() const;

    // Get string representations of type and value.  These are used
    // in AttrBundle::serialize().
    virtual String getTypeString() const;
    virtual String getValueString() const;

    // Comparison operators.  These compare both the keys and the
    // values of the two Attributes.
    virtual bool operator==(const Attribute& other) const;
    virtual bool operator!=(const Attribute& other) const;

    // A "null" attribute.
    static const Attribute NUL;

  private:

    // Construct an Attribute with no value.  This is used for the NUL
    // Attribute.
    Attribute(const String& key);

    void setValueFromString(const String& value);

    void destroy();
    void copyValue(const Attribute& other);

    // Value can store any of the types of values we support.
    union Value {
	String* _sval;
	long long _lval;
	bool _bval;
	double _dval;
    };

    // Strings used for the string representation of type.
    static const char* _typeStrings[];

    String _key;
    Value _value;
    EType _type;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_ATTRIBUTE_H
